from sandbox import sdk2
from sandbox.projects.common.binary_task import LastBinaryTaskRelease, LastBinaryReleaseParameters
from sandbox.projects.geobase.GeodataTreeLingStable.resource import GEODATA_TREE_LING_STABLE
from sandbox.projects.geobase.common.export.export2yt import get_regions, get_scheme_from_attrs
from sandbox.projects.yabs.qa.utils.general import html_hyperlink


class YabsGeoTableArchive(LastBinaryTaskRelease, sdk2.Task):
    '''Save geodata resource to YT'''

    class Parameters(LastBinaryReleaseParameters):
        clusters = sdk2.parameters.List('YT clusters', default=['hahn', 'arnold'])
        archive_folder = sdk2.parameters.String(
            'Destination directory path',
            required=True,
            description='Recursive and force create table in folder with name of resource id',
        )
        geodata_resource = sdk2.parameters.Resource(
            'Resource to archive',
            resource_type=GEODATA_TREE_LING_STABLE,
            required=True,
        )
        expiration_time = sdk2.parameters.Integer('TTL in days', default=32)
        with sdk2.parameters.Output:
            destination_table = sdk2.parameters.String('Saved path for table')

    def on_execute(self):
        from library.python.yt import expiration_time
        from yt.wrapper import YtClient, ypath_join, yson

        geodata = sdk2.ResourceData(self.Parameters.geodata_resource).path
        regions = get_regions(str(geodata))
        if not regions:
            raise sdk2.common_errors.TaskFailure('Empty regions list')
        schema = yson.loads('[{}]'.format(';'.join(get_scheme_from_attrs(regions[0]))))
        self.Parameters.destination_table = ypath_join(self.Parameters.archive_folder, str(self.Parameters.geodata_resource.id))
        attributes = {
            'schema': schema,
            'sandbox_resource_id': self.Parameters.geodata_resource.id,
            'sandbox_resource_description': self.Parameters.geodata_resource.description,
            'sandbox_task_id': self.id,
        }
        if self.Parameters.expiration_time:
            attributes['expiration_time'] = expiration_time(days=self.Parameters.expiration_time)

        secret = sdk2.yav.Secret('sec-01d6apzcex5fpzs5fcw1pxsfd5')
        yt_token = secret.data()['yt_token']
        saved = []
        for cluster in self.Parameters.clusters:
            yt = YtClient(proxy=cluster, token=yt_token)
            with yt.Transaction():
                yt.create(
                    'table',
                    self.Parameters.destination_table,
                    attributes=attributes,
                    recursive=True,
                    force=True,
                )
                yt.write_table(self.Parameters.destination_table, regions)

            saved.append(html_hyperlink(
                'https://yt.yandex-team.ru/{cluster}/navigation?path={path}&'.format(
                    cluster=cluster,
                    path=self.Parameters.destination_table,
                ),
                cluster
            ))
        self.set_info('Saved tables to: ' + ', '.join(saved), do_escape=False)
