import logging
import json

from jinja2 import Template

from sandbox import sdk2
from sandbox.common.urls import get_resource_link
from sandbox.projects.common.binary_task import LastBinaryTaskRelease, LastBinaryReleaseParameters
from sandbox.projects.geobase.GeodataTreeLingStable.resource import GEODATA_TREE_LING_STABLE
from sandbox.projects.geobase.common.export.export2yt import get_regions
from sandbox.projects.yabs.GeoTable.Resources import YabsGeoBaseReport
from sandbox.projects.yabs.qa.utils.general import html_hyperlink


DIRECT_OPTIONS = ['ad', 'bs']
NEEDED_FIELDS = ['reg_id', 'name']


class YabsGeoBaseReportHtml(sdk2.Resource):
    pass


def get_html_report_template():
    import library.python.resource as lpr
    return Template(lpr.find('sandbox/projects/yabs/GeoTable/CompareResources/templates/report.html'))


def get_resource_fields(resource):
    return {
        'id': resource.id,
        'description': resource.description,
        'url': get_resource_link(resource.id),
    }


def iterate_zip_regions_bases(regions_base, regions_test):
    base_idx = 0
    test_idx = 0
    base_len = len(regions_base)
    test_len = len(regions_test)
    while base_idx < base_len and test_idx < test_len:
        region_base = regions_base[base_idx]
        region_test = regions_test[test_idx]

        reg_id_base = region_base['reg_id']
        reg_id_test = region_test['reg_id']

        if reg_id_base == reg_id_test:
            yield (reg_id_base, region_base, region_test)
            base_idx += 1
            test_idx += 1
        elif reg_id_base < reg_id_test:
            yield (reg_id_base, region_base, None)
            base_idx += 1
        else:
            yield (reg_id_test, None, region_test)
            test_idx += 1

    while base_idx < base_len:
        region_base = regions_base[base_idx]
        yield (region_base['reg_id'], region_base, None)
        base_idx += 1

    while test_idx < test_len:
        region_test = regions_test[test_idx]
        yield (region_test['reg_id'], None, region_test)
        test_idx += 1


def is_direct_region(region):
    return region and any([region[option] for option in DIRECT_OPTIONS])


def compare_regions(self, regions):
    logging.info('Start compare regions')
    added = {
        'direct': [],
        'other': []
    }
    deleted = {
        'direct': [],
        'other': []
    }
    changed = {
        'direct': [],
        'other': []
    }

    def append_to(dict_to_append, region, is_direct):
        dict_to_append['direct' if is_direct else 'other'].append({
            field: region[field] for field in NEEDED_FIELDS
        })

    for reg_id, region_base, region_test in regions:
        is_base_direct = is_direct_region(region_base)
        is_test_direct = is_direct_region(region_test)

        if region_base is None:
            append_to(added, region_test, is_test_direct)
            continue

        if region_test is None:
            append_to(deleted, region_base, is_base_direct)
            continue

        def compare_field(field):
            return region_base[field] == region_test[field]

        if is_base_direct != is_test_direct:
            if is_base_direct:
                append_to(deleted, region_base, True)
            else:
                append_to(added, region_test, True)
            continue

        changed_fileds = []

        def add_field(field, is_important):
            changed_fileds.append({
                'field': field,
                'base': region_base[field],
                'test': region_test[field],
                'is_important': is_important,
            })

        for field in self.Parameters.important_columns:
            if not compare_field(field):
                add_field(field, True)
        for field in region_base.keys():
            if field not in self.Parameters.important_columns and not compare_field(field):
                add_field(field, False)
        if changed_fileds:
            changed['direct' if is_base_direct else 'other'].append({
                'reg_id': region_base['reg_id'],
                'name': region_base['name'],
                'changed_fileds': changed_fileds,
            })

    logging.info('Finish compare resource')

    return {
        'changed': changed,
        'added': added,
        'deleted': deleted,
        'resource_base': get_resource_fields(self.Parameters.resource_base),
        'resource_test': get_resource_fields(self.Parameters.resource_test),
    }


class YabsGeoTableCompareResources(LastBinaryTaskRelease, sdk2.Task):
    '''Compare GeoBase resources'''

    class Parameters(LastBinaryReleaseParameters):
        resource_base = sdk2.parameters.Resource(
            'Base resource',
            resource_type=GEODATA_TREE_LING_STABLE,
            required=True,
        )
        resource_test = sdk2.parameters.Resource(
            'Test resource',
            resource_type=GEODATA_TREE_LING_STABLE,
            required=True,
        )
        important_columns = sdk2.parameters.List(
            'Columns to check',
            default=[
                'parent_id',
                'type',
            ],
        )
        with sdk2.parameters.Output:
            has_diff = sdk2.parameters.Bool('Has diff')

    def on_execute(self):
        logging.info('Start load resources: ' + ', '.join(map(str, [
            self.Parameters.resource_base.id,
            self.Parameters.resource_test.id,
        ])))
        geobase_base = sdk2.ResourceData(self.Parameters.resource_base)
        geobase_test = sdk2.ResourceData(self.Parameters.resource_test)
        logging.info('Finish load resources')

        logging.info('Start parse regions')
        regions_base = get_regions(str(geobase_base.path))
        regions_test = get_regions(str(geobase_test.path))
        logging.info('Finish parse regions')

        report = compare_regions(self, iterate_zip_regions_bases(regions_base, regions_test))

        logging.info('Start saving reports')
        self.Parameters.has_diff = any([
            report['added']['direct'],
            report['deleted']['direct'],
            report['changed']['direct'],
        ])

        report_resource = YabsGeoBaseReport(
            task=self,
            description='Report for geobase',
            path='report.json',
        )
        with report_resource.path.open('w') as f:
            f.write(json.dumps(report, indent=4).decode('utf-8'))

        report_html_template = get_html_report_template()
        report_html = report_html_template.render(report=report)
        report_html_resource = YabsGeoBaseReportHtml(
            task=self,
            description='Report for geobase in html',
            path='report.html',
        )
        with report_html_resource.path.open('w') as f:
            f.write(report_html.decode('utf-8'))

        self.set_info(html_hyperlink(report_html_resource.http_proxy, "Report"), do_escape=False)
        logging.info('Finished saving reports')
