import re
from pathlib import PurePath
from typing import Any, Union

from sandbox import sdk2
import sandbox.common.types.client as ctc
import sandbox.common.types.task as ctt
from sandbox.projects.resource_types import BUILD_LOGS
from sandbox.projects.common.build.YaMake2 import YaMake2

from sandbox.projects.yabs.LogBuildTimes.yt_write import save_yt_report


def get_evlog_from_task(task_id: Union[int, Any]):  # noqa
    return next(res for res in BUILD_LOGS.find(task_id=int(task_id)).limit(40)
                if re.fullmatch(r'evlog_\d+\.json', res.description))


def ya_make(parent: sdk2.Task, arcadia_url, **params):
    make_task = YaMake2(parent, ya_yt_store=False, checkout_arcadia_from_url=arcadia_url, **params)
    make_task.Requirements.client_tags = ctc.Tag.INTEL_E5_2650V2 & ctc.Tag.HDD
    make_task.save()
    make_task.enqueue()
    return make_task


class YaMakeParams(YaMake2.Parameters):
    checkout_arcadia_from_url = None
    ya_yt_store = None


class BuildLogToYt(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 2048

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        yt_token = sdk2.parameters.YavSecret(
            "Yav secret with YT token",
            required=True
        )
        yt_path = sdk2.parameters.String("Where to save parsed evlog in YT", required=True)
        run_ymake2 = sdk2.parameters.Bool("Run YA_MAKE_2 to collect old evlog", default=True)
        with run_ymake2.value[False]:
            evlog_param = sdk2.parameters.Resource("Old evlog", resource_type=BUILD_LOGS, required=True)
        with run_ymake2.value[True]:
            repo_url = sdk2.parameters.ArcadiaUrl("Arcadia URL for old version")
            with sdk2.parameters.Group("YaMake2 params", collapse=True) as build_params_group:
                build_params = YaMakeParams

    def on_execute(self):
        with self.memoize_stage.start_chlid:
            ya_task_task = None
            if self.Parameters.run_ymake2:
                ya_task_task = ya_make(self, self.Parameters.repo_url, **dict(self.Parameters.build_params))
                self.Context.build_task_id = ya_task_task.id
            if ya_task_task:
                raise sdk2.WaitTask([ya_task_task], [ctt.Status.Group.FINISH, ctt.Status.Group.BREAK])
        with self.memoize_stage.save_report_to_yt:
            yt_token = self.Parameters.yt_token.data().get(self.Parameters.yt_token.default_key, None) or \
                self.Parameters.yt_token.data().get("yt_token", None) or self.Parameters.yt_token.data()["token"]
            resource = self.Parameters.evlog_param or get_evlog_from_task(self.Context.build_task_id)
            path = sdk2.ResourceData(resource).path
            save_yt_report(PurePath(path), self.Parameters.yt_path, yt_token)
