import json
import re
from attr import dataclass
from pathlib import PurePath
from typing import Optional, Dict, List, Iterable

RUN_PATTERN = re.compile(r'Run\('
                         r'(.*?)'  # some id like 3k_NDR6W4piFQpzrJKU0mQ
                         r'\$\(BUILD_ROOT\)'
                         r'(/.*?)'
                         r'(\)| \$)')
UID_PATTERN = re.compile(r'\((.*?)\)')


def _pair_subtraction(p):
    if len(p) != 2:
        raise RuntimeError(f"p={p} is not a pair")
    return p[1] - p[0]


@dataclass
class Node:
    len_seconds: float = 0  # only significant operation, i.e. compilation, linking
    full_length: float = 0  # All operations: writing cache, etc.


class BuildOperationNode(Node):
    path: Optional[PurePath] = None
    name: str
    uid: str

    def __init__(self, line,
                 UID_DICT: Dict[str, List['BuildOperationNode']],
                 PATH_DICT: Dict[PurePath, 'BuildOperationNode']):
        json_repr = json.loads(line)
        self.name = json_repr["value"]["name"]
        self.len_seconds = _pair_subtraction(json_repr["value"]["time"])
        if match := re.search(RUN_PATTERN, self.name):
            self.path = PurePath(match.group(2))
            self.uid = match.group(1)
        else:
            try:
                match = re.search(UID_PATTERN, self.name)
                self.uid = match.group(1)
            except Exception as e:
                print(self.name)
                raise e
        if UID_DICT is not None:
            UID_DICT[self.uid].append(self)
        if PATH_DICT is not None and self.path:
            # if PATH_DICT.get(self.path, None):
            #     raise RuntimeError(f"Duplicate nodes for path {self.path}")
            PATH_DICT[self.path] = self


class BuildNode(Node):
    path: PurePath

    def __init__(self, main_node: BuildOperationNode, all_nodes: Iterable[BuildOperationNode]):
        self.path = main_node.path
        self.len_seconds = main_node.len_seconds
        self.full_length = sum(node.len_seconds for node in all_nodes)
