from yql_iterator import YqlIterator
from datetime import datetime, timedelta
from os.path import join

ATTRIBUTES = {
    "schema": [
        {
            "name": "Index",
            "sort_order": "ascending",
            "type": "uint64"
        },
        {
            "name": "PageID",
            "type": "int64"
        },
        {
            "name": "Name",
            "type": "string"
        }
    ],
    "dynamic": True
}


def insert_rows(proxy, yt_token, path_prefix, input_stream):
    from yt.wrapper import YtClient, JsonFormat
    yt_client = YtClient(
        token=yt_token,
        proxy=proxy
    )

    destination_path = join(path_prefix, 'TopPages')

    transaction = yt_client.start_transaction()
    try:
        if not yt_client.exists(destination_path):
            yt_client.create('table', path=destination_path, attributes=ATTRIBUTES)
        else:
            delete_stream = yt_client.select_rows("Index from [{}]".format(destination_path))
            yt_client.delete_rows(destination_path, delete_stream, format='json')

        yt_client.mount_table(destination_path)
        yt_client.insert_rows(destination_path, input_stream, format=JsonFormat(attributes={"encode_utf8": False}))
    except Exception as e:
        yt_client.abort_transaction(transaction)
        raise e
    else:
        yt_client.commit_transaction(transaction)


def get_top_pages(yql_token):
    def build_query(formatted_date):
        # noinspection SqlDialectInspection
        return '''
           SELECT S.PageID AS PageID, S.Cost as Cost, D.Name AS Name FROM (
               FROM [logs/bs-chevent-log/1d/{SOURCE_TABLE_NAME}]
               SELECT PageID, sum(CAST(eventcost as Int64)) as Cost
               WHERE fraudbits = '0' AND (options LIKE '%flat-page%')
               AND (options LIKE '%commerce%')
               AND (CAST(testtag AS Uint64) & 15u) = 1u
               AND placeid IN ('542', '1542')
               GROUP BY CAST(pageid as Int64) as PageID
               ORDER BY Cost DESC
               LIMIT 1000
           ) AS S LEFT JOIN [//home/yabs/dict/Page] AS D USING(PageID)
           ORDER BY Cost DESC
        '''.format(
            SOURCE_TABLE_NAME=formatted_date
        )

    yesterday = datetime.now() - timedelta(days=1, hours=12)
    formatted_date = yesterday.strftime("%Y-%m-%d")

    query = build_query(formatted_date)
    from yql.api.v1.client import YqlClient
    yql_client = YqlClient(
        db='hahn',
        token=yql_token
    )
    yql_data = list(YqlIterator(yql_client, 'hahn', query))

    def map_yql_row(index, row):
        return {"PageID": int(row["PageID"]), "Index": index, "Name": row["Name"]}

    stream = [map_yql_row(i, r) for i, r in enumerate(yql_data)]
    return stream
