import copy
import logging

import sandbox.projects.common.yabs.performance_best_cmp.stats as stats

ROLES = ["BS", "YABS", "BSRANK"]
OLD_RUN_POSTFIXES = ["", "_B", "_C"]
NEW_RUN_POSTFIXES = ["_A", "_B", "_C"]

BS_PERF_TESTS = ["YABS_SERVER_40_PERFORMANCE_BEST_BS{}".format(run) for run in OLD_RUN_POSTFIXES]
YABS_PERF_TESTS = ["YABS_SERVER_40_PERFORMANCE_BEST_YABS{}".format(run) for run in OLD_RUN_POSTFIXES]
BSRANK_PERF_TESTS = ["YABS_SERVER_40_PERFORMANCE_BEST_BSRANK{}".format(run) for run in OLD_RUN_POSTFIXES]
PERFORMANCE_TESTS = BS_PERF_TESTS + YABS_PERF_TESTS + BSRANK_PERF_TESTS
SIZE_TESTS = ["YABS_SERVER_38_DB_SIZE{}".format(run) for run in NEW_RUN_POSTFIXES]

YABS_CORES = 96424
BSRANK_CORES = 24384
BS_CORES = 20016
ALL_CORES = YABS_CORES + BSRANK_CORES + BS_CORES


class PlotParams(object):
    def __init__(self, tests, metric, aggregator, value_extractor=None, get_switch_points=True, use_base_revision=True):
        self.tests = tests
        self.metric = metric
        self.aggregator = aggregator
        self.value_extractor = value_extractor
        self.use_base_revision = use_base_revision
        self.get_switch_points = get_switch_points


def hodges_lehmann_rps(context):
    rps = context['requests_per_sec']
    return stats.hodges_lehmann_median(rps)


class BaseWeightedSumAggregator(object):
    def __init__(self, weights):
        self.weights = list(copy.deepcopy(weights))
        super(BaseWeightedSumAggregator, self).__init__()

    def __call__(self, values):
        logging.info("Take average of {} with weights {}".format(values, self.weights))
        if len(self.weights) != len(values):
            raise RuntimeError(
                "Wrong len of values. Input len is {}, expected: {}".format(len(values), len(self.weights))
            )
        result = 0
        for weight, value in zip(self.weights, values):
            result += weight * value
        return result


class BaseAverageAggregator(BaseWeightedSumAggregator):
    def __init__(self):
        super(BaseAverageAggregator, self).__init__([1. / 3] * 3)


class FullCpuWeightedAggregator(BaseWeightedSumAggregator):
    def __init__(self):
        super(FullCpuWeightedAggregator, self).__init__(
            [BS_CORES * 1. / 3 / ALL_CORES] * 3 +
            [YABS_CORES * 1. / 3 / ALL_CORES] * 3 +
            [BSRANK_CORES * 1. / 3 / ALL_CORES] * 3
        )


class EmptyAggregator(object):
    def __call__(self, values):
        return values[0]


performance_params = (
    PlotParams(PERFORMANCE_TESTS, "RPS_HL", min, value_extractor=hodges_lehmann_rps, get_switch_points=False),
)
all_performance_params = tuple(
    PlotParams((test,), "RPS_HL", min, value_extractor=hodges_lehmann_rps)
    for test in PERFORMANCE_TESTS
)

size_params = (PlotParams(SIZE_TESTS, "Total size", max, get_switch_points=False),)
all_size_params = tuple(PlotParams((test,), "Total size", max) for test in SIZE_TESTS)

all_average_by_base_params = tuple(
    PlotParams(subset, "RPS_HL", BaseAverageAggregator(), value_extractor=hodges_lehmann_rps, get_switch_points=False)
    for subset in [BS_PERF_TESTS, YABS_PERF_TESTS, BSRANK_PERF_TESTS]
)
full_weighted_average_params = (
    PlotParams(PERFORMANCE_TESTS, "RPS_HL", FullCpuWeightedAggregator(), get_switch_points=False),
)

avg_perf_params_new = tuple(
    PlotParams(('YABS_SERVER_40_PERFORMANCE_AVG_{}'.format(role),), 'rps', EmptyAggregator())
    for role in ROLES
)
total_avg_weighted_params_new = (
    PlotParams(('YABS_SERVER_40_PERFORMANCE_TOTAL_AVG_WEIGHTED',), 'rps', EmptyAggregator()),
)

recipes = (
    performance_params + size_params + all_performance_params + all_size_params +
    all_average_by_base_params + full_weighted_average_params +
    avg_perf_params_new + total_avg_weighted_params_new
)
