import logging
import itertools

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.common.utils import get_resource_link, get_task_link
from sandbox.projects.yabs.qa.solomon.mixin import SolomonTaskMixin, SolomonTaskMixinParameters

from .report import create_report


logger = logging.getLogger(__name__)


class YabsServerCSVersionsMap(sdk2.Resource):
    name = "YABS_SERVER_CS_VERSIONS_MAP"
    releasable = True


class SolomonService(object):
    frontends = "frontends"
    cs = "cs"


DEFAULT_DESCRIPTION = {
    '1': 'Production',
    '2': 'Production',
    '3': 'Production Meta',
    '4': 'Production Meta',
    '5': 'Experiment',
    '6': 'Experiment Meta',
}


class ServiceVersionMonitor(SolomonTaskMixin, sdk2.Task):
    """A task to monitor Yabs server versions installed in production"""

    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 4096  # 4GiB or less

        environments = (
            environments.PipEnvironment('retrying'),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(SolomonTaskMixinParameters, sdk2.Parameters):

        with sdk2.parameters.Group("CS version settings") as cs_version_settings:
            project = sdk2.parameters.String("Samogon project", default="yabscs")
            descriptions = sdk2.parameters.Dict("Map namespace -> description", default=DEFAULT_DESCRIPTION)

        with sdk2.parameters.Group("Server version settings") as server_version_settings:
            nanny_dashboard = sdk2.parameters.String("Nanny dashboard to get services from", default="bsfront_production")
            resource_type = sdk2.parameters.String("Resource type to check", default="BS_RELEASE_TAR")

        with sdk2.parameters.Group("Auth options") as auth_options:
            nanny_token = sdk2.parameters.String("Nanny OAuth token vault name", default='nanny-oauth-token-server-version-monitor')

        with sdk2.parameters.Group("Debug options") as debug_options:
            debug_mode = sdk2.parameters.Bool("Debug mode", default=False)
            with debug_mode.value[True]:
                use_production_sandbox = sdk2.parameters.Bool("Use production sandbox", default=True)

    @staticmethod
    def create_cs_versions_report(project, descriptions, cs_version_helper):
        deployed_cs_resources = {
            namespace: cs_version_helper.get_meta_resource_id(project, namespace)
            for namespace in descriptions.keys()
        }

        rows = sorted([
            {
                "project": project,
                "namespace": namespace,
                "description": descriptions[namespace],
                "resource": {
                    "version": str(cs_version_helper.get_full_version_from_resource(resource_id)),
                    "link": get_resource_link(resource_id),
                    "state": cs_version_helper.get_resource_state(resource_id),
                },
            } for namespace, resource_id in deployed_cs_resources.items()
        ], key=lambda row: (row["project"], row["namespace"]))
        return rows

    def write_cs_versions_report(self, rows):
        report = create_report(rows, task_link=get_task_link(self.id))

        report_resource = YabsServerCSVersionsMap(self, "Versions table", "versions.html")
        report_resource_data = sdk2.ResourceData(report_resource)
        report_resource_data.path.write_bytes(report.encode("utf-8"))

    def on_execute(self):
        from sandbox.projects.yabs.release.version.cs import CSVersionHelper, get_stage_label
        from sandbox.projects.yabs.release.version.server import ServerVersionHelper

        if self.Parameters.debug_mode and self.Parameters.use_production_sandbox:
            sandbox_client = None
        else:
            sandbox_client = self.server

        nanny_token = sdk2.Vault.data(self.Parameters.nanny_token)

        server_version_helper = ServerVersionHelper(sandbox_client=sandbox_client, nanny_token=nanny_token)
        cs_version_helper = CSVersionHelper(sandbox_client=sandbox_client)

        cs_versions = {  # TODO: Remove after switching to new version dashboard with clusterapi's info
            namespace: cs_version_helper.get_cs_version(self.Parameters.project, namespace)
            for namespace in self.Parameters.descriptions.keys()
        }

        server_versions = server_version_helper.get_deployed_versions(self.Parameters.nanny_dashboard, self.Parameters.resource_type)
        server_version_sensors = [  # TODO: Remove after switching to new version dashboard with clusterapi's info
            {
                "labels": {
                    "version": str(version),
                    "service_id": service_id,
                    "stage": server_version_helper.get_stage_label(service_id),
                },
                "value": 1
            }
            for service_id, version in server_versions.items()
        ]
        cs_version_sensors = [  # TODO: Remove after switching to new version dashboard with clusterapi's info
            {
                "labels": {
                    "version": str(version),
                    "service_id": "{}_{}".format(self.Parameters.project, namespace),
                    "stage": get_stage_label(namespace),
                },
                "value": 1
            }
            for namespace, version in cs_versions.items()
        ]
        cs_versions_by_hosts = {
            "{}_{}".format(self.Parameters.project, namespace): cs_version_helper.get_deployed_cs_versions(self.Parameters.project, namespace)
            for namespace in self.Parameters.descriptions.keys()
        }
        cs_version_by_host_sensors = list(itertools.chain(*[
            [
                {
                    "labels": {
                        "version": str(version),
                        "service_id": service_id,
                        "stage": get_stage_label(service_id),
                    },
                    "value": float(1) / len(versions)
                } for version in versions
            ]
            for service_id, versions in cs_versions_by_hosts.items()
        ]))

        # Old metrics
        self.solomon_push_client.add(server_version_sensors, cluster='production', service=SolomonService.frontends)
        self.solomon_push_client.add(cs_version_sensors, cluster='production', service=SolomonService.cs)

        # New metrics
        self.solomon_push_client.add(server_version_sensors, cluster='frontends', service="versions")
        self.solomon_push_client.add(cs_version_by_host_sensors, cluster="cs", service="versions")

        rows = self.create_cs_versions_report(self.Parameters.project, self.Parameters.descriptions, cs_version_helper)
        self.write_cs_versions_report(rows)
