import logging
from functools import wraps

import requests


def simple_cache(func):
    cache = {}

    @wraps(func)
    def wrapped(*args):
        if args in cache:
            return cache[args]
        else:
            result = func(*args)
            cache[args] = result
            return result

    return wrapped


class ResponseWithJSONErrorsLogging(object):
    class AccessProxy(object):
        WRAPPABLE_TYPES = [dict, list]

        def __init__(self, json_object, response):
            self._wrapped = json_object
            self._response = response

        def __getitem__(self, item):
            try:
                item_value = self._wrapped[item]

                # For existing key the method should return a new proxy instance (for container types)
                # or the value itself (for scalar types)
                if any(isinstance(item_value, t) for t in self.WRAPPABLE_TYPES):
                    return type(self)(
                        json_object=self._wrapped[item],
                        response=self._response
                    )
                else:
                    return item_value
            except Exception:
                if item != len(self._wrapped):
                    # Hack: skip error logging when iterating over a list and reaching it's end (otherwise this error
                    # message will be created before every StopIteration)
                    error_msg = (
                        'Can\'t get field from JSON object: field_name={field!r}, json_object={json_object!r}'
                        ', request_url={request.url!r}, request_body={request.body!r}'
                        ', response_status_code={response.status_code!r}, response_text={response.text!r}'
                    ).format(
                        field=item,
                        json_object=self._wrapped,
                        request=self._response.request,
                        response=self._response,
                    )
                    logging.error(error_msg)
                raise

        def __getattr__(self, item):
            return getattr(self._wrapped, item)

        def __len__(self):
            return len(self._wrapped)

    def __init__(self, response):
        self._response = response

    def __repr__(self):
        return repr(self._response)

    def __getattr__(self, item):
        return getattr(self._response, item)

    def json(self, **kwargs):
        try:
            json_data = self._response.json(**kwargs)
            return self.AccessProxy(
                json_object=json_data,
                response=self
            )
        except ValueError:
            error_msg = (
                'Can\'t decode JSON response: request_url={request.url!r}, request_body={request.body!r}'
                ', response_status_code={response.status_code!r}, response_text={response.text!r}'
            ).format(
                request=self._response.request,
                response=self._response
            )
            logging.error(error_msg)
            raise


class SessionWithLogging(requests.Session):
    def request(self, *args, **kwargs):
        logging.debug('Sending HTTP request: %r, %r', args, kwargs)
        response = super(SessionWithLogging, self).request(*args, **kwargs)
        logging.debug('Got HTTP response: status=%r, text=%r', response.status_code, response.text)
        response.raise_for_status()
        return ResponseWithJSONErrorsLogging(response)
