# -*- coding: utf-8 -*-

import logging
import re
import os

from sandbox import sdk2
from sandbox.common.types import resource as ctr
import sandbox.common.types.task as ctt
import sandbox.projects.yabs.qa.utils.general as yabs_qa_utils
from sandbox.sandboxsdk import ssh
from sandbox.sdk2.vcs import svn

from sandbox.projects.common.binary_task import (
    LastBinaryTaskRelease,
    binary_release_parameters
)
from sandbox.projects.yabs.SysConstLifetime.const_proto.const_proto_helper import (
    ConstProtoHelper, StartrekHelper, EConstProtoMode, DO_NOT_CHANGE_OWNER, InvalidOwner, ConstantsLimitExceeded
)
from sandbox.projects.yabs.SysConstLifetime.lib.arcadia_helper import ArcadiaHelper
from sandbox.projects.yabs.SysConstLifetime.lib.abc_client import ABCClient

from sandbox.projects.yabs.SysConstLifetime.lib.staff_client import StaffClient
from sandbox.projects.yabs.SysConstLifetime.lib.limits_helper import LimitsHelper

from sandbox.projects.yabs.SysConstLifetime.lib.utils import join_file_path_with_checkout_dir


class ConstProtoUpdate(LastBinaryTaskRelease, sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024

        semaphores = ctt.Semaphores(
            acquires=[
                ctt.Semaphores.Acquire(name='CONST_PROTO_UPDATE_QUEUE')
            ]
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Technical parameters', collapse=True) as technical_parameters:
            _lbrp = binary_release_parameters(stable=True)
            checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl('SVN url for arcadia', required=True)
            commit_user = sdk2.parameters.String('Commit user', default='robot-yabs-const', required=True)
            commit_user_ssh_vault_key = sdk2.parameters.String('SSH vault key', default='ssh_key', required=True)
            startrek_token_vault_key = sdk2.parameters.String('Startrek token', default='st_token', required=True)
            abc_token_vault_key = sdk2.parameters.String('ABC token', default='abc_token', required=True)
            abc_id_list = sdk2.parameters.List("ABC IDs to get members from", default=[179, 30906], required=True)
            proto_file_path = sdk2.parameters.String('Path to *_const.proto file from arcadia root', default='yabs/server/proto/quality/sys_const.proto', required=True)
            owners_file_path = sdk2.parameters.String('Path to file with constant owners from arcadia root', default='yabs/server/proto/quality/sys_const.owners.json', required=True)
            limits_file_path = sdk2.parameters.String('Path to file with constant limits from arcadia root', default='yabs/server/proto/quality/sys_const.limits.json', required=True)

        ticket = sdk2.parameters.StrictString(
            'Startrek ticket',
            required=True,
            regexp=r'^[A-Z]+-\d+$'
        )
        with sdk2.parameters.RadioGroup('Task mode', required=True) as task_mode:
            task_mode.values.ADD = task_mode.Value('ADD')
            task_mode.values.UPDATE = task_mode.Value('UPDATE')
        make_task_author_owner = sdk2.parameters.Bool('Task author is the owner of the constant', default=True)
        with make_task_author_owner.value[False]:
            who_to_make_owner = sdk2.parameters.String('Who is the owner of the constant?', default=DO_NOT_CHANGE_OWNER)
        const_name = sdk2.parameters.StrictString(
            'Constant name',
            required=True,
            regexp=r'^[A-Z][a-zA-Z0-9]*$'
        )
        const_value = sdk2.parameters.Integer('Constant value in proto file', required=True)
        with sdk2.parameters.RadioGroup('Constant type', required=True) as const_type:
            const_type.values.FEATURE_FLAG = const_type.Value('FEATURE_FLAG')
            const_type.values.CONSTANT_VALUE = const_type.Value('CONSTANT_VALUE')

        db_null_path = sdk2.parameters.String(
            'Path to db_null_snapshot.py file from arcadia root',
            default='yabs/server/test/qabs_bsserver_pytest/data/db_null_snapshot.py',
            required=True
        )

        is_set_const_in_db_null = sdk2.parameters.Bool('Set constant in db_null_snapshot', default=True)
        with is_set_const_in_db_null.value[True]:
            const_value_db_null = sdk2.parameters.Integer(
                'Constant value in db_null_snapshot (please enable feature flags in tests)',
                default=1,
                required=True
            )
            const_description_db_null = sdk2.parameters.String(
                'Constant description',
                default='',
                required=True
            )

    @property
    def binary_executor_query(self):
        return {
            "attrs": {"task_type": "SYS_CONST_LIFETIME", "released": self.Parameters.binary_executor_release_type},
            "state": [ctr.State.READY]
        }

    def _author_can_resolve(self):
        if self.staff_client.get_person_is_dismissed(self.author):
            return False

        if re.match(r'^robot-', self.author, re.IGNORECASE):
            return False

        return True

    def on_save(self):
        super(ConstProtoUpdate, self).on_save()

    def on_execute(self):
        super(ConstProtoUpdate, self).on_execute()

        from startrek_client import Startrek

        self.staff_client = StaffClient(sdk2.Vault.data(self.Parameters.commit_user, self.Parameters.startrek_token_vault_key))

        arcadia_helper = ArcadiaHelper(
            arcadia_client=svn.Arcadia,
            ssh_client=ssh,
            commit_user=self.Parameters.commit_user,
            commit_user_ssh_vault_key=self.Parameters.commit_user_ssh_vault_key,
            logger=logging.getLogger(__name__)
        )

        assert os.path.dirname(self.Parameters.proto_file_path) == os.path.dirname(self.Parameters.owners_file_path)
        assert os.path.dirname(self.Parameters.proto_file_path) == os.path.dirname(self.Parameters.limits_file_path)

        checkout_dir_local = arcadia_helper.checkout(os.path.dirname(self.Parameters.proto_file_path))

        get_path = lambda path: join_file_path_with_checkout_dir(checkout_dir_local, path)

        proto_local_path = get_path(self.Parameters.proto_file_path)
        owners_local_path = get_path(self.Parameters.owners_file_path)
        limits_local_path = get_path(self.Parameters.limits_file_path)

        startrek_client = Startrek(useragent='Sandbox task CONST_PROTO_UPDATE', token=sdk2.Vault.data(self.Parameters.commit_user, self.Parameters.startrek_token_vault_key))
        startrek_helper = StartrekHelper(startrek_client=startrek_client)
        const_proto_helper = ConstProtoHelper(
            arcadia_helper=arcadia_helper,
            startrek_helper=startrek_helper,
            ticket=self.Parameters.ticket,
            proto_local_path=proto_local_path,
            owners_local_path=owners_local_path,
            db_null_path=self.Parameters.db_null_path
        )

        abc_client = ABCClient(
            token=sdk2.Vault.data(self.Parameters.commit_user, self.Parameters.abc_token_vault_key),
            id_list=self.Parameters.abc_id_list
        )

        const_name = self.Parameters.const_name
        mode = EConstProtoMode.fromstring(self.Parameters.task_mode)
        if mode == EConstProtoMode.ADD:
            self.set_info('Adding constant {name} with value {value} and type {type}...'.format(
                name=const_name,
                value=self.Parameters.const_value,
                type=self.Parameters.const_type)
            )
        elif mode == EConstProtoMode.UPDATE:
            self.set_info('Updating constant {name} with new value {value} and new type {type}...'.format(
                name=const_name,
                value=self.Parameters.const_value,
                type=self.Parameters.const_type)
            )

        owner = self.author if self.Parameters.make_task_author_owner else self.Parameters.who_to_make_owner
        if owner is None or owner == '':
            raise InvalidOwner('Owner cannot be empty')
        if re.match(r'^robot-', owner, re.IGNORECASE):
            raise InvalidOwner('Owner must not be robot, found {}'.format(owner))
        if owner != DO_NOT_CHANGE_OWNER:
            if self.staff_client.get_person_is_dismissed(owner):
                raise InvalidOwner('{} is dismissed or does not exist'.format(owner))
            if owner not in abc_client.get_abc_members():
                raise InvalidOwner('{} is not in abc_members'.format(owner))
        else:
            owner = const_proto_helper.get_current_constant_owner(const_name, mode)

        self.set_info('Owner of the constant is {}'.format(owner))

        if mode == EConstProtoMode.ADD:
            limits_helper = LimitsHelper(
                owners_local_path=owners_local_path,
                proto_local_path=proto_local_path,
                limits_local_path=limits_local_path,
                staff_client=self.staff_client
            )

            limit_info = limits_helper.get_constant_limit_info_by_owner(owner)

            if limit_info.owned >= limit_info.limit:
                message = 'ERROR: у вас уже есть {owned} констант, вы превысили лимит в {limit}' \
                    ' констант для пользователя {owner} (URL лимитирующей группы: {group})\n' \
                    'Пожалуйста, удалите из кода свои старые константы, чтобы упростить код и исключить возможность' \
                    ' рассинхронизации тестов с продом\n' \
                    'Фича-константы это техдолг, который необходимо отдавать. Подробнее https://clubs.at.yandex-team.ru/yabs-dev/61'.format(
                    owned=limit_info.owned,
                    limit=limit_info.limit,
                    owner=owner,
                    group=limit_info.group_url
                )
                raise ConstantsLimitExceeded(message)

            if limit_info.owned >= limit_info.warning_ratio * limit_info.limit:
                self.set_info('WARNING: У вас уже {owned} констант типа FEATURE_FLAG, вы приближаетесь к ограничению в {limit}'
                                ' констант. Пожалуйста, удалите из кода свои старые константы, чтобы упростить код и исключить возможность'
                                ' рассинхронизации тестов с продом.\n Подробнее https://clubs.at.yandex-team.ru/yabs-dev/61'.format(
                    owned=limit_info.owned,
                    limit=limit_info.limit)
                )

        review_url, db_null_replace_count = const_proto_helper.generate_and_commit(
            const_name=self.Parameters.const_name,
            const_value=self.Parameters.const_value,
            const_type=self.Parameters.const_type,
            const_owner=owner,
            mode=mode,
            need_set_const_in_db_null=self.Parameters.is_set_const_in_db_null,
            const_value_db_null=self.Parameters.const_value_db_null,
            const_description_db_null=self.Parameters.const_description_db_null,
            diff_resolve_login=self.author if self._author_can_resolve() else owner
        )

        review_url = yabs_qa_utils.html_hyperlink(link=review_url, text=review_url)
        self.set_info(
            'Review url: {review_url}\n'
            'If you added constant to db_null_snapshot, '
            'you can safely merge review with red TEST_DIFF_CHKDB diff in "constant".'.format(review_url=review_url),
            do_escape=False
        )
        if mode == EConstProtoMode.ADD and db_null_replace_count > 0:
            self.set_info('WARNING: constant with same name already existed in db_null and was updated')
