from enum import Enum
import json
import logging
import os
import re

from sandbox.projects.yabs.SysConstLifetime.lib.filepipe import FilePipe
from sandbox.projects.yabs.SysConstLifetime.lib.utils import get_all_constant_ids_and_options, join_file_path_with_checkout_dir
import sandbox.common.errors as sandbox_errors


logger = logging.getLogger(__name__)

DO_NOT_CHANGE_OWNER = 'NO CHANGE'


class StartrekHelper:
    def __init__(self, startrek_client):
        self._startrek_client = startrek_client

    def ensure_user_follows_issue(self, login, ticket):
        """
        Obtains credentials and startrek ticket data, adds login to followers because
        svn requires commit user to be follower or assignee of the given ticket.
        """
        from startrek_client.exceptions import Forbidden, NotFound

        try:
            issue = self._startrek_client.issues[ticket]
        except Forbidden:
            error_message = 'Commit user {login} has no access to startrek issue {ticket}. ' \
                            'Try to add him to ticket assignee or followers.'\
                .format(
                    login=login,
                    ticket=ticket
                )
            raise sandbox_errors.TaskError(error_message)
        except NotFound:
            error_message = 'Issue {ticket} not found'.format(ticket=ticket)
            raise sandbox_errors.TaskFailure(error_message)
        issue.update(followers=issue.followers + [login])


class ConstantNotFoundError(Exception):
    pass


class ConstantAlreadyExistsError(Exception):
    pass


class ConstantsLimitExceeded(Exception):
    pass


class InvalidOwner(Exception):
    pass


class CommitError(Exception):
    pass


class InvalidConstProtoMode(Exception):
    pass


class EConstProtoMode(Enum):
    UNKNOWN = 0
    ADD = 1
    UPDATE = 2

    @classmethod
    def fromstring(cls, s):
        try:
            return cls[s]
        except KeyError:
            return cls.UNKNOWN


class ConstProtoHelper:
    """
    Method generate_and_commit obtains constant file from svn, generates new default value and creates new PR with changed default and type.
    All other methods are some additional helpers, which are tested in test_const_proto_helper.
    """
    def __init__(self, arcadia_helper, startrek_helper, ticket, proto_local_path, owners_local_path, db_null_path):
        self._arcadia_helper = arcadia_helper
        self._startrek_helper = startrek_helper
        self._ticket = ticket
        self._proto_local_path = proto_local_path
        self._owners_local_path = owners_local_path
        self._db_null_path = db_null_path

    def _get_const_id_options(self, file_obj, const_name):
        """
        Reads constant file from file_obj, looks for const_name and returns its id and options,
        raises ConstantNotFoundError if constant was not found.
        """
        answer = get_all_constant_ids_and_options(file_obj)
        file_obj.seek(0)
        if const_name not in answer:
            raise ConstantNotFoundError(const_name)
        return answer[const_name]

    def _write_with_new_const(self, in_file_obj, out_file_obj, const_name, const_id, options):
        """
        Reads from constant file and writes the same file with given const_id and options for const_name.
        """
        def build_const_line():
            options_str = ', '.join(('{k} = {v}'.format(k=k, v=v) for k, v in options.items()))
            s = u'    optional int64 {const} = {const_id} [{options_str}];'.format(const=const_name, const_id=const_id, options_str=options_str)
            return s

        found = False
        for line in in_file_obj:
            if 'int64 ' + const_name + ' =' in line or ('// LAST: ' in line and not found):
                out_file_obj.write(build_const_line() + '\n')
                found = True
                if '// LAST: ' in line:
                    out_file_obj.write(u'} // LAST: ' + str(const_id) + '\n')
            else:
                out_file_obj.write(line)
        in_file_obj.seek(0)

    def _update_default_in_file(self, in_file_obj, out_file_obj, const_name, const_value, const_type):
        """
        Obtains const_name id and options, updates them with new values and writes new file to out_file_obj.
        """
        logger.info('Updating constant {} with new value {} and new type {}'.format(const_name, const_value, const_type))
        const_id, options = self._get_const_id_options(in_file_obj, const_name)
        logger.info('Found constant {} with id {} and options {}'.format(const_name, const_id, options))
        options['default'] = const_value
        options['(type)'] = const_type
        self._write_with_new_const(in_file_obj, out_file_obj, const_name, const_id, options)

    def _get_last_const_id(self, file_obj):
        """
        Gets the biggest index used in protobuf
        """
        const_pattern = r'^    optional int64 \w+ = (-?[0-9]+)'
        res = max(int(index) for index in re.findall(const_pattern, file_obj.read(), re.MULTILINE))
        file_obj.seek(0)
        return res

    def _check_const_does_not_exist(self, file_obj, const_name):
        for line in file_obj:
            if (' ' + const_name + ' ') in line:
                file_obj.seek(0)
                raise ConstantAlreadyExistsError(line)
        file_obj.seek(0)

    def _add_const_to_file(self, in_file_obj, out_file_obj, const_name, const_value, const_type):
        logger.info('Adding new const {} with value {} and type {}'.format(const_name, const_value, const_type))
        self._check_const_does_not_exist(in_file_obj, const_name)
        const_id = self._get_last_const_id(in_file_obj) + 1
        logger.info('Constant id will be {}'.format(const_id))
        options = {'default': const_value, '(type)': const_type}
        self._write_with_new_const(in_file_obj, out_file_obj, const_name, const_id, options)

    def _set_const_owner(self, in_file_obj, out_file_obj, const_name, const_owner):
        owners = json.load(in_file_obj)
        owners['owners'][const_name] = const_owner
        out_file_obj.write(json.dumps(owners, indent=4, sort_keys=True, separators=(',', ': ')))
        out_file_obj.write('\n')
        in_file_obj.seek(0)

    def _set_const_in_db_null(self, db_null_content, const_name, const_value, const_description, is_sys_const):
        def generate_replacement(const):
            yield const
            while True:
                yield ''

        const_type = 'RuntimeConstant' if is_sys_const else 'MkdbConstant'
        const_new = "base.{const_type}(_from_snapshot=True, Name='{const_name}', " \
                    "Value={const_value}, Description='{const_description}')\n".format(
                        const_type=const_type,
                        const_name=const_name,
                        const_value=const_value,
                        const_description=str(const_description).encode("string_escape")
                    )

        g = generate_replacement(const_new)
        result, replace_count = re.subn(
            pattern=r"^base\.{const_type}\(.*?Name='{const_name}'.*?\)\n".format(
                const_type=const_type,
                const_name=const_name
            ),
            repl=lambda _: g.next(),
            string=db_null_content,
            flags=re.M
        )

        if replace_count == 0:
            result, match_count = re.subn(
                pattern=r'(^base\.{}\(.*?\)$)'.format(const_type),
                repl='{}\\g<1>'.format(const_new),
                string=db_null_content,
                count=1,
                flags=re.M
            )

            assert match_count == 1

        return result, replace_count

    def _update_db_null(self, db_null_local_path, const_name, const_value, const_description):
        with FilePipe(db_null_local_path) as f:
            result, repl_count = self._set_const_in_db_null(
                f.in_io.read(),
                const_name,
                const_value,
                const_description,
                self._proto_local_path.endswith('sys_const.proto')
            )

            f.out_io.write(result)
            f.in_io.seek(0)

        return repl_count

    def get_current_constant_owner(self, const_name, mode):
        if mode == EConstProtoMode.ADD:
            raise InvalidOwner('Can\'t use {} with new constant'.format(DO_NOT_CHANGE_OWNER))

        if mode == EConstProtoMode.UPDATE:
            with open(self._owners_local_path, 'r') as f:
                owners = json.load(f)['owners']
                if const_name not in owners:
                    raise InvalidOwner('Can\'t found constant owner in {}. Maybe you update not existing constant?'.format(self._owners_local_path))
                return owners[const_name]

        raise InvalidConstProtoMode('Invalid Const Proto Mode. Please, choose ADD or UPDATE.')

    def generate_and_commit(
        self,
        const_name,
        const_value,
        const_type,
        const_owner,
        mode,
        need_set_const_in_db_null,
        const_value_db_null,
        const_description_db_null,
        diff_resolve_login
    ):
        """
        Performs following actions:
        1. Checkout proto file
        2. Find line
        3. Generate line with new value and type
        4. Write file
        5. Commit changes of the file with some message

        Items 2-4 are proceeded in self._update_default_in_file method.
        """
        logger.info('Parsing local file \'{file}\', looking for constant {const}'.format(file=self._proto_local_path, const=const_name))
        with FilePipe(self._proto_local_path) as proto_file:
            if mode == EConstProtoMode.UPDATE:
                self._update_default_in_file(proto_file.in_io, proto_file.out_io, const_name, const_value, const_type)
            elif mode == EConstProtoMode.ADD:
                self._add_const_to_file(proto_file.in_io, proto_file.out_io, const_name, const_value, const_type)
            else:
                raise RuntimeError('Unsupported mode of ConstProtoHelper: {}'.format(mode))

        with FilePipe(self._owners_local_path, read_mode='rb', write_mode='wb') as owners_file:
            self._set_const_owner(owners_file.in_io, owners_file.out_io, const_name, const_owner)

        self._startrek_helper.ensure_user_follows_issue(self._arcadia_helper.get_commit_user(), self._ticket)

        local_paths_to_commit = [self._proto_local_path, self._owners_local_path]

        db_null_replace_count = 0
        if need_set_const_in_db_null:
            checkout_dir_local = self._arcadia_helper.checkout(os.path.dirname(self._db_null_path))
            db_null_local_path = join_file_path_with_checkout_dir(checkout_dir_local, self._db_null_path)

            db_null_replace_count = self._update_db_null(
                db_null_local_path,
                const_name,
                const_value_db_null,
                const_description_db_null
            )

            local_paths_to_commit.append(db_null_local_path)

        review_url = self._arcadia_helper.commit(
            paths=local_paths_to_commit,
            commit_message='{ticket}: New default of {const} is {value} [diff-resolver:{login}]'.format(
                ticket=self._ticket,
                const=const_name,
                value=const_value,
                login=diff_resolve_login
            ),
            no_tests=(mode == EConstProtoMode.ADD)
        )

        if review_url is None:
            raise CommitError('Could not commit changes. Maybe files have no changes.')

        return review_url, db_null_replace_count
