# -*- coding: utf-8 -*-

import logging
import textwrap

from sandbox import sdk2


class CsSettingsLifetime(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        dry_run = sdk2.parameters.Bool("Dry run, dont make anything in st", default=False, required=True)
        st_queue = sdk2.parameters.String("ST queue", default="BSSERVER", required=True)
        st_token = sdk2.parameters.String("ST token vault name", default="st_token", required=True)
        yt_token = sdk2.parameters.String("YT token vault name", default="yt_token", required=True)
        yt_cluster = sdk2.parameters.String("YT cluster", default="locke", required=True)
        yt_path = sdk2.parameters.String("Yt path", default="//home/samogon/yabscs_production/yabs-cs/settings/v5_head", required=True)

    def _create_st_client(self):
        from startrek_client import Startrek

        token = sdk2.Vault.data(self.author, self.Parameters.st_token)
        return Startrek(useragent='Sandbox SYS_CONST_LIFETIME task', token=token)

    def _create_yt_client(self):
        from yt.wrapper.client import Yt

        token = sdk2.Vault.data(self.author, self.Parameters.yt_token)
        return Yt(proxy=self.Parameters.yt_cluster, token=token)

    def _get_settings(self):
        return self.yt_client.get(self.Parameters.yt_path)

    def _create_issue(self, queue, assignee, summary, description, tags):
        if self.Parameters.dry_run:
            return
        try:
            issue = self.st_client.issues.create(
                queue=queue,
                assignee=assignee,
                summary=summary,
                description=description,
                tags=tags
            )
        except Exception as e:
            logging.warn('failed create issue with %s: %s', type(e), e)
        else:
            logging.info('created issue %s', issue.key)

    def _update_issue(self, issue, summary, description, tags):
        if self.Parameters.dry_run:
            return
        try:
            issue.update(
                summary=summary,
                description=description,
                tags=tags
            )
        except Exception as e:
            logging.warn('failed update issue %s with %s: %s', issue.key, type(e), e)
        else:
            logging.info('updated issue %s', issue.key)

    def find_issues(self, name, setting_tags):
        queue = self.Parameters.st_queue
        detail_tags = [u'CsSettingsLifetime', name] + setting_tags
        issues = self.st_client.issues.find('queue: {queue} tags: {name}'.format(queue=queue, name=name))
        logging.info('Found issues by setting name {}'.format(len(issues)))
        logging.info('Detail tags: [{}]'.format(','.join(detail_tags)))
        detail_match = filter(lambda x: x.tags == detail_tags, issues)
        if len(detail_match) > 0:
            logging.info('Found issue with details tags {} (total matched {})'.format(detail_match[0].key, len(detail_match)))
            return detail_match
        return filter(lambda x: x.tags == [u'CsSettingsLifetime', name], issues)

    def _call_st(self, name, setting_tags, assignee, summary, description):
        tags = [u'CsSettingsLifetime', name] + setting_tags
        issues = self.find_issues(name, setting_tags)

        logging.info('processing %s', name)
        if len(issues) == 0:
            logging.info('issue not found, create it')
            logging.info('summary(%s): %s', type(summary), summary)
            logging.info('description(%s): %s', type(description), description)
            logging.info('tags(%s): %s', ','.join(str(type(tag)) for tag in tags), tags)
            self._create_issue(self.Parameters.st_queue, assignee, summary, description, tags)
            return
        issue = issues[0]
        old_summary = issue.summary
        old_description = issue.description
        old_tags = issue.tags
        if old_summary != summary or old_description != description or set(old_tags) != set(tags):
            logging.info('issue %s found, update it', issue.key)
            logging.info('old_summary(%s): %s', type(old_summary), old_summary)
            logging.info('summary(%s): %s', type(summary), summary)
            logging.info('old_description(%s): %s', type(old_description), old_description)
            logging.info('description(%s): %s', type(description), description)
            logging.info('old_tags(%s): %s', ','.join(str(type(tag)) for tag in old_tags), old_tags)
            logging.info('tags(%s): %s', ','.join(str(type(tag)) for tag in tags), tags)
            self._update_issue(issue, summary, description, tags)
            return
        logging.info('issue %s is in actual state', issue.key)

    def _reset_to_default(self, name, metadata, tags):
        tags = [t.replace(' ', '_') for t in tags]
        self._call_st(
            name.decode('utf-8'),
            tags,
            metadata['login'].decode('utf-8'),
            'Вернуть настройку {} в дефолт'.format(name).decode('utf-8'),
            textwrap.dedent('''
                Настройка, установленная {}, не совпадает с дефолтом.
                Необходимо это исправить - обновить дефолт или сбросить настройку.

                Save comment: {}
            '''.format(
                metadata['time'],
                metadata['save_comment'],
            )).decode('utf-8')
        )

    def on_execute(self):
        self.yt_client = self._create_yt_client()
        self.st_client = self._create_st_client()

        settings = self._get_settings()

        for key, values in settings['settings'].items():
            for value in values:
                if 'metadata' in value:
                    self._reset_to_default(key, value['metadata'], value['tags'] if 'tags' in value else [])
