import json
import logging
from collections import defaultdict
from sandbox.projects.yabs.SysConstLifetime.lib.utils import get_all_constant_ids_and_options


logger = logging.getLogger('LimitsHelper')


class LimitsHelper:
    def __init__(self, owners_local_path, proto_local_path,  limits_local_path, staff_client):
        self._owners = json.load(open(owners_local_path, 'rb'))['owners']
        self._limits = json.load(open(limits_local_path, 'rb'))
        self._staff_client = staff_client
        self._const_options = {}
        ids_and_options = get_all_constant_ids_and_options(open(proto_local_path, 'r'))
        for const_name, (const_id, const_options) in ids_and_options.iteritems():
            self._const_options[const_name] = const_options

    def _get_const_owner_by_name(self, const_name):
        return self._owners.get(const_name)

    def _get_group_url_by_owner(self, owner):
        return self._staff_client.get_person_group_url(owner)

    def _get_group_name_by_owner(self, owner):
        return self._staff_client.get_person_group_name(owner)

    def get_constants_owned_by_owner(self, const_owner, const_type='FEATURE_FLAG'):
        count = 0
        for name, options in self._const_options.iteritems():
            owner = self._get_const_owner_by_name(name)
            if owner is not None and owner == const_owner and options['(type)'] == const_type:
                count += 1
        return count

    def get_constant_limit_by_group(self, group, const_type='FEATURE_FLAG'):
        limits = self._limits['staff_group_limits'][const_type]
        return limits.get(group)

    def get_constant_warning_ratio_by_group(self, group, const_type='FEATURE_FLAG'):
        warnings = self._limits['staff_group_warnings_ratio'][const_type]
        return warnings.get(group, warnings['default'])

    class LimitInfo:
        def __init__(self, owned, limit, warning_ratio, group_url, group_name):
            self.owned = owned
            self.limit = limit
            self.warning_ratio = warning_ratio
            self.group_url = group_url
            self.group_name = group_name

    def get_constant_limit_info_by_owner(self, const_owner, const_type='FEATURE_FLAG'):
        owned = self.get_constants_owned_by_owner(const_owner)
        while const_owner is not None:
            logger.info('Currently looking at {}'.format(const_owner))
            group_url = self._get_group_url_by_owner(const_owner)
            limit = self.get_constant_limit_by_group(group_url, const_type=const_type)
            if limit is not None:
                warning_ratio = self.get_constant_warning_ratio_by_group(group_url, const_type=const_type)
                group_name = self._get_group_name_by_owner(const_owner)
                return self.LimitInfo(owned, limit, warning_ratio, group_url, group_name)
            const_owner = self._staff_client.get_person_chief(const_owner)
        limit = self.get_constant_limit_by_group('default', const_type=const_type)
        warning_ratio = self.get_constant_warning_ratio_by_group('default', const_type=const_type)
        return self.LimitInfo(owned, limit, warning_ratio, 'yandex', 'NO LIMIT FOUND')

    def get_owners(self):
        return set(self._owners.values()) - set(['HAS NO OWNER'])

    def get_sys_const_lifetime_stats(self):
        """
        Returns a dict of dicts dict[group_url][login] = number of alive constants.
        """
        answer = defaultdict(lambda: defaultdict(int))
        for name, options in self._const_options.iteritems():
            owner = self._get_const_owner_by_name(name)
            if owner is not None and owner != 'HAS NO OWNER':
                group = self._staff_client.get_person_group_url(owner)
                if owner not in answer[group]:
                    answer[group][owner] = self.get_constants_owned_by_owner(owner)
        return answer
