from abc import ABCMeta, abstractmethod
from datetime import datetime
import json
import logging
import re

from sandbox.projects.yabs.SysConstLifetime.lib.utils import commit_changes, MAX_DAYS_WITHOUT_ABC
from sandbox.projects.yabs.SysConstLifetime.lib.filepipe import FilePipe

REASSIGN_IGNORE = {'poldnev', 'sankear'}


class MakeRelevantHelper():
    __metaclass__ = ABCMeta

    def __init__(self, json_local_path, arcadia_helper, blame, dry_run):
        self._json_local_path = json_local_path
        self._arcadia_helper = arcadia_helper
        self._blame = blame
        self._dry_run = dry_run

    @abstractmethod
    def _update_data(self, data, const, value):
        pass

    @abstractmethod
    def _get_commit_msg(self):
        return None

    def make_relevant(self):
        with FilePipe(self._json_local_path) as json_file:
            data = json.load(json_file.in_io)
            for const, value in self._blame.items():
                self._update_data(data, const, value)
            json_file.out_io.write(json.dumps(data, indent=4, sort_keys=True, separators=(',', ': ')))
            json_file.out_io.write('\n')
            json_file.in_io.seek(0)
        return commit_changes(
            path=[self._json_local_path],
            msg=self._get_commit_msg(),
            arcadia_helper=self._arcadia_helper,
            dry_run=self._dry_run
        )


class MakeOwnersRelevantHelper(MakeRelevantHelper):
    def __init__(self, owners_local_path, users_local_path, arcadia_helper, blame, abc_client, dry_run=False):
        super(MakeOwnersRelevantHelper, self).__init__(owners_local_path, arcadia_helper, blame, dry_run)
        self._abc_members = abc_client.get_abc_members()

        with open(users_local_path, 'r') as f:
            self._users = json.load(f)

        self._reassign_users_cache = {}

    def _get_commit_msg(self):
        return 'Update constants owners information in'

    def _update_data(self, owners, const, value):
        if const not in owners['owners']:
            logging.info('Set owner of const {} as {}'.format(const, value['author']))
            owners['owners'][const] = value['author']
        if owners['owners'][const] == "HAS NO OWNER":
            logging.info('Found HAS NO OWNER, skip set owner of const {} : {} -> {}'.format(const, owners['owners'][const], value['author']))
            return
        if re.match(r'^robot-', owners['owners'][const], re.IGNORECASE):
            logging.info('{robot_name} found as owner of const {constant}, skip it and change to HAS NO OWNER'.format(
                robot_name=owners['owners'][const],
                constant=const)
            )
            owners['owners'][const] = 'HAS NO OWNER'
            return
        if owners['owners'][const] in self._abc_members:
            logging.info('{} found in abc members, skip staff check'.format(owners['owners'][const]))
            return

        user_login = owners['owners'][const]

        logging.info('{} not found in abc members, found leader from user.json'.format(user_login))

        owners['owners'][const] = self._get_reassign(user_login)

    def _get_reassign(self, user_login):
        reassign = self._reassign_users_cache.get(user_login)
        if reassign is not None:
            return reassign

        user_info = self._users.get(user_login)
        if user_info is None:
            self._save_to_cache(user_login, user_login)
            return user_login

        if len(user_info['days_without_abc']) < MAX_DAYS_WITHOUT_ABC:
            self._save_to_cache(user_login, user_login)
            return user_login

        for lead in reversed(user_info['leaders']):
            if lead in REASSIGN_IGNORE:
                continue

            if lead in self._abc_members:
                self._save_to_cache(user_login, lead)
                return lead

            lead_info = self._users.get(lead)
            if lead_info is None:
                continue

            if len(lead_info['days_without_abc']) < MAX_DAYS_WITHOUT_ABC:
                self._save_to_cache(user_login, user_login)
                return user_login

        self._save_to_cache(user_login, user_login)
        return user_login

    def _save_to_cache(self, key, value):
        self._reassign_users_cache[key] = value
        logging.info('{owner_old} reassign to {owner_new}'.format(
            owner_old=key,
            owner_new=value
        ))


class MakeMetaRelevantHelper(MakeRelevantHelper):
    def __init__(self, meta_local_path, arcadia_helper, blame, dry_run=False):
        super(MakeMetaRelevantHelper, self).__init__(meta_local_path, arcadia_helper, blame, dry_run)

    def _get_commit_msg(self):
        return 'Update history information about constants in'

    def _update_data(self, meta, const, value):
        if const not in meta:
            meta[const] = {}
            meta[const]['initial'] = {
                'commit_message': value['commit_message'],
                'date': datetime.now().date().isoformat(),
                'revision': value['revision'],
            }
            logging.info('Init meta of const {}: {}'.format(const, meta[const]['initial']))
