# -*- coding: utf-8 -*-

import requests
import logging
from sandbox.common.errors import SandboxException


logger = logging.getLogger(__name__)


class StaffClient:
    cache = {}
    cache_old = {}
    url = 'https://staff-api.yandex-team.ru/v3/persons'

    def __init__(self, token):
        self.token = token
        self.headers = {'Authorization': 'OAuth %s' % self.token}
        self.params = {'_fields': 'department_group.ancestors,department_group.department,department_group.level,'
                                  'official.is_dismissed,chief,login'}

    @staticmethod
    def is_dismissed(person):
        return person['official']['is_dismissed']

    @staticmethod
    def get_json_from_response(resp):
        if resp.status_code != 200:
            raise SandboxException('Something wrong with communication with /staff: '
                                   '\n\tstatus_code = {}, reason = {}, request = {}'
                                   .format(resp.status_code, resp.reason, resp.request.url))

        return resp.json()

    @staticmethod
    def filter_yabs_department(departments):
        return filter(lambda x: x['url'] == 'yandex_monetize_banner', departments)

    def parse_response(self, resp):
        json_data = self.get_json_from_response(resp)
        assert json_data['total'] == 1, 'One and only one person should be returned by login, but returned {}:\n{}' \
            .format(json_data['total'], json_data['result'])
        return json_data['result'][0]

    def update_cache_old(self, login, is_dismissed, reassign):
        info = dict(is_dismissed=is_dismissed, reassign=reassign)
        self.cache_old[login] = info
        return info

    def make_cache_request(self, login):
        if login not in self.cache:
            self.params['login'] = login
            resp = requests.get(self.url, headers=self.headers, params=self.params)
            self.cache[login] = self.parse_response(resp)
        return self.cache[login]

    def is_suitable(self, login):
        info = self.get_person_info_old(login)
        return not info['is_dismissed'] and info['reassign'] == login

    def get_head(self, dep):
        for chief in dep['department']['heads']:
            if not self.is_dismissed(chief['person']):
                return self.get_person_info_old(chief['person']['login'])['reassign']
        return None

    def get_person_is_dismissed(self, login):
        info = self.make_cache_request(login)
        return info['official']['is_dismissed']

    def get_person_group_url(self, login):
        info = self.make_cache_request(login)
        return info['department_group']['department']['url']

    def get_person_group_name(self, login):
        info = self.make_cache_request(login)
        return info['department_group']['department']['name']['full']['en']

    def get_person_chief(self, login):
        info = self.make_cache_request(login)['chief']
        if info is None:
            return None
        return info['login']

    def get_person_info_old(self, login):
        if login in self.cache_old:
            return self.cache_old[login]

        self.params['login'] = login
        resp = requests.get(self.url, headers=self.headers, params=self.params)
        res = self.parse_response(resp)
        dep = res['department_group']
        if self.is_dismissed(res):
            # person is dismissed -> check his chief
            reassignee = self.get_head(dep)
            if reassignee:
                return self.update_cache_old(login, is_dismissed=True, reassign=reassignee)

            # no heads for department found (mb this department is not actual now)
            deps = [d for d in dep['ancestors'] if d['level'] >= 6]
            deps = sorted(deps, key=lambda d: d['level'], reverse=True)
            for dep in deps:
                reassignee = self.get_head(dep)
                if reassignee:
                    return self.update_cache_old(login, is_dismissed=True, reassign=reassignee)
            return self.update_cache_old(login, is_dismissed=True, reassign=self.default_assignee)
        elif dep['level'] < 6 or not self.filter_yabs_department(dep['ancestors']):
            # person is working in other department
            return self.update_cache_old(login, is_dismissed=False, reassign=self.default_assignee)
        return self.update_cache_old(login, is_dismissed=False, reassign=login)

    def get_persons_info_full(self, logins_set):
        logins = ','.join(logins_set)
        self.params['login'] = logins
        self.params['_limit'] = '{}'.format(len(logins_set) * 2)
        resp = requests.post(self.url, headers=self.headers, data=self.params)
        json_data = self.get_json_from_response(resp)
        return json_data['result']
