from datetime import datetime
import json
import logging
import os
import re

from sandbox.sdk2.vcs import svn
from sandbox.projects.yabs.qa.utils.general import html_hyperlink


LOCAL_ARCADIA_DIR = 'arcadia'
DATA_UPDATE_TICKET = 'BSEFFECTIVE-250'
MAX_DAYS_WITHOUT_ABC = 10
RESET_TICKET_ASSIGNEE = 'RESET TICKET ASSIGNEE'


def checkout(path, arcadia_client=svn.Arcadia, logger=logging.getLogger(__name__)):
    """
    Checkouts file in `path`, which is given from arcadia root, for example: yabs/server/proto/quality/sys_const.proto
    """
    checkout_dir = os.path.join(arcadia_client.ARCADIA_TRUNK_URL, os.path.dirname(path))
    local_dir = os.path.join(LOCAL_ARCADIA_DIR, os.path.dirname(path))
    if not os.path.exists(local_dir):
        arcadia_client.checkout(url=checkout_dir, path=local_dir)
        logger.debug("CHECKOUT {} {}".format(checkout_dir, local_dir))
    path = os.path.join(LOCAL_ARCADIA_DIR, path)
    assert os.path.exists(path)
    return path


def commit_changes(path, msg, arcadia_helper, dry_run):
    if not dry_run:
        review_url = arcadia_helper.commit(
            paths=path,
            commit_message='{ticket}: {msg} {file}, {time}'.format(
                ticket=DATA_UPDATE_TICKET,
                msg=msg,
                file=os.path.basename(path[0]),
                time=datetime.now().strftime('%Y-%m-%d %H:%M:%S')
            ),
            revprop=['arcanum:review-automerge=yes']
        )
        if review_url is None:
            return 'Could not commit changes. Maybe your files have no changes.'
        review_url = html_hyperlink(link=review_url, text=review_url)
        return review_url
    else:
        return 'No review in dry_run mode'


def tokenize_blame(blame):
    def make_regexp_pattern():
        """
        Builds regex to parse svn blame (Arcadia.blame) output
        """
        revision = r'(\d+)'
        author = r'([\w-]+)'
        date_nu = r'[\d-]+'
        time_nu = r'[\d:]+'
        not_used = r'\+\d+'
        date_text_nu = r'\([^)]+\)'
        constant = r'optional int64 (\w+) = \d+ \[([ _=,\-\(\)\da-zA-Z]*)\];'
        pattern = r'\s+'.join([revision, author, date_nu, time_nu, not_used, date_text_nu, constant])
        return pattern
    return re.findall(make_regexp_pattern(), blame)


def get_const_proto_blame(path, arcadia_client):
    """
    Get from svn blame (Arcadia.blame)
    """
    blame = arcadia_client.blame(arcadia_client.trunk_url(path), True)
    logging.info('raw_blame: \n{}'.format(blame.encode('utf-8')))
    matched = tokenize_blame(blame)
    blame_dict = {}
    for match in matched:
        options = dict(
            tuple(map(lambda y: y.strip(), x.split('=')))
            for x in match[3].split(',') if x != ''
        )
        blame_dict[match[2]] = dict(
            revision=match[0],
            author=match[1],
            options=options,
            commit_message=None
        )
    for const, value in blame_dict.items():
        value['source'] = 'ARC_' + os.path.basename(path)
        arcadia_log = arcadia_client.log(arcadia_client.trunk_url(), value['revision'], value['revision'])
        if len(arcadia_log) > 0:
            value['commit_message'] = arcadia_log[0]['msg'].encode('utf-8').replace('\n', ' ')
        else:
            value['commit_message'] = 'No commit message found for revision {}'.format(value['revision'])
    return blame_dict


def get_blame(paths, arcadia_client=svn.Arcadia):
    result = {}
    for path in paths.split(','):
        result.update(get_const_proto_blame(path, arcadia_client))
    return result


def get_json(path, logger=logging.getLogger(__name__)):
    path = checkout(path, logger=logger)
    logger.debug("Get json: {}".format(path))
    return json.load(open(path, 'r'))


def get_all_constant_ids_and_options(file_obj):
    """
    Reads all constants' names their ids and options.
    Returns a dict of lists dict[const_name] = (const_id, options), where options is a dict.
    """
    logger = logging.getLogger("get_const_id_and_options")
    pattern = r'    optional int64 ([0-9a-zA-Z]*) = (\d+) \[([ _=,\-\(\)\da-zA-Z]*)\];'
    answer = {}
    for line in file_obj:
        match = re.match(pattern, line)
        if match is not None and len(match.groups()) == 3:
            match = match.groups()
            const_name = match[0]
            const_id = int(match[1])
            options = dict(
                tuple(map(lambda y: y.strip(), x.split('=')))
                for x in match[2].split(',') if x != ''
            )
            answer[const_name] = (const_id, options)
    logger.info('{} alive constants in proto file'.format(len(answer)))
    return answer


def join_file_path_with_checkout_dir(checkout_dir_local, path):
    return os.path.join(checkout_dir_local, os.path.basename(path))
