# -*- coding: utf-8 -*-

import os

from sandbox import sdk2
from sandbox.sandboxsdk import ssh

from sandbox.projects.common.binary_task import LastBinaryTaskRelease, binary_release_parameters
from sandbox.projects.yabs.SysConstLifetime.lib.make_relevant_helper import MakeOwnersRelevantHelper, MakeMetaRelevantHelper
from sandbox.projects.yabs.SysConstLifetime.lib.arcadia_helper import ArcadiaHelper
from sandbox.projects.yabs.SysConstLifetime.lib.staff_client import StaffClient
from sandbox.projects.yabs.SysConstLifetime.lib.utils import get_blame, join_file_path_with_checkout_dir
from sandbox.projects.yabs.SysConstLifetime.lib.limits_helper import LimitsHelper

from sysconst_helper import SysConstHelper
from users_json_helper import UsersJsonHelper


class SYS_CONST_UPDATE(sdk2.Resource):
    """
    Resource with JSON update to sys const blame
    """


class SysConstLifetime(LastBinaryTaskRelease, sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        lbrp = binary_release_parameters(stable=True)

        commit_path = sdk2.parameters.String("Arcanum path", default="https://a.yandex-team.ru/arc/commit/{}", required=True)
        const_path = sdk2.parameters.String("Constants proto file", default="yabs/server/proto/quality/sys_const.proto", required=True)
        owners_json_path = sdk2.parameters.String('Path to file with constant owners from arcadia root', default='yabs/server/proto/quality/sys_const.owners.json', required=True)
        limits_json_path = sdk2.parameters.String('Path to file with constant limits from arcadia root', default='yabs/server/proto/quality/sys_const.limits.json', required=True)
        meta_json_path = sdk2.parameters.String('Path to file with meta data for each constant from arcadia root', default='yabs/server/proto/quality/sys_const.meta.json', required=True)
        users_json_path = sdk2.parameters.String('Path to file with ABC users leaders and groups', default='yabs/server/proto/quality/users.json', required=True)
        reassign = sdk2.parameters.Bool("Reassign startek ticket based on blame data (if ticket was created previously on other person)", default=False, required=True)
        dry_run = sdk2.parameters.Bool("Dry run, dont make anything in st", default=False, required=True)
        st_queue = sdk2.parameters.String("ST queue", default="BSSERVER", required=True)
        st_token = sdk2.parameters.String("ST token vault name", default="st_token", required=True)
        staff_token = sdk2.parameters.String("Staff token vault name", default="staff_token", required=True)
        default_assignee = sdk2.parameters.String("Person who would be assign as task executor if the author of constant can not be applied", default="poldnev", required=True)
        yql_token = sdk2.parameters.String("YQL token vault name", default="yql_token", required=True)
        abc_token = sdk2.parameters.String("ABC token vault name", default="abc_token", required=True)
        abc_id_list = sdk2.parameters.List("ABC IDs to get members from", default=[179, 30906], required=True)
        sandbox_tasks_owner = sdk2.parameters.String("Generated sandbox tasks owner", default="YABS_SERVER_CONSTANTS", required=True)
        sandbox_token = sdk2.parameters.String("Sandbox token vault name", default="sandbox_token", required=True)
        sandbox_token_owner = sdk2.parameters.String("Sandbox token owner", default="robot-yabs-const", required=True)
        yt_cluster = sdk2.parameters.String("YT cluster", default="hahn", required=True)
        yt_path = sdk2.parameters.String("YT path", default="//home/yabs-cs/data/RuntimeConstant", required=True)
        tag = sdk2.parameters.String("StarTrek Tag to be added to tickets", default="SysConstLifetime", required=True)
        manual_reassign_tag = sdk2.parameters.String("StarTrek tag to say that this ticket should not be reassign", default="manual_reassign", required=True)
        manual_close_tag = sdk2.parameters.String("StarTrek tag to say that this ticket should not be reopened", default="manual_close", required=True)
        wiki_url = sdk2.parameters.String("Wiki url with yabs dev rules", default="https://wiki.yandex-team.ru/bannernajakrutilka/server/o-processax-razrabotki-dvizhka/product-dev-workflow/#akkuratnajarabotaskonstantami", required=True)  # noqa

    def _create_st_client(self):
        """
        Creates and returns Startrek client using self.author (from parent sdk2.Task) and self.Parameters.st_token
        """
        from startrek_client import Startrek

        token = sdk2.Vault.data(self.author, self.Parameters.st_token)
        return Startrek(useragent='Sandbox SYS_CONST_LIFETIME task', token=token)

    def _create_staff_client(self):
        """
        Creates and returns Staff client using self.author (from parent sdk2.Task) and self.Parameters.st_token
        """
        token = sdk2.Vault.data(self.author, self.Parameters.st_token)
        return StaffClient(token=token)

    def _create_abc_client(self):
        """
        Creates and returns ABC client using self.author (from parent sdk2.Task) and self.Parameters.abc_token
        """
        from sandbox.projects.yabs.SysConstLifetime.lib.abc_client import ABCClient
        token = sdk2.Vault.data(self.author, self.Parameters.abc_token)
        return ABCClient(token=token, id_list=self.Parameters.abc_id_list)

    def _create_yql_client(self):
        """
        Creates and returns YQL client using self.author (from parent sdk2.Task) and self.Parameters.st_token
        """
        from yql.api.v1.client import YqlClient

        token = sdk2.Vault.data(self.author, self.Parameters.yql_token)
        return YqlClient(db=self.Parameters.yt_cluster, token=token)

    def _create_sandbox_client(self):
        """
        Creates and returns YQL client using self.Parameters.sandbox_token_owner and self.Parameters.sandbox_token.
        """
        from sandbox.taskbox.binary import Sandbox, LazyAuth

        sandbox_token = sdk2.Vault.data(self.Parameters.sandbox_token_owner, self.Parameters.sandbox_token)
        return Sandbox(Sandbox.BASE_URL, Sandbox.BASE_PROXY_URL, LazyAuth(sandbox_token))

    def _create_solomon_helper(self):
        """
        Creates and returns Solomon helper using self.Parameters.staff_token
        """
        from solomon_helper import SysConstSolomonHelper

        solomon_token = sdk2.Vault.data(self.Parameters.sandbox_token_owner, self.Parameters.staff_token)
        return SysConstSolomonHelper(project='yabs', cluster='const_lifetime', service='sandbox', token=solomon_token)

    def on_execute(self):
        """
        Inherited from sdk2.Task, this method is executed by Sandbox.

        0. Updates users.json if task is running on sys_const.proto file (only once per day)
        1. Retrieves blame on constants and its exceptions (_get_blame_update)
        2. Overrides some blame entries in self.blame
        3. Retrieves info on authors from blame, overrodes dismissed with their supervisor
        4. Retrieves constants database from YT
        5. Runs corresponding action for constants from blame and base
        """
        self.yql_client = self._create_yql_client()
        self.st_client = self._create_st_client()
        self.staff_client = self._create_staff_client()
        self.abc_client = self._create_abc_client()
        self.sandbox_client = self._create_sandbox_client()
        self.solomon_helper = self._create_solomon_helper()

        from sandbox.sdk2.vcs.svn import Arcadia
        self.arcadia_client = Arcadia

        arcadia_helper = ArcadiaHelper(
            arcadia_client=self.arcadia_client,
            ssh_client=ssh,
            commit_user='robot-yabs-const',
            commit_user_ssh_vault_key='ssh_key'
        )

        assert os.path.dirname(self.Parameters.const_path) == os.path.dirname(self.Parameters.meta_json_path)
        assert os.path.dirname(self.Parameters.const_path) == os.path.dirname(self.Parameters.owners_json_path)
        assert os.path.dirname(self.Parameters.const_path) == os.path.dirname(self.Parameters.users_json_path)
        assert os.path.dirname(self.Parameters.const_path) == os.path.dirname(self.Parameters.limits_json_path)

        checkout_dir_local = arcadia_helper.checkout(os.path.dirname(self.Parameters.const_path))

        get_path = lambda path: join_file_path_with_checkout_dir(checkout_dir_local, path)

        owners_local_path = get_path(self.Parameters.owners_json_path)
        users_local_path = get_path(self.Parameters.users_json_path)

        limits_helper = LimitsHelper(
            owners_local_path=owners_local_path,
            proto_local_path=get_path(self.Parameters.const_path),
            limits_local_path=get_path(self.Parameters.limits_json_path),
            staff_client=self.staff_client
        )

        usage_stats = []

        make_link = lambda link, text: '<a href="{link}">{text}</a>'.format(link=link, text=text)

        for owner in limits_helper.get_owners():
            limit_info = limits_helper.get_constant_limit_info_by_owner(owner)
            limit = limit_info.limit
            owned = limit_info.owned
            group_url = self.staff_client.get_person_group_url(owner)
            group_name = self.staff_client.get_person_group_name(owner)
            group_hyperlink = make_link('https://staff.yandex-team.ru/departments/' + group_url, group_name)
            limiting_group_url = limit_info.group_url
            limiting_group_name = limit_info.group_name
            limiting_group_hyperlink = make_link('https://staff.yandex-team.ru/departments/' + limiting_group_url, limiting_group_name)
            st_link = 'https://st.yandex-team.ru/issues/?_o=votes+DESC&' \
            '_f=type+priority+key+summary+description+status+resolution+updated+assignee+tags+parent+votes&' \
            'assignee=[%22{login}%22]&tags=%5B%22SysConstLifetime%22%5D&resolution=%5B%22empty%28%29%22%5D'.format(login=owner)
            owner_hyperlink = make_link(st_link, owner)
            usage_stats.append((owner_hyperlink, owned, limit, None if limit is None else max(0, owned - limit), group_hyperlink, limiting_group_hyperlink))

        self.set_info('Click on login to see open Startrek tickets')

        usage_stats = sorted(usage_stats, key=lambda a: a[1], reverse=True)
        message = ''
        message += '<table border = "1">\n'
        message += '<tr>\n' \
                    '<th> Login </th>\n' \
                    '<th> Number of FEATURE_FLAG constants </th>\n' \
                    '<th> Constants limit </th>\n' \
                    '<th> Limit exceedance </th>\n' \
                    '<th> Group </th>\n<th> Limiting Group </th>\n' \
                    '</tr>\n'
        for stats in usage_stats:
            message += '<tr>\n'
            for value in stats:
                message += '<td>'
                message += str(value)
                message += '</td>\n'
            message += '</tr>\n'
        message += '</table>\n'

        self.set_info(message, do_escape=False)

        users_json_helper = UsersJsonHelper(
            users_local_path=users_local_path,
            staff_client=self.staff_client,
            arcadia_helper=arcadia_helper,
            abc_client=self.abc_client,
            dry_run=self.Parameters.dry_run
        )
        review_url = users_json_helper.run_task()
        self.set_info('users.json review url: {}'.format(review_url), do_escape=False)

        blame = get_blame(self.Parameters.const_path, self.arcadia_client)

        make_owners_relevant_helper = MakeOwnersRelevantHelper(
            owners_local_path=owners_local_path,
            users_local_path=users_local_path,
            arcadia_helper=arcadia_helper,
            blame=blame,
            abc_client=self.abc_client,
            dry_run=self.Parameters.dry_run
        )

        make_meta_relevant_helper = MakeMetaRelevantHelper(
            meta_local_path=get_path(self.Parameters.meta_json_path),
            arcadia_helper=arcadia_helper,
            blame=blame,
            dry_run=self.Parameters.dry_run
        )

        owners_review_url = make_owners_relevant_helper.make_relevant()
        self.set_info('Owners review url: {}'.format(owners_review_url), do_escape=False)

        meta_review_url = make_meta_relevant_helper.make_relevant()
        self.set_info('Meta review url: {}'.format(meta_review_url), do_escape=False)

        sysconst_helper = SysConstHelper(
            yql_client=self.yql_client,
            st_client=self.st_client,
            staff_client=self.staff_client,
            arcadia_client=self.arcadia_client,
            sandbox_client=self.sandbox_client,
            solomon_helper=self.solomon_helper,
            blame=blame,
            parameters=self.Parameters
        )
        sysconst_helper.run_task()
