import json
import re
import subprocess as sp

from functools import reduce


def to_camel(name):
    return ''.join([x[0].upper() + x[1:] for x in name.split('_') if x])


def make_const_dict(pattern, lines, name_to_index={'const': 2, 'author': 1, 'revision': 0}):
    consts = [re.findall(pattern, x.decode('utf')) for x in lines]
    consts = reduce(lambda x, y: x + y, consts)
    return {to_camel(x[name_to_index['const']]): {'author': x[name_to_index['author']],
                                                  'revision': x[name_to_index['revision']]} for x in consts}


def get_blame(path, revision):
    cmd = 'ya svn blame --verbose svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/{}@{}'.format(path, revision).split()
    process = sp.Popen(cmd, stdout=sp.PIPE)
    return process.stdout


def make_regexp_pattern():
    """
    Builds regex to parse svn blame (Arcadia.blame) output
    """
    revision = r'(\d+)'
    author = r'([\w-]+)'
    date_nu = r'[\d-]+'
    time_nu = r'[\d:]+'
    not_used = r'\+\d+'
    date_text_nu = r'\([^)]+\)'
    constant = r'optional int64 (\w+) = \d+ \[([ _=,\-\(\)\da-zA-Z]*)\];'
    pattern = r'\s+'.join([revision, author, date_nu, time_nu, not_used, date_text_nu, constant])
    return pattern


def example():
    # EXAMPLE
    # 1) get blame of revision that is pre of target
    pre_dzap = get_blame('yabs/server/libs/db/consts.h', 2880873)[0].split('\n')
    pre_borman = get_blame('yabs/server/libs/db/consts.h', 2609562)[0].split('\n')
    pre_melnichuk = get_blame('yabs/server/libs/constant/consts.h', 3621660)[0].split('\n')

    # 2) parse blame output
    # pattern for getting const info
    pattern_pre_dzap = '^([0-9]+)\\s+([0-9a-zA-Z_-]+)\\s+assign_constant\\(([a-z0-9_]+),.*'
    pre_dzap_const = make_const_dict(pattern_pre_dzap, pre_dzap)
    pre_borman_const = make_const_dict(pattern_pre_dzap, pre_borman)
    # pattern for getting const info was changed, update pattern
    pattern_melnichuk = '^([0-9]+)\\s+([0-9a-zA-Z_-]+)\\s+assign_constant\\(([a-z0-9_]+),\\s*[0-9]+,.*'
    pre_melnichuk_const = make_const_dict(pattern_melnichuk, pre_melnichuk)

    # 3) merge updates to be assign into one dict
    update_const = {
        '2880873': pre_dzap_const,
        '2609562': pre_borman_const,
        '3621660': pre_melnichuk_const
    }

    # 4) load previous blame_update
    blame_update = json.loads(open('blame_update.json').read())

    # 5) update info
    while True:
        # repeat update operations till no update was made (if const was updated in r3621661->r2880874->r2609563)
        cnt = {'2880874': 0, '2609563': 0, '3621661': 0}
        for k, v in blame_update.items():
            if v['revision'] in update_const:
                print('const[{}] points to {}, {}'.format(k, v['revision'], update_const[v['revision']][k]))
                cnt[v['revision']] += 1
                v['author'] = update_const[v['revision']][k]['author']
                v['revision'] = update_const[v['revision']][k]['revision']
        print(cnt)
        if sum(cnt.values()) == 0:
            break

    # 6) save results to blame_update.json
    open('blame_update.json', 'w').write(json.dumps(blame_update, indent=2, sort_keys=True))


def update_after_refactoring():
    revision_to_update = 6597846
    correct_mkdb_rev = 6579322
    correct_sys_rev = 6597309
    mkdb_blame = get_blame('yabs/server/proto/quality/mkdb_const.proto', correct_mkdb_rev)
    sys_blame = get_blame('yabs/server/proto/quality/sys_const.proto', correct_sys_rev)

    mkdb_blame_dict = make_const_dict(make_regexp_pattern(), mkdb_blame)
    sys_blame_dict = make_const_dict(make_regexp_pattern(), sys_blame)
    cur_blame = mkdb_blame_dict.copy()
    cur_blame.update(sys_blame_dict)

    for const, meta in cur_blame.items():
        meta['target_revision'] = revision_to_update

    blame_update = json.loads(open('blame_update.json').read())

    cur_blame.update(blame_update)

    open('blame_update.json', 'w').write(json.dumps(cur_blame, indent=2, sort_keys=True))


if __name__ == '__main__':
    update_after_refactoring()
