from collections import defaultdict, namedtuple
import requests
import time


class BaseSolomonHelper(object):
    _sensors_list = None  # : list

    def __init__(self, project, cluster, service, token):
        """
        project: str
        cluster: str
        service: str
        """
        self._project = project
        self._cluster = cluster
        self._service = service
        self._token = token
        self._sensors_list = list()

    def add_sensor_value(self, sensor_name, value, ts=None, labels=None):
        """
        sensor_name: str
        value: int
        ts: int
        labels: Dict[str, str]
        """
        if ts is None:
            ts = int(time.time())
        labels_with_sensor = labels.copy() if labels is not None else dict()
        labels_with_sensor['sensor'] = sensor_name
        self._sensors_list.append({
            'labels': labels_with_sensor,
            'value': value,
            'ts': ts
        })

    def send_sensors(self):
        headers = {'Content-Type': 'application/json', 'Authorization': 'OAuth {}'.format(self._token)}
        url = 'http://solomon.yandex.net/api/v2/push'
        params = {
            'project': self._project,
            'service': self._service,
            'cluster': self._cluster
        }

        resp = requests.post(url, params=params, headers=headers, json={'sensors': self._sensors_list})
        self._sensors_list = list()
        return resp


PERIOD_LENGTH_SEC = 60 * 60 * 24  # Sandbox task is run every 24 hours


class SysConstSolomonHelper(BaseSolomonHelper):
    _const_events_count = None  # : defaultdict
    Sensor = namedtuple('Sensor', ('const_source', 'const_type', 'user', 'status'))

    def __init__(self, *args, **kwargs):
        super(SysConstSolomonHelper, self).__init__(*args, **kwargs)
        self._const_events_count = defaultdict(int)
        self._ts = self._current_period_ts()

    def _current_period_ts(self):
        ts = int(time.time())
        return ts - (ts % PERIOD_LENGTH_SEC)

    def add_const(self, const_source, const_type, user, status):
        """
        const_source: str
        const_type: str
        user: str
        status: str
        """
        self._const_events_count[self.Sensor(const_source, const_type, user, status)] += 1

    def send_sensors(self):
        for sensor, value in self._const_events_count.items():
            self.add_sensor_value(
                sensor_name='constants',
                value=value,
                ts=self._ts,
                labels=dict(sensor._asdict())
            )
        return super(SysConstSolomonHelper, self).send_sensors()
