# -*- coding: utf-8 -*-

from datetime import datetime
import json
import logging
import os
import textwrap
from collections import Counter, namedtuple

from sandbox.projects.yabs.SysConstLifetime.sys_const.utils import build_arc_search_link
from sandbox.projects.yabs.SysConstLifetime.lib.utils import MAX_DAYS_WITHOUT_ABC, RESET_TICKET_ASSIGNEE


class SandboxHelper:
    def __init__(self, sandbox_client):
        """
        :type sandbox_client: `sandbox.taskbox.binary.Sandbox`
        """
        self._sandbox_client = sandbox_client

    def create_task(self, task_type, description, owner, **kwargs):
        custom_fields = [
            {"name": key, "value": value}
            for key, value in kwargs.iteritems()
        ]
        task = self._sandbox_client.rest.task(
            type=task_type,
            description=description,
            owner=owner,
            custom_fields=custom_fields
        )
        return task

    def task_url(self, task_id):
        return self._sandbox_client.task_url(task_id)


class SysConstStartrekHelper:
    StartrekCallResult = namedtuple('StartrekCallResult', ('msg', 'issue', 'extra'))
    StartrekCallResult.__new__.func_defaults = (None,) * len(StartrekCallResult._fields)

    class EStartrekResultMsg:
        CREATED = 'issue created'
        ALREADY_CLOSED = 'issue is already closed'
        UPDATED = 'issue updated'
        ACTUAL = 'issue state is actual'

    """
    Contains methods that work with startrek — issues changes,
    issues texts for different action types and other statrek-related helpers.
    """
    def __init__(self, st_client, dry_run, st_queue, reassign, tag, manual_reassign_tag, manual_close_tag, wiki_url, commit_path):
        self._st_client = st_client
        self._dry_run = dry_run
        self._st_queue = st_queue
        self._reassign = reassign
        self._tag = tag
        self._manual_reassign_tag = manual_reassign_tag
        self._manual_close_tag = manual_close_tag
        self._wiki_url = wiki_url
        self._commit_path = commit_path

    def _from_timestamp(self, update_time):
        """
        Converts unix timestamp update_time to datetime object of the format from .strftime(...) argument
        """
        if update_time is None:
            return None
        return datetime.fromtimestamp(int(update_time)).strftime('%Y-%m-%d в %H:%M:%S')

    def _commit_link(self, revision):
        """
        Builds arcanum commit link using template from self._parameters.commit_path and returns string
        """
        return self._commit_path.format(revision)

    def _create_issue(self, queue, assignee, summary, description, tags):
        """
        Creates issue in Startrek  with given args if self._dry_run is False
        """
        if self._dry_run:
            return
        try:
            issue = self._st_client.issues.create(
                queue=queue,
                assignee=assignee,
                summary=summary,
                description=description,
                tags=tags
            )
        except Exception as e:
            logging.warn('failed create issue with %s: %s', type(e), e)
        else:
            logging.info('created issue %s', issue.key)
            return issue

    def _update_issue(self, issue, summary, description, tags, assignee):
        """
        Updates existing issue in Startrek with given args if self._dry_run is False
        """
        fields = dict(
            summary=summary,
            description=description,
            tags=tags
        )

        if assignee == RESET_TICKET_ASSIGNEE:
            fields['assignee'] = None
        elif assignee:
            fields['assignee'] = assignee

        if self._dry_run:
            return issue

        try:
            issue.update(**fields)
        except Exception as e:
            logging.warn('failed update issue %s with %s: %s', issue.key, type(e), e)
        else:
            logging.info('updated issue %s', issue.key)
            return issue

    def close_issue_if_exists(self, const):
        """
        Looks for all issues with tags {self._tag, const}
        """
        issues = self._st_client.issues.find('Queue: {queue} AND Tags: {const} AND Tags: {tag} AND Status: !closed'.format(
            queue=self._st_queue, const=const, tag=self._tag)
        )
        keys = []
        for issue in issues:
            keys.append(issue.key)
            if self._dry_run:
                continue
            issue.transitions['close'].execute(resolution='fixed')
        if len(keys) == 0:
            logging.info('No issues found to close for constant {}'.format(const))
            return
        logging.info('Closed {} issues for constant {}: {}'.format(len(keys), const, ', '.join(keys)))
        return keys

    def close_inactual_issues(self, actual_constants):
        """
        Looks for all issues with tag self._tag (which is Parameters.tag) and closes all, which constants does not intersect with actual_constants
        """
        issues = self._st_client.issues.find('Queue: {queue} AND Tags: {tag} AND Status: !closed'.format(queue=self._st_queue, tag=self._tag))
        issues = filter(lambda issue: len(set(issue.tags) & set(actual_constants)) == 0, issues)

        keys = []
        for issue in issues:
            keys.append(issue.key)
            if self._dry_run:
                continue
            try:
                issue.transitions['close'].execute(resolution='fixed')
            except Exception as e:
                logging.error('{exception}: can\'t close ticket {ticket_id} with status {status}'.format(
                    exception=e,
                    ticket_id=issue.key,
                    status=issue.status
                ))
        logging.info('Closed {} inactual issues: {}'.format(len(keys), ', '.join(keys)))
        return keys

    def _find_issues(self, queue, tags):
        issues = self._st_client.issues.find('queue: {queue} AND {tags_expr}'.format(
            queue=queue,
            tags_expr=' AND '.join(('tags:' + str(tag) for tag in tags))
        ))
        return issues

    def _call_st(self, const, const_type, assignee, summary, description):
        """
        1. Finds existing issues for given constant, takes the first one (there should be only one)
        2.1. If there is no issue
            2. Creates new and exits
        2.2. If there exists issue
            2. Logs old and new issue data
            3. Updates issue with new data
        """
        queue = self._st_queue
        reassign = self._reassign
        tags = [self._tag, const]
        logging.info('processing {} with tags {}'.format(const, set(tags) - {self._manual_reassign_tag, self._manual_close_tag}))

        # Looking for issues with constant name as tag
        issues = self._find_issues(queue, tags)
        logging.info('Total issues found {}: {}'.format(len(issues), ','.join('{}[{}]'.format(issue.key, issue.tags) for issue in issues)))

        if len(issues) == 0:
            logging.info('issue not found, create it')
            issue = self._create_issue(queue, assignee, summary, description, tags)
            return self.StartrekCallResult(self.EStartrekResultMsg.CREATED, issue)

        issue = issues[0]

        if 'closed' == issue.status.key:
            if self._manual_close_tag in issue.tags:
                logging.info('Found issue {}, that was closed manually, skip it'.format(issue.key))
                return self.StartrekCallResult(self.EStartrekResultMsg.ALREADY_CLOSED, issue)
            logging.info('Reopenning issue {} since it does not have \'{}\' tag'.format(issue.key, self._manual_close_tag))
            if not self._dry_run:
                issue.transitions['reopen'].execute(comment='Этот тикет был закрыт без тега %%manual_close%%. '
                                                            'Если он точно должен быть закрыт — нужно закрыть и добавить этот тег.')

        old_summary = issue.summary
        old_description = issue.description
        old_assignee = issue.assignee.login if issue.assignee else 'None'
        need_reassign = reassign and assignee is not None and old_assignee != assignee and self._manual_reassign_tag not in issue.tags
        if old_summary != summary or old_description != description or need_reassign:
            logging.info('issue %s found, update it', issue.key)
            logging.info('old_summary(%s): %s', type(old_summary), old_summary)
            logging.info('summary(%s): %s', type(summary), summary)
            logging.info('old_description(%s): %s', type(old_description), old_description)
            logging.info('description(%s): %s', type(description), description)
            logging.info('tags(%s): %s', ','.join(str(type(tag)) for tag in issue.tags), issue.tags)
            reassign_changes = None
            if need_reassign:
                logging.info('old_assignee(%s): %s', type(old_assignee), old_assignee)
                logging.info('assignee(%s): %s', type(assignee), assignee)
                reassign_changes = dict(
                    old_assignee=old_assignee,
                    assignee=assignee,
                    const=const,
                )
            self._update_issue(issue, summary, description, issue.tags, assignee if need_reassign else None)
            if need_reassign and not self._dry_run and issue is not None:
                issue.comments.create(text='Исполнитель этого тикета был изменён. Если не хочешь, чтобы таска '
                                           'меняла его — добавь тег %%manual_reassign%% при переназначении исполнителя')
            return self.StartrekCallResult(self.EStartrekResultMsg.UPDATED, issue, {'reassign_changes': reassign_changes})
        logging.info('issue %s is in actual state', issue.key)
        return self.StartrekCallResult(self.EStartrekResultMsg.ACTUAL, issue)

    def set_type_in_code(self, const, blame):
        """
        Creates or updates issue about given constant, asks to set a type for constant
        """
        return self._call_st(
            const=const.decode('utf-8'),
            const_type=SysConstHelper.EConstantType.get_const_type(blame),
            assignee=blame['ticket_reassign'],
            summary='Указать значение опции (type) для константы {}'.format(const).decode('utf-8'),
            description=textwrap.dedent("""
                В ревизии {} была добавлена константа {} без опции (type).

                Author: {}
                Commit message: {}

                Пожалуйста, укажи, к какому типу принадлежит данная константа:
                    - FEATURE_FLAG если эта константа включает какую-то функциональность и когда-то её надо будет удалить,
                    - CONSTANT_VALUE если это числовое значение, которое навсегда останется в коде.

                Перед коммитом удостоверься, что в %%ya make -t yabs/server/proto%% проходят все тесты — там есть тест, проверяющий, что опция %%(type)%% установлена у всех констант.

                Подробнее о работе с константами в движке и общих правилах разработки: {}
            """.format(
                self._commit_link(blame['revision']),
                const,
                blame['author'],
                blame['commit_message'],
                self._wiki_url
            )).decode('utf-8')
        )

    def delete_from_code_sys_const(self, const, blame, base):
        """
        Creates or updates issue about given sys constant, asks to delete constant from code
        """
        return self._call_st(
            const=const.decode('utf-8'),
            const_type=SysConstHelper.EConstantType.get_const_type(blame),
            assignee=blame['ticket_reassign'],
            summary='Удалить константу {} из кода'.format(const).decode('utf-8'),
            description=textwrap.dedent("""
                Значение в базе: {}, установлено {}.
                Значение в коде: {}, ревизия {}.

                Поскольку константа включена в базе, надо удалить её из кода.
                Не забудь прогнать прекоммитные проверки в ШМ перед коммитом!

                Поиск константы в аркадии: {}

                Constant type: {}
                Initial commit message: {}
                Last commit message: {}
                Description: {}

                Подробнее о работе с константами в движке и общих правилах разработки: {}
                Подробнее об удалении констант из кода: https://wiki.yandex-team.ru/bannernajakrutilka/server/o-processax-razrabotki-dvizhka/product-dev-workflow/#udaleniekonstant
            """.format(
                base['value'],
                self._from_timestamp(base['update_time']),
                blame['options']['default'],
                self._commit_link(blame['revision']),
                build_arc_search_link(const),
                blame['options']['(type)'],
                blame['initial']['commit_message'] if 'initial' in blame else None,
                blame['commit_message'],
                base['description'],
                self._wiki_url
            )).decode('utf-8')
        )

    def delete_from_code_mkdb_const(self, const, blame):
        """
        Creates or updates issue about given mkdb constant, asks to delete constant from code
        """
        return self._call_st(
            const=const.decode('utf-8'),
            const_type=SysConstHelper.EConstantType.get_const_type(blame),
            assignee=blame['ticket_reassign'],
            summary='Удалить константу {} из кода'.format(const).decode('utf-8'),
            description=textwrap.dedent("""
                Значение в коде: {}, ревизия {}.

                Поскольку константа имеет тип FEATURE_FLAG, её надо удалить из кода, если её жизненный цикл завершен. Или изменить тип на CONSTANT_VALUE.
                Не забудь прогнать прекоммитные проверки в ШМ перед коммитом!

                Поиск константы в аркадии: {}

                Constant type: {}
                Initial commit message: {}
                Last commit message: {}

                Подробнее о работе с константами в движке и общих правилах разработки: {}
                Подробнее об удалении констант из кода: https://wiki.yandex-team.ru/bannernajakrutilka/server/o-processax-razrabotki-dvizhka/product-dev-workflow/#udaleniekonstant
                Время обновления тикета: {}
            """.format(
                blame['options']['default'],
                self._commit_link(blame['revision']),
                build_arc_search_link(const),
                blame['options']['(type)'],
                blame['initial']['commit_message'] if 'initial' in blame else None,
                blame['commit_message'],
                self._wiki_url,
                datetime.now().strftime('%Y-%m-%d в %H:%M:%S')
            )).decode('utf-8')
        )

    def update_default_in_code(self, const, blame, base, update_default_task_url):
        """
        Creates or updates issue about given constant, asks to update default value, taken from prod.
        """
        return self._call_st(
            const=const.decode('utf-8'),
            const_type=SysConstHelper.EConstantType.get_const_type(blame),
            assignee=blame['ticket_reassign'],
            summary='Обновить дефолт {}'.format(const).decode('utf-8'),
            description=textwrap.dedent("""
                Значение в базе: {}, установлено {}.
                Значение в коде: {}, ревизия {}.

                Для синхронизации поведения движка в тестах и продакшене, следует обновить дефолт.
                Для этого достаточно отредактировать proto-файл, описывающий константы.
                Не забудь прогнать прекоммитные проверки и починить все упавшие тесты.

                Constant type: {}
                Initial commit message: {}
                Last commit message: {}
                Description: {}

                SB таска, которая создаст PR с обновлённым дефолтом: {}

                Подробнее о работе с константами в движке и общих правилах разработки: {}
            """.format(
                base['value'],
                self._from_timestamp(base['update_time']),
                blame['options']['default'],
                self._commit_link(blame['revision']),
                blame['options']['(type)'],
                blame['initial']['commit_message'] if 'initial' in blame else None,
                blame['commit_message'],
                base['description'],
                update_default_task_url,
                self._wiki_url
            )).decode('utf-8')
        )

    def turn_on(self, const, blame, generate_oneshot_task_url):
        """
        Creates or updates issue about given constant, asks to enable constant in prod
        """
        return self._call_st(
            const=const.decode('utf-8'),
            const_type=SysConstHelper.EConstantType.get_const_type(blame),
            assignee=blame['ticket_reassign'],
            summary='Включить константу {} в production'.format(const).decode('utf-8'),
            description=textwrap.dedent("""
                Константа, добавленная в {}, на данный момент отсутствует в таблице.
                Если константа требует включения, пора завести oneshot.
                При тестировании следует убедиться в наличии ожидаемой разницы.
                Если включение не требуется, ты можешь закрыть тикет с резолюцией Won't fix.

                Constant type: {}
                Initial commit message: {}
                Last commit message: {}

                SB таска, которая сгенерирует ваншот на включение константы: {}

                Подробнее о работе с константами в движке и общих правилах разработки: {}
            """.format(
                self._commit_link(blame['initial']['revision']),
                blame['options']['(type)'],
                blame['initial']['commit_message'] if 'initial' in blame else None,
                blame['commit_message'],
                generate_oneshot_task_url,
                self._wiki_url
            )).decode('utf-8')
        )

    def delete_from_base(self, const, base, generate_oneshot_task_url):
        """
        Creates or updates issue about given constant, asks to delete unused constant from database.
        """
        return self._call_st(
            const=const.decode('utf-8'),
            const_type=None,
            assignee=None,
            summary='Удалить константу {} из базы'.format(const).decode('utf-8'),
            description=textwrap.dedent("""
                Константа, установленная {}, не используется в коде.
                Это значит, что ее можно безопасно удалить из базы.
                На этом ее жизненный цикл будет завершен.

                Description: {}

                SB таска, которая сгенерирует ваншот на удаление константы: {}

                Подробнее о работе с константами в движке и общих правилах разработки: {}
            """.format(
                self._from_timestamp(base['update_time']),
                base['description'],
                generate_oneshot_task_url,
                self._wiki_url
            )).decode('utf-8')
        )


class SysConstHelper:
    """
    Helper for SysConstLifetime Sandbox task, separated from task to write tests
    """
    class EConstantType:
        UNKNOWN = 'UNKNOWN'
        CONSTANT_VALUE = 'CONSTANT_VALUE'
        FEATURE_FLAG = 'FEATURE_FLAG'

        @classmethod
        def get_const_type(cls, const_dict=None):
            if const_dict is None:
                return
            const_type = const_dict['options'].get('(type)')
            if const_type is None:
                return cls.UNKNOWN
            if const_type in (cls.CONSTANT_VALUE, cls.FEATURE_FLAG):
                return const_type
            return cls.UNKNOWN

    def __init__(self, yql_client, st_client, staff_client, arcadia_client, sandbox_client, solomon_helper, blame, parameters):
        self._yql_client = yql_client
        self._startrek_helper = SysConstStartrekHelper(
            st_client=st_client,
            dry_run=parameters.dry_run,
            st_queue=parameters.st_queue,
            reassign=parameters.reassign,
            tag=parameters.tag,
            manual_reassign_tag=parameters.manual_reassign_tag,
            manual_close_tag=parameters.manual_close_tag,
            wiki_url=parameters.wiki_url,
            commit_path=parameters.commit_path
        )
        self._solomon_helper = solomon_helper
        self._staff_client = staff_client
        self._arcadia_client = arcadia_client
        self._sandbox_helper = SandboxHelper(sandbox_client)
        self._blame = blame
        self._parameters = parameters

    def _camel_case(self, name):
        """
        Converts name of the format `xxx-yyy-zzz` to camel case string `XxxYyyZzz`.
        WARNING: Correct camel case should not be broken so `XxxYyyZzz` -> `XxxYyyZzz`.
        """
        return ''.join(part[0].upper() + part[1:] for part in name.split('-'))

    def _get_base(self):
        """
        Returns dict with current values of constants and their meta from YT path self._parameters.yt_path
        """

        if self._parameters.const_path.endswith('mkdb_const.proto'):
            return {}

        query = self._yql_client.query("""
            select Name, Value, UpdateTime, Description from `{}` where ContentSystemKey = 0
        """.format(self._parameters.yt_path), syntax_version=1)
        query.run()
        table = next(iter(query.get_results()))
        table.fetch_full_data()
        return dict((self._camel_case(row[0]), dict(
            value=row[1],
            update_time=row[2],
            description=(row[3] or '').encode('utf-8'),
            name_in_base=row[0],
            source='YT_' + os.path.basename(self._parameters.yt_path)
        )) for row in table.rows)

    def print_reassign_changes(self, changes_list):
        """
        Logs reassign changes table in wiki format
        """
        def row(x):
            revision_url = '((https://a.yandex-team.ru/arc/commit/{} r{}))'.format(x['revision'], x['revision'])
            return '|| {} | {} | {} | {} ||'.format(x['const'], x['old_assignee'], x['assignee'], revision_url)
        if changes_list:
            logging.info('Reassign changes table (in wiki syntax):\n#|\n' +
                         '|| Constant | From | To | Revision ||\n' +
                         '\n'.join([row(x) for x in changes_list]) +
                         '\n|#')
            logins = [x['assignee'] for x in changes_list]
            top_victims = Counter(logins).most_common()[:5]
            logging.info('**Top victims** (people for reassign):\n' +
                         '\n'.join('* {} - {}'.format(x[0], x[1]) for x in top_victims))

    def _update_blame(self, blame):
        """
        Updating blame using owners.json and meta.json
        """
        get_data_json = lambda path: json.load(open(os.path.join('arcadia', path)))

        owners = get_data_json(self._parameters.owners_json_path)['owners']
        meta = get_data_json(self._parameters.meta_json_path)
        users = get_data_json(self._parameters.users_json_path)

        for const, value in blame.items():
            owner = owners[const]
            value['owner'] = owner

            if const in meta:
                value['initial'] = meta[const]['initial']

            if owner not in users or len(users[owner]['days_without_abc']) >= MAX_DAYS_WITHOUT_ABC:
                value['ticket_reassign'] = RESET_TICKET_ASSIGNEE
                logging.debug('Reassign ticket from  {} to None'.format(owner))
            else:
                value['ticket_reassign'] = owner

    def run_task(self):
        blame = self._blame
        self._update_blame(blame)
        base = self._get_base()

        # There are constants of three types — CONSTANT_VALUE, FEATURE_FLAG and None, and two constant resources - mkdb_const and sys_const
        # 1. If type is None then we have to ask to set type explicitly in ticket
        # 2.1. If type is FEATURE_FLAG and resouce is sys_const then
        #     - if constant is in code, but not in base, we ask to turn it on in base
        #     - if constant is in base and in code and value in base != 0, then we ask to delete it from code
        # 2.2. If type is FEATURE_FLAG and resouce is mkdb_const then
        #     - if constant is in code then we ask to delete it from code
        #     - additionally CsSettingsLifetime bot will ask update default and drop setting, if it differs from default
        # 3.1 If type is CONSTANT_VALUE and resource is sys_const then
        #     - if constant is in code, but not in base then we do nothing
        #     - if constant is in code and in base and values are different then we ask to update it in code
        #     - if constant is in code and in base and values are equal then we close ticket or do nothing (if there is no ticket)
        # 3.2 If type is CONSTANT_VALUE and resource is mkdb_const then
        #     - if constant is in code, we do nothing
        #     - additionally CsSettingsLifetime bot will ask update default and drop setting, if it differs from default
        # 4. If constant is not in blame (so it does not have type), then it is in base and we ask to delete it from base, should never happen in mkdb_const case
        # 5. ADDITIONALY: Look for all issues with our tag and if it was not related to any constant during this run — close them all.

        reassign_changes = []
        actual_constants = set(blame.keys() + base.keys())

        def add_const_to_stat(const_blame, status, const_type, st_res=None):
            if st_res == SysConstStartrekHelper.EStartrekResultMsg.ALREADY_CLOSED:
                return
            self._solomon_helper.add_const(
                const_source=const_blame.get('source', 'unknown'),
                const_type=const_type,
                user=const_blame.get('owner', 'unknown'),
                status=status
            )

        for const in set(blame.keys() + base.keys()):
            if const in blame and blame[const]['owner'] == "HAS NO OWNER":
                logging.debug("Const {} skipped, HAS NO OWNER found".format(const))
                continue

            const_type = self.EConstantType.get_const_type(blame.get(const))
            const_default = int(blame[const]['options'].get('default', 0)) if const in blame else None
            const_value = int(base[const]['value']) if const in base else None
            st_res = None

            def ask_update_default():
                if self._parameters.dry_run:
                    return
                task = self._sandbox_helper.create_task(
                    task_type='CONST_PROTO_UPDATE',
                    description='Update default of constant {const} to {value}'.format(const=const, value=const_value),
                    owner=self._parameters.sandbox_tasks_owner,
                    proto_file_path=self._parameters.const_path,
                    make_task_author_owner=False,
                    const_name=const,
                    const_value=const_value,
                    const_type=const_type
                )
                res = self._startrek_helper.update_default_in_code(const, blame[const], base[const], self._sandbox_helper.task_url(task['id']))
                return res

            if const_type is self.EConstantType.UNKNOWN:
                st_res = self._startrek_helper.set_type_in_code(const, blame[const])
                add_const_to_stat(blame[const], 'set_type_in_code', const_type, st_res)
            elif const_type == self.EConstantType.FEATURE_FLAG:
                if self._parameters.const_path.endswith('mkdb_const.proto'):
                    st_res = self._startrek_helper.delete_from_code_mkdb_const(const, blame[const])
                    add_const_to_stat(blame[const], 'delete_from_code', const_type, st_res)
                else:
                    if const not in base:
                        task = self._sandbox_helper.create_task(
                            task_type='GENERIC_CONSTANT_YT_ONESHOT',
                            description='Turn on {const}'.format(const=const),
                            owner=self._parameters.sandbox_tasks_owner,
                            oneshot_assignee=blame[const]['owner'],
                            action_type='add',
                            constant_name=const,
                            constant_value=1,
                            constant_table=self._parameters.yt_path
                        )
                        st_res = self._startrek_helper.turn_on(const, blame[const], self._sandbox_helper.task_url(task['id']))
                        add_const_to_stat(blame[const], 'turn_on', const_type, st_res)
                    else:
                        if const_default != const_value:
                            st_res = ask_update_default()
                            add_const_to_stat(blame[const], 'update_default_in_code', const_type, st_res)
                        else:
                            st_res = self._startrek_helper.delete_from_code_sys_const(const, blame[const], base[const])
                            add_const_to_stat(blame[const], 'delete_from_code', const_type, st_res)
            elif const_type == self.EConstantType.CONSTANT_VALUE:
                if self._parameters.const_path.endswith('mkdb_const.proto'):
                    actual_constants.remove(const)
                    add_const_to_stat(blame[const], 'mkdb_constant_value', const_type)
                else:
                    if const in base:
                        if const_default != const_value:
                            st_res = ask_update_default()
                            add_const_to_stat(blame[const], 'update_default_in_code', const_type, st_res)
                        else:
                            self._startrek_helper.close_issue_if_exists(const)
                            add_const_to_stat(blame[const], 'close_issue_if_exists', const_type)
                    else:
                        actual_constants.remove(const)
                        add_const_to_stat(blame[const], 'constant_value_not_in_base', const_type)
            elif const_type is None:
                if self._parameters.const_path.endswith('mkdb_const.proto'):
                    logging.error('Unexpectedly got mkdb constant %s from table', const, const_type)
                else:
                    task = self._sandbox_helper.create_task(
                        task_type='GENERIC_CONSTANT_YT_ONESHOT',
                        description='Turn on {const}'.format(const=const),
                        owner=self._parameters.sandbox_tasks_owner,
                        action_type='delete',
                        constant_names=const,
                        constant_table=self._parameters.yt_path
                    )
                    st_res = self._startrek_helper.delete_from_base(const, base[const], self._sandbox_helper.task_url(task['id']))
                    add_const_to_stat(base[const], 'delete_from_base', 'unknown', st_res)
            else:
                logging.error('Got unexpected type for constant %s, type=\'%s\'', const, const_type)
            if st_res is not None and st_res.extra is not None:
                reassign_change_dict = st_res.extra.get('reassign_changes')
                if reassign_change_dict is not None:
                    reassign_change_dict['revision'] = blame[const]['revision'] if const in blame else ''
                    reassign_changes.append(reassign_change_dict)
        solomon_resp = self._solomon_helper.send_sensors()
        logging.debug('Solomon request:\n%s\n%s\n%s', solomon_resp.request.url, solomon_resp.request.headers, solomon_resp.request.body)
        logging.debug('Solomon response json: %s', solomon_resp.json())
        self._startrek_helper.close_inactual_issues(actual_constants)
        self.print_reassign_changes(reassign_changes)
