# -*- coding: utf-8 -*-

from collections import defaultdict
from datetime import datetime
import json
import logging

from sandbox.projects.yabs.SysConstLifetime.lib.filepipe import FilePipe
from sandbox.projects.yabs.SysConstLifetime.lib.utils import commit_changes, MAX_DAYS_WITHOUT_ABC


class UsersJsonHelper(object):
    def __init__(self, users_local_path, staff_client, arcadia_helper, abc_client, dry_run):
        self._users_local_path = users_local_path
        self._staff_client = staff_client
        self._arcadia_helper = arcadia_helper
        self._abc_client = abc_client
        self._dry_run = dry_run

    def run_task(self):
        abc_users_set = self._abc_client.get_abc_members()
        staff_users_dict = self._get_staff_users_by_abc(abc_users_set)

        with FilePipe(self._users_local_path) as users_file:
            users = json.load(users_file.in_io)

            self.update_users(users, staff_users_dict)

            users_file.out_io.write(json.dumps(users, indent=2, sort_keys=True, separators=(',', ': ')))
            users_file.out_io.write('\n')
            users_file.in_io.seek(0)

        return commit_changes(
            path=[self._users_local_path],
            msg='Update users information in',
            arcadia_helper=self._arcadia_helper,
            dry_run=self._dry_run
        )

    def update_users(self, users, staff_users_dict):
        for login in staff_users_dict:
            staff_info = staff_users_dict[login]

            if login not in users:
                users[login] = self._create_users_info(staff_info)
                continue

            info = users[login]
            if info['leaders'][-1] != staff_info['leader']:
                info['leaders'].append(staff_info['leader'])

            last_group = info['groups'][-1]
            group_from_staff = staff_info['group']
            if last_group['id'] != group_from_staff['id']:
                info['groups'].append(group_from_staff)
            else:
                last_group['url'] = group_from_staff['url']
                last_group['name'] = group_from_staff['name']

            info['days_without_abc'] = []

        date_format = '%Y-%m-%d'
        current_date = datetime.now()
        current_date_str = current_date.strftime(date_format)

        for login in users:
            if login in staff_users_dict:
                continue

            days_without_abc = users[login]['days_without_abc']

            if len(days_without_abc) == 0:
                days_without_abc.append(current_date_str)
                continue

            last_date = datetime.strptime(days_without_abc[-1], date_format)
            diff = current_date - last_date

            if diff.days < 1:
                continue

            if len(days_without_abc) >= MAX_DAYS_WITHOUT_ABC:
                users[login]['days_without_abc'] = days_without_abc[1:]

            users[login]['days_without_abc'].append(current_date_str)

        logging.info('total users count by update: %s', len(users))
        logging.debug(json.dumps(users, indent=2))

    def _get_staff_users_by_abc(self, abc_logins):
        staff_users = self._staff_client.get_persons_info_full(abc_logins)

        self._check_staff_users_match_abc(abc_logins, staff_users)

        result = {}
        for user in staff_users:
            info = user['login']
            group = user['department_group']['department']
            result[info] = {
                'leader': user['chief']['login'] if user['chief'] else None,
                'group': {
                    'id': group['id'],
                    'url': group['url'],
                    'name': group['name']['full']['ru']
                }
            }

        return result

    def _check_staff_users_match_abc(self, abc_logins, staff_users):
        def raise_no_match_exception(msg):
            raise Exception('Staff does not match with ABC: {}'.format(msg))

        staff_counter = defaultdict(int)
        for user in staff_users:
            staff_counter[user['login']] += 1

        for login in abc_logins:
            if login not in staff_counter:
                raise_no_match_exception('{} not in staff'.format(login))

            if staff_counter[login] != 1:
                raise_no_match_exception('count of {} is {}'.format(login, staff_counter[login]))

        for login in staff_counter:
            if login not in abc_logins:
                raise_no_match_exception('{} not in abc_logins'.format(login))

    def _create_users_info(self, staff_info):
        return {
            'leaders': [staff_info['leader']],
            'groups': [staff_info['group']],
            'days_without_abc': []
        }
