import logging
import Queue
import tempfile
import threading
from collections import defaultdict

from sandbox import sdk2
from sandbox.projects.common.binary_task import (LastBinaryTaskRelease,
                                                 binary_release_parameters)

logger = logging.getLogger('YabsBuildFlameGraph')
logging.basicConfig(level=logging.INFO)

YT_TOKEN_VAULT_NAME = 'yabs-flame-graph-yt-token'
NANNY_TOKEN_VAULT_NAME = 'yabs-flame-graph-nanny-token'
DEFAULT_FLAME_GRAPH_TTL = 200
DEFAULT_YT_CLUSTER = "arnold"


def get_gencfg_group_from_nanny_service(nanny_service, token):
    try:
        import requests
        from retry import retry

        @retry(exceptions=requests.RequestException, tries=5, delay=1, backoff=1.6)
        def get_nanny_service_attrs(service, token):
            url = 'http://nanny.yandex-team.ru/v2/services/{}/active/runtime_attrs'.format(service)
            r = requests.get(url, headers={'Authorization': 'OAuth {}'.format(token)})

            if r.status_code >= 500:
                r.raise_for_status()

            return r.json()

        try:
            nanny_service_attrs = get_nanny_service_attrs(nanny_service, token)

            if 'error' in nanny_service_attrs:
                logger.warning(nanny_service_attrs)
                return None
        except requests.RequestException as e:
            logger.warning(e)
            return None

        gencfg_group = nanny_service_attrs['content']['instances']['extended_gencfg_groups']['groups'][0]['name']
        gencfg_group = gencfg_group.replace('YABS_FRONTEND_SERVER', 'YFS')

        return gencfg_group
    except Exception:
        return nanny_service


class YabsFlameGraph(sdk2.Resource):
    pass


class ExcThreadWithException(threading.Thread):
    def __init__(self, target):
        super(ExcThreadWithException, self).__init__()
        self.queue = Queue.Queue()
        self.target = target

    def run(self):
        try:
            self.queue.put(self.target())
        except Exception as e:
            logger.info(e, exc_info=True)

    def return_value(self):
        self.join()
        try:
            return self.queue.get(block=False)
        except Queue.Empty:
            return None


class YabsBuildFlameGraph(LastBinaryTaskRelease, sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        disk_space = 2048

    class Parameters(sdk2.Parameters):
        description = 'Build flame graph'
        max_restarts = 3
        kill_timeout = 5 * 60 * 60

        binary_archive_parameters = binary_release_parameters(stable=True)
        with sdk2.parameters.Group('Flame graph parameters') as flame_graph_block:
            mode = sdk2.parameters.RadioGroup(
                'Work mode',
                choices=(
                    ('Base', 'base'),
                    ('Diff', 'diff'),
                ),
                sub_fields={
                    'base': [
                        'base_parameters',
                    ],
                    'diff': [
                        'base_parameters',
                        'diff_parameters',
                    ],
                },
                default='base',
                required=True,
            )
            base_parameters = sdk2.parameters.JSON('Base flame graph parameters')
            diff_parameters = sdk2.parameters.JSON('Diff flame graph parameters')

        with sdk2.parameters.Group('Vault parameters') as vault_block:
            yt_token_vault_name = sdk2.parameters.String('YT token vault name', default=YT_TOKEN_VAULT_NAME, required=True)
            nanny_token_vault_name = sdk2.parameters.String('Nanny token vault name', default=NANNY_TOKEN_VAULT_NAME, required=True)

        with sdk2.parameters.Group('YT parameters') as yt_block:
            yt_cluster = sdk2.parameters.RadioGroup(
                'YT cluster to run operation',
                choices=(
                    ('hahn', 'hahn'),
                    ('arnold', 'arnold'),
                ),
                default=DEFAULT_YT_CLUSTER,
                required=True,
            )

        with sdk2.parameters.Group('Other parameters') as other_block:
            flame_graph_ttl = sdk2.parameters.Integer('Flame graph TTL', default=DEFAULT_FLAME_GRAPH_TTL, required=True)

        with sdk2.parameters.Output():
            report = sdk2.parameters.Resource('Flame graphs resource', resource_type=YabsFlameGraph)

    def _build_report(self, params, cluster, yt_token, nanny_token):
        from yabs.poormansprofiler.fetch.lib import build_report, ExperimentConfigs
        out_file = tempfile.NamedTemporaryFile(delete=False).name

        safe_params = defaultdict(lambda: None)
        safe_params.update(params)

        gencfg_groups = safe_params['gencfg_groups'] or []
        nanny_services = safe_params['nanny_services']
        if nanny_services:
            gencfg_groups.extend([get_gencfg_group_from_nanny_service(nanny_service, nanny_token) for nanny_service in nanny_services])
        gencfg_groups = filter(lambda x: x is not None, gencfg_groups)

        exp_cfg = ExperimentConfigs(experiment_id=safe_params['experiment_id'], test_id=safe_params['test_id'], page_id=safe_params['page_id'], bigb_experiment=safe_params['bigb_experiment'])
        with open(out_file, "w") as f:
            build_report(
                output=f,
                start_time=safe_params['start_time'],
                end_time=safe_params['end_time'],
                program=safe_params['program'],
                role=safe_params['role'],
                gencfg_groups=gencfg_groups,
                experiment_configs=exp_cfg,
                yt_token=yt_token,
                cluster=cluster,
                pmatch_fails_bits_count_min=safe_params['pmatch_fails_bits_count_min'],
            )

        return out_file

    def on_execute(self):
        from yabs.poormansprofiler.flames.lib import make_flame_graph

        yt_token = str(sdk2.Vault.data(self.Parameters.yt_token_vault_name))
        nanny_token = str(sdk2.Vault.data(self.Parameters.nanny_token_vault_name))

        logger.info('Building reports...')

        base_thread = ExcThreadWithException(lambda: self._build_report(self.Parameters.base_parameters, self.Parameters.yt_cluster, yt_token, nanny_token))
        base_thread.start()

        diff_report = self._build_report(self.Parameters.diff_parameters, self.Parameters.yt_cluster, yt_token, nanny_token) if self.Parameters.mode == 'diff' else None

        base_report = base_thread.return_value()

        flame_graph_resource = YabsFlameGraph(
            self,
            'Flame graph for task {}'.format(self.id),
            'flame_graph.html',
            ttl=self.Parameters.flame_graph_ttl
        )
        resource_data = sdk2.ResourceData(flame_graph_resource)

        logger.info('Making flame graph...')
        make_flame_graph(base_report, diff_report, str(resource_data.path))
        resource_data.ready()
        self.Parameters.report = flame_graph_resource

        self.set_info('<a href="{}">Link to flamegraph</a>'.format(flame_graph_resource.http_proxy), do_escape=False)
