# coding: utf-8

import logging
import time
from copy import deepcopy

import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.common.types.misc import NotExists
from sandbox.projects.common.utils2 import resource_redirect_link
from sandbox.projects.common.binary_task import (LastBinaryTaskRelease,
                                                 binary_release_parameters)
from sandbox.projects.yabs.YabsBuildFlameGraph import (YabsBuildFlameGraph,
                                                       YabsFlameGraph)

logger = logging.getLogger('YabsBuildFlameGraphReport')
logging.basicConfig(level=logging.INFO)

MIRROR_GROUPS = [
    "mirror_asan_yabs_frontend_server_bs_sas1_meta",
    "mirror_asan_yabs_frontend_server_bs_sas1_stat",
    "mirror_yabs_frontend_server_bs_sas1_meta",
    "mirror_yabs_frontend_server_bs_sas1_stat",
    "mirror_yabs_frontend_server_yabs_sas1_meta",
    "mirror_yabs_frontend_server_yabs_sas1_stat"
]

METADSP_GROUPS = [
    "prestable_yabs_frontend_server_metadsp_vla_meta",
    "prestable_yabs_frontend_server_metadsp_vla_stat",
    "stable_yabs_frontend_server_metadsp_vla_meta",
    "stable_yabs_frontend_server_metadsp_vla_stat",
    "production_yabs_frontend_server_metadsp_sas2_meta",
    "production_yabs_frontend_server_metadsp_sas2_stat",
    "production_yabs_frontend_server_metadsp_man1_meta",
    "production_yabs_frontend_server_metadsp_man1_stat",
    "production_yabs_frontend_server_metadsp_vla1_meta",
    "production_yabs_frontend_server_metadsp_vla1_stat",
    "production_yabs_frontend_server_metadsp_man2_meta",
    "production_yabs_frontend_server_metadsp_man2_stat",
    "production_yabs_frontend_server_metadsp_sas1_stat",
    "production_yabs_frontend_server_metadsp_sas1_meta",
    "production_yabs_frontend_server_metadsp_vla2_stat",
    "production_yabs_frontend_server_metadsp_vla2_meta",
    "production_yabs_frontend_server_metadsp_man1_exchange"
]

ROLES = ('metarank', 'metasearch', 'metapartner')
HR_ROLES = {
    'metarank': 'Rank',
    'metasearch': 'Search',
    'metapartner': 'Partner'
}
PROGRAMS = ('meta', 'stat')
MODES = ('base', 'diff')

SMALL_DURATION = 3 * 60 * 60
BIG_DURATION = 24 * 60 * 60

MODE_DURATIONS = (('base', SMALL_DURATION), ('base', BIG_DURATION), ('diff', SMALL_DURATION))

YT_TOKEN_VAULT_NAME = 'yabs-flame-graph-yt-token'
DEFAULT_YT_CLUSTER = "hahn"

PMATCH_FAILS_BITS = 1


def is_need_bad_pmatch(role, program):
    return role in ('metapartner', 'metasearch') and program == 'stat'


def generate_config(mode, is_mirror, role, program, gencfg_groups, start_time, end_time, pmatch_fails_bits=None):
    config = {
        'mode': mode,
        'base_config': {
            'start_time': start_time,
            'end_time': end_time,
        },
    }

    if role is not None:
        config['base_config']['role'] = role
    if program is not None:
        config['base_config']['program'] = program
    if pmatch_fails_bits is not None and mode != 'diff':
        config['base_config']['pmatch_fails_bits_count_min'] = pmatch_fails_bits

    if is_mirror:
        config['base_config']['gencfg_groups'] = MIRROR_GROUPS
    elif gencfg_groups is not None:
        config['base_config']['gencfg_groups'] = gencfg_groups

    if mode == 'diff':
        config['diff_config'] = deepcopy(config['base_config'])
        if pmatch_fails_bits is not None:
            config['diff_config']['pmatch_fails_bits_count_min'] = pmatch_fails_bits
        else:
            config['diff_config']['end_time'] = start_time
            config['diff_config']['start_time'] = start_time - (end_time - start_time)
    else:
        config['diff_config'] = None

    return config


def list_configs(end_ts):
    configs = {}

    def add_config(name, mode, duration, program=None, role=None, pmatch_fails_bits=None, is_mirror=False, gencfg_groups=None):
        start_ts = max(0, end_ts - duration)
        configs[(name, program, mode, duration, pmatch_fails_bits, is_mirror)] = generate_config(
            mode=mode,
            is_mirror=is_mirror,
            role=role,
            program=program,
            gencfg_groups=gencfg_groups,
            start_time=start_ts,
            end_time=end_ts,
            pmatch_fails_bits=pmatch_fails_bits,
        )

    for role in ROLES:
        for program in PROGRAMS:
            for mode, duration in MODE_DURATIONS:
                add_config(name=HR_ROLES[role], role=role, program=program, mode=mode, duration=duration)
                if is_need_bad_pmatch(role=role, program=program):
                    add_config(name=HR_ROLES[role], role=role, program=program, mode=mode, duration=duration, pmatch_fails_bits=PMATCH_FAILS_BITS)

                # No data for mirror metarank
                if role == 'metarank':
                    continue
                add_config(name=HR_ROLES[role], role=role, program=program, mode=mode, duration=duration, is_mirror=True)

    for program in PROGRAMS:
        for mode, duration in MODE_DURATIONS:
            add_config(name='SSP', program=program, mode=mode, duration=duration, role='metapartner')
            if is_need_bad_pmatch(role='metapartner', program=program):
                add_config(name='SSP', program=program, mode=mode, duration=duration, role='metapartner', pmatch_fails_bits=PMATCH_FAILS_BITS)

    for mode, duration in MODE_DURATIONS:
        add_config(name='Total', mode=mode, duration=duration)

    return configs


class YabsFlameGraphReport(sdk2.Resource):
    ttl = 200


class YabsBuildFlameGraphReport(LastBinaryTaskRelease, sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        disk_space = 2048

    class Parameters(sdk2.Parameters):
        description = 'Build flame graph report ending at specific timestamp'
        max_restarts = 3
        kill_timeout = 3 * 60 * 60

        binary_archive_parameters = binary_release_parameters(stable=True)

        with sdk2.parameters.Group('Report parameters') as report_block:
            end_timestamp = sdk2.parameters.Integer('End timestamp', default=None)

        with sdk2.parameters.Group('YT parameters') as report_block:
            yt_token_vault_name = sdk2.parameters.String('YT token vault name', default=YT_TOKEN_VAULT_NAME, required=True)
            yt_cluster = sdk2.parameters.RadioGroup(
                'YT cluster to run operation',
                choices=(
                    ('hahn', 'hahn'),
                    ('arnold', 'arnold'),
                ),
                default=DEFAULT_YT_CLUSTER,
                required=True,
            )

    def on_execute(self):
        from sandbox.projects.yabs.YabsBuildFlameGraphReport.html_report import get_html_report

        with self.memoize_stage.first_run():
            end_time = self.Parameters.end_timestamp or int(time.time())
            end_time = end_time - end_time % (2 * 60 * 60)  # round to last whole 2 hours
            self.Context.end_time = end_time

            logger.info('Obtaining configs..')
            configs = list_configs(self.Context.end_time)

        if self.Context.configs in (NotExists, None):
            self.Context.configs = {}

            def get_key(k):
                return ";".join(map(str, k))

            for k, c in configs.iteritems():
                self.Context.configs[get_key(k)] = c

            logger.info('Building flame graphs..')
            flame_graph_subtasks = []
            for (role, program, mode, duration, pmatch_fails, is_mirror), config in configs.iteritems():
                description = "{role} {program} {mode} flamegraph{pmatch_postfix}".format(
                    role=role,
                    program=program,
                    mode=mode,
                    pmatch_postfix=' for PmatchFailsBitsCount >= {}'.format(pmatch_fails) if pmatch_fails else ''
                )
                if is_mirror:
                    description += " (mirror)"
                build_one_flame_graph = YabsBuildFlameGraph(
                    self,
                    description=description,
                    mode=config['mode'],
                    base_parameters=config['base_config'],
                    diff_parameters=config['diff_config'],
                    yt_cluster=self.Parameters.yt_cluster,
                    yt_token_vault_name=self.Parameters.yt_token_vault_name,
                ).enqueue()
                flame_graph_subtasks.append(build_one_flame_graph)

                self.Context.configs[get_key((role, program, mode, duration, pmatch_fails, is_mirror))]['task_id'] = build_one_flame_graph.id

            raise sdk2.WaitTask(
                flame_graph_subtasks,
                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                wait_all=True
            )

        logger.info('Obtaining flame graph resources..')
        for k, config in self.Context.configs.iteritems():
            subtask_id = config['task_id']
            subtask = sdk2.Task[subtask_id]
            if subtask.status not in ctt.Status.Group.SUCCEED:
                logger.info('Subtask %s failed, omitting flame graph', subtask.Parameters.description)
                continue

            flame_graph_resource = sdk2.Resource.find(
                YabsFlameGraph,
                task=subtask
            ).first()
            if not flame_graph_resource:
                raise sdk2.common.errors.TaskFailure(
                    'YabsFlameGraph resource for subtask %s not found' % subtask.Parameters.description
                )
            config['url'] = flame_graph_resource.http_proxy

        flame_graph_report_resource = YabsFlameGraphReport(self, 'Flame graph report', 'flame_graph_report.html')
        resource_data = sdk2.ResourceData(flame_graph_report_resource)

        with open(str(resource_data.path), 'w') as f:
            f.write(get_html_report(end_time=self.Context.end_time, configs=self.Context.configs))
        resource_data.ready()

        self.set_info(
            resource_redirect_link(
                resource_id=flame_graph_report_resource.id,
                title='Ссылка на репорт',
            ),
            do_escape=False
        )
