from collections import OrderedDict
from sandbox import sdk2
from sandbox.projects.common.binary_task import LastBinaryTaskRelease, binary_release_parameters

from yabs.utils.yabs_duty_scheduler.lib.config import PERSONS_DUTY_PAGE
from yabs.utils.yabs_duty_scheduler.lib.fetchers import PersonsPageGrid, WikiPage


SCORES_WIKI_PAGE = 'https://wiki-api.yandex-team.ru/_api/frontend/BannernajaKrutilka/duty/duty-scores'


class YabsDutyScores(LastBinaryTaskRelease, sdk2.Task):

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        bert = binary_release_parameters(stable=True)
        vault_token_path = sdk2.parameters.String('Vault key token name')

    def on_execute(self):
        token = sdk2.Vault.data('YABS-YTSTAT', self.Parameters.vault_token_path)

        ppg = PersonsPageGrid(url=PERSONS_DUTY_PAGE, token=token)
        scores_page = WikiPage(url=SCORES_WIKI_PAGE, token=token)

        persons, tours = ppg.get_persons_and_tours()
        scores = self._calculate_scores(persons, tours)

        scores_page.change_content(scores)

    def _calculate_scores(self, persons, tours):
        scores = {}
        for p in persons:
            pg = {
                t.group_name: '{0:.2f}'.format(persons[p].get_inc_score_by_tour(t))
                for t in persons[p].tours_assignee
            }
            scores[persons[p]] = pg

        scores = OrderedDict(
            sorted(scores.items(), key=lambda kv: kv[0].fullname)
        )

        groups = sorted({tours[t].group_name for t in tours})

        header_list = [' ']
        for g in groups:
            header_list.append(g)

        header = '|'.join(header_list)

        result = list()
        result.append('#|')

        for i, p in enumerate(scores):
            if i % 10 == 0:
                result.append('||{}||'.format(header))

            ps_list = ['staff:{}'.format(p.nickname)]
            pg = scores[p]
            for g in groups:
                value = pg.get(g, '0')
                ps_list.append(value)

            result.append('||{}||'.format('|'.join(ps_list)))

        result.append('|#')

        return ''.join(result)
