import logging
import datetime
from enum import Enum
import requests
import urllib.request
import uuid
from sandbox import sdk2
from sandbox.projects.common import requests_wrapper
from sandbox.projects.common.testenv_client import TEClient
from sandbox.projects.release_machine.helpers.staff_helper import StaffApi

RESOURCE_TYPE='JUGGLER_PREVIOUS_SENT_UNRESOLVED_PROBLEM_IDS'


def get_unresolved_problems(testenv_project):
    unresolved_problems = TEClient.get_te_problems(testenv_project, unresolved_only=True)['rows']
    return unresolved_problems


def get_task_type_description(task_id):
    url = f'https://sandbox.yandex-team.ru/api/v1.0/task/{task_id}'
    response = requests_wrapper.get(url, verify=False)
    try:
        requests_wrapper.check_status_code(response)
    except Exception as e:
        logging.info(f'task does not exist: {e}')
        return None
    ans = response.json()
    return ans['type']


class JugglerPreviousSentUnresolvedProblemIds(sdk2.Resource):
    """ Text data """


class RecipientType(Enum):
    person = 0
    ShMGroup = 1
    BotGroup = 2


class YabsPingUnresolvedProblems(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        tokens = sdk2.parameters.YavSecret(
            'access token',
            default='sec-01fgc78rpcr5nebctrm8vn4wwf'
        )
        telegram_notice_group = sdk2.parameters.String(
            'Telegram group for duplicating sent messages',
            default='Yabs_ping_unresolved_problems'
        )
        shm_group = sdk2.parameters.String(
            'ShM group',
            default='yabs_server_testing_chat'
        )
        testenv_project = sdk2.parameters.String(
            'testenv project name',
            default='yabs-2.0'
        )
        first_working_hour = sdk2.parameters.Integer(
            'First working hour of working day',
            default=12
        )
        last_working_hour = sdk2.parameters.Integer(
            'Last working hour of working day',
            default=18
        )
        first_working_day = sdk2.parameters.Integer(
            'first working day',
            default=1
        )
        last_working_day = sdk2.parameters.Integer(
            'last working day',
            default=5
        )
        max_smart_len = sdk2.parameters.Integer(
            'max number of characters in smart report',
            default=300
        )
        mx_line_num = sdk2.parameters.Integer(
            'maximum number of lines in smart report',
            default=15
        )

    def create_shm_message(self, markdown, staff_login):
        markdown.insert(0,
                        f'{self.get_telegram_login(staff_login)} please start dialog with @YaJnsBot to receive diff personally')
        return markdown

    def create_notice_group_message(self, markdown, staff_login):
        markdown.insert(0, f'*staff login: * {staff_login}({self.get_telegram_login(staff_login)})')
        return markdown

    def is_robot(self, staff_login):
        tokens = self.Parameters.tokens.data()
        staff_api = StaffApi(tokens['juggler_token'])
        return staff_api.check_person_is_robot(staff_login)

    def get_telegram_login(self, staff_login):
        tokens = self.Parameters.tokens.data()
        staff_api = StaffApi(tokens['juggler_token'])
        groups = staff_api.get_persons('persons', {
            'login': staff_login,
            '_one': 1,
            '_fields': 'person',
        })
        for account in groups['accounts']:
            if account['type'] == 'telegram':
                return '@' + account['value']
        if self.is_robot(staff_login):
            return 'is robot'
        else:
            return 'telegram is not specified'

    def send_message(self, markdown, staff_login, recipient_type):
        tokens = self.Parameters.tokens.data()
        data = {
            'project': 'YabsServerQA',
            'template': 'default',
            'request_id': str(uuid.uuid4()),
            'abc_service': 'yabs_server_sandbox_tests',
            'params': {
                'message': {
                    'string_value': '\n'.join(markdown),
                },
            },
            'recipient': {

            }
        }
        if recipient_type == RecipientType.person:
            logging.info(f'send to staff_login: {staff_login}', )
            data['recipient']['telegram'] = {
                'internal': [{
                    'login': staff_login
                }]
            }
        elif recipient_type == RecipientType.BotGroup:
            logging.info(f'send message to chat_name: {self.Parameters.telegram_notice_group}')
            data['recipient']['telegram'] = {
                'chat_name': [self.Parameters.telegram_notice_group],
            }
        else:
            logging.info(f'send message to chat_name: {self.Parameters.shm_group}')
            data['recipient']['telegram'] = {
                'chat_name': [self.Parameters.shm_group],
            }
        response = requests.post(
            'https://jns.yandex-team.ru/api/messages/send',
            json=data,
            headers={'Authorization': 'OAuth ' + tokens['juggler_token']}
        )
        if not response.ok and recipient_type == RecipientType.person:
            logging.info(f'status: {response.status_code}\n text: {response.text}')
            self.send_message(self.create_shm_message(markdown, staff_login), staff_login, RecipientType.ShMGroup)
        elif response.ok and recipient_type == RecipientType.person:
            self.send_message(self.create_notice_group_message(markdown, staff_login), staff_login,
                              RecipientType.BotGroup)

    def get_sent_message_task_id(self):
        resource = sdk2.Resource.find(
            type=RESOURCE_TYPE,
            state='READY',
        ).first()
        if not resource:
            logging.info('ATTENTION!!! Could not find resource of previous run')
            return []
        tasks_ids = []
        path = sdk2.ResourceData(sdk2.Resource[resource.id]).path
        with open(path, 'r') as f:
            file_contents = f.read()
        logging.info('Previous run information')
        for task_id in file_contents.splitlines():
            logging.info(f'task_id: {task_id}')
            tasks_ids.append(int(task_id))
        return tasks_ids

    def get_smart_report(self, file_url):
        file_url += '/smart_report.txt'
        logging.info(f'smart report url: {file_url}')
        try:
            text = urllib.request.urlopen(file_url).read().decode('utf-8')
        except Exception as e:
            logging.info(f'smart report status: {e}')
            return ''
        if len(text) < int(self.Parameters.max_smart_len) and len(text.splitlines()) < int(self.Parameters.mx_line_num):
            return '``` {}```\n'.format(text)
        return ''

    def get_report_message(self, task_id):
        url = f'https://sandbox.yandex-team.ru/api/v1.0/task/{task_id}/resources'
        response = requests_wrapper.get(url, verify=False)
        requests_wrapper.check_status_code(response)
        ans = response.json()

        for i in range(ans['total']):
            if ans['items'][i]['file_name'] == 'report_dir':
                return '*smart report:* [{}]({})\n'.format(str(ans['items'][i]['id']),
                                                           ans['items'][i]['http']['proxy']) + self.get_smart_report(
                    ans['items'][i]['http']['proxy'])
        return ''

    def add_new_un_problem(self, task_id):
        message = []
        type_description = get_task_type_description(task_id)
        if type_description is None:
            return message
        smart_report = self.get_report_message(task_id)
        message.append('```{}```'.format(type_description))
        message.append(f'*Task:* [{task_id}](https://sandbox.yandex-team.ru/task/{task_id})')
        message.append(smart_report)
        return message

    def aggregate_messages_and_send(self, tests_type, revision, revision_id, closed_intervals):
        def check_test(test_name):
            logging.info(test_name)
            closed_interval = closed_intervals[test_name]
            logging.info(f'closed_interval: {closed_interval}')
            if len(closed_interval) == 0:
                return True
            first_revision = closed_interval[0]['first_revision']
            logging.info(f'revision_id  {revision_id}   test_name{test_name}')
            logging.info(closed_interval)
            return first_revision > revision_id

        if len(revision[tests_type]) == 0:
            return False
        revision_problem_tests_name = [id_test[1] for id_test in revision[tests_type]]
        revision_owner = revision['owner']

        logging.info(f'revision owner: {revision_owner}   revision_id: {revision_id}')
        logging.info(f'tests type: {tests_type}   revision problems ids: {revision[tests_type]}')

        remaining_tests = [
            test_name for test_name in closed_intervals.keys()
            if ('PERFORMANCE' in test_name) == (tests_type == 'perf') and test_name not in revision_problem_tests_name
        ]
        logging.info(f'remaining tests: {remaining_tests}')

        for test in remaining_tests:
            if not check_test(test):
                logging.info(f'tests did not pass: {test}')
                return False

        logging.info('tests passed')
        message = []
        for problem in revision[tests_type]:
            message += self.add_new_un_problem(problem[0])
        message.insert(0, revision['comment'])
        message.insert(0, '*Unresolved TE Problems*')
        logging.info('----MESSAGE----\n')
        logging.info('\n'.join(message))
        if self.is_robot(revision_owner):
            logging.info(f'revision id({revision_id})  owner({revision_owner})- is bot')
            logging.info(f'send message to chat_name: {self.Parameters.telegram_notice_group}')
            self.send_message(
                self.create_notice_group_message(message, revision_owner),
                revision_owner, RecipientType.BotGroup)
        else:
            logging.info(f'send message to {revision_owner}')
            self.send_message(message, revision_owner, RecipientType.person)
        return True

    def write_current_report(self, un_problems, closed_intervals):
        previous_report = self.get_sent_message_task_id()

        revisions = {}
        notified_problems = []
        for un_problem in un_problems:
            task_id = un_problem['test_diff/task_id']
            revision_id = un_problem['revision']
            owner = un_problem['owner']
            logging.info('Unresolved problems: revision_id({}) task_id({}) owner({})'.format(revision_id, task_id,
                                                                                             owner))
            if task_id in previous_report:
                notified_problems.append(str(task_id))
                continue
            test_name = un_problem['test_name']
            if revision_id not in revisions:
                revisions[revision_id] = {'owner': owner, 'perf': [], 'not_perf': [],
                                          'comment': '``` {}```'.format(un_problem['test_diff/revision2_info/comment'])
                                          }

            if 'PERFORMANCE' in test_name:
                logging.info(f'test name: {test_name} is PERFORMANCE')
                revisions[revision_id]['perf'].append((task_id, test_name))
            else:
                logging.info(f'test name: {test_name}  is not PERFORMANCE')
                revisions[revision_id]['not_perf'].append((task_id, test_name))

        for revision_id, revision in revisions.items():
            if self.aggregate_messages_and_send('perf', revision, revision_id, closed_intervals):
                notified_problems += [item[0] for item in revision['perf']]
            if self.aggregate_messages_and_send('not_perf', revision, revision_id, closed_intervals):
                notified_problems += [item[0] for item in revision['not_perf']]
        resource = JugglerPreviousSentUnresolvedProblemIds(self, 'Unresolved Problems', 'YabsLastRevisionOut.txt')
        notified_problems_str = '\n'.join(str(task_id) for task_id in notified_problems)
        logging.info(f'current list of notified problems: {notified_problems_str}')

        notified_problems_as_bytes = str.encode(notified_problems_str)
        resource.path.write_bytes(notified_problems_as_bytes)

    def is_work_time(self):
        now = datetime.datetime.now()
        is_work_hour = self.Parameters.first_working_hour <= now.hour < self.Parameters.last_working_hour
        is_work_day = self.Parameters.first_working_day <= now.isoweekday() <= self.Parameters.last_working_day
        if is_work_hour and is_work_day:
            return True
        return False

    def on_execute(self):
        if not self.is_work_time():
            logging.info('Failed to send messages\n Reason: not working time')
            return
        testenv_project = self.Parameters.testenv_project
        un_problems = get_unresolved_problems(testenv_project)
        testenv_job_names = [
            job['name'] for job in TEClient.get_jobs(testenv_project)
            if job['notifications_enabled']
        ]
        closed_intervals = {}
        for job_name in testenv_job_names:
            closed_intervals[job_name] = list(TEClient.get_unchecked_intervals(testenv_project, job_name, 0))

        self.write_current_report(un_problems, closed_intervals)
