import json
import logging

from sandbox import sdk2
from sandbox.common import rest

from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.yabs import YabsServerMeter
from sandbox.projects.common.yabs.server.util.general import find_last_ready_resource
from sandbox.projects.common.yabs.server.util import truncate_output_parameters


class YabsServerCrossCheck(YabsServerMeter.YabsServerMeter):
    type = 'YABS_SERVER_CROSS_CHECK'
    description = 'Run for crosscheck'

    def run_shoot_crosscheck(self, cl, get_clean_bases=False):
        shard_list = self.ctx['_func_shard_list']

        if get_clean_bases:
            db_resource_list = self.get_clean_bases(cl)
        else:
            mbb_set_non_switcher = set(cl.task[self.ctx['subtask_make_bin_bases_non_switcher']].context.read().get('bin_base_res_ids'))
            mbb_set_switcher_patched = set(cl.task[self.ctx['subtask_make_bin_bases_switcher_patched']].context.read().get('bin_base_res_ids'))
            db_resource_list = list(mbb_set_non_switcher | mbb_set_switcher_patched)

        if not db_resource_list:
            raise SandboxTaskFailureError("Something wrong with MAKE_BIN_BASES results.")

        subtask_list = []

        self.ctx['_func_tasks'] = {'bs': {}, 'yabs': {}, 'bsrank': {}}

        for meta_mode in self.ctx['func_modes']:
            if self.ctx.get('global_request_log_' + meta_mode):
                request_log_id = self.ctx['global_request_log_' + meta_mode]
            else:
                request_log_id = find_last_ready_resource(
                    'YABS_SERVER_REQUEST_LOG_GZ',
                    attrs={"testenv_switch_trigger": meta_mode + '_func'}
                ).id
                self.ctx['global_request_log_' + meta_mode] = request_log_id

            request_log_resource = sdk2.Resource[request_log_id]
            cachedaemon_dump_id = request_log_resource.cachedaemon_dump_res_id
            for shard in shard_list:
                update_parameters_resource = self.ctx.get('global_ft_shoot_settings_resource', None)
                ft_input_parameters = {
                    'server_resource': self.ctx['local_bs_release_tar_id'],
                    'meta_mode': meta_mode,
                    'stat_shards': [int(shard)],
                    'binary_base_resources': db_resource_list,
                    'cache_daemon_stub_resource': cachedaemon_dump_id,
                    'requestlog_resource': request_log_id,
                    'update_parameters_resource': update_parameters_resource,
                    'update_parameters_resources': [update_parameters_resource] if update_parameters_resource else [],
                }
                shoot_task_class = sdk2.Task['YABS_SERVER_B2B_FUNC_SHOOT_2']
                subtask = shoot_task_class(
                    shoot_task_class.current,
                    description='shoot {}, shard {}'.format(meta_mode, shard),
                    owner=self.owner,
                    prioriy=self.priority,
                    **truncate_output_parameters(ft_input_parameters, shoot_task_class.Parameters)
                ).save()
                headers_update_dict = subtask.Parameters.headers_update_dict
                debug_options = json.loads(headers_update_dict['x-yabs-debug-options-json'])
                debug_options['mx'] = True
                headers_update_dict['x-yabs-debug-options-json'] = json.dumps(debug_options)
                subtask.Parameters.headers_update_dict = headers_update_dict

                if self.ctx.get('experiment_id'):
                    queryargs_update_dict = subtask.Parameters.queryargs_update_dict
                    queryargs_update_dict.update({YabsServerMeter.EXP_ID_ARG: str(self.ctx['experiment_id'])})
                    subtask.Parameters.queryargs_update_dict = queryargs_update_dict

                subtask.save()
                subtask.enqueue()
        subtask_list.append(subtask.id)
        self.ctx['_func_tasks'][meta_mode][shard] = subtask.id

        self.ctx['subtasks_shoot'] = subtask_list

        self.wait_tasks(
            tasks=subtask_list,
            statuses=[self.Status.SUCCESS, self.Status.FAILURE],
            wait_all=True
        )

    def on_execute(self):
        cl = rest.Client()

        if not self.ctx.get('local_bs_release_tar_id'):
            logging.info("Get spec")
            self.put_spec_into_ctx(cl)

            self.ctx['func_modes'] = []
            if self.ctx.get('launch_rank_bs'):
                self.ctx['func_modes'] = ['bs', 'bsrank']

            if self.ctx.get('launch_yabs'):
                self.ctx['func_modes'].append('yabs')

            self.ctx['_child_description'] = 'Run for crosscheck'
            self.ctx['_func_shard_list'] = self.get_shard_list(cl)

        if self.ctx.get('local_bs_release_tar_id') and not self.ctx.get('switcher_revision') and self.ctx.get('switcher_patch'):
            self.get_switcher_revision(cl)

        if (self.ctx.get('subtask_make_bin_bases_non_switcher') or not self.ctx.get('switcher_patch')) and not self.ctx.get('subtasks_shoot') and self.ctx['func_modes']:
            logging.info("Run shoot")
            self.run_shoot_crosscheck(cl, not self.ctx.get('switcher_patch'))
