import json
import logging

from sandbox.common import rest

from sandbox import sdk2
from sandbox.common.types.resource import State
from sandbox.common.errors import TaskError
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxBoolParameter, SandboxIntegerParameter
from sandbox.projects.yabs import YabsServerOneShot
from sandbox.projects.common.yabs.server.util.general import find_last_ready_resource
from sandbox.projects.common.yabs.server.util import truncate_output_parameters


GROUP_SWITCHER = 'experiment-switcher'
EXP_ID_ARG = 'experiment-id'


def _run_shoot_task(
        parent_task,
        baseline_task_id,
        task_type,
        binary_base_resources,
        description=None,
        other_params=None,
):
    input_parameters = truncate_output_parameters(
        dict(sdk2.Task[baseline_task_id].Parameters),
        task_type.Parameters)
    input_parameters.update(
        binary_base_resources=binary_base_resources,
        **(other_params or {})
    )
    return task_type(
        parent_task,
        description=description,
        owner=parent_task.Parameters.owner,
        priority=parent_task.Parameters.priority,
        tags=parent_task.Parameters.tags,
        **input_parameters
    ).enqueue().id


class SwitcherBases(SandboxStringParameter):
    name = 'switcher_bases'
    description = 'bin bases affected by switcher (space-separated)'
    default_value = 'dbe formula'
    group = GROUP_SWITCHER


class SwitcherRevision(SandboxStringParameter):
    name = 'switcher_revision'
    description = 'experiment-switcher revision (bs_release_tar base_revision by default)'
    group = GROUP_SWITCHER


class SwitcherPatch(SandboxStringParameter):
    name = 'switcher_patch'
    description = 'arcadia patch of experiment-switcher dir'
    group = GROUP_SWITCHER


class SwitcherArguments(SandboxStringParameter):
    name = 'switcher_argruments'
    description = 'experiment-switcher running argruments comma separated'
    group = GROUP_SWITCHER


GROUP_LOAD = 'Child load shoots'


class LaunchYabs(SandboxBoolParameter):
    name = 'launch_yabs'
    description = 'launch yabs(search) shoots'
    group = GROUP_LOAD


class LaunchRankBs(SandboxBoolParameter):
    name = 'launch_rank_bs'
    description = 'launch bs(partner)+rank shoots'
    group = GROUP_LOAD


class ExperimentIdForB2B(SandboxIntegerParameter):
    name = 'experiment_id'
    description = 'Add experiment-id in query for b2b shooting (do not work for performance yet)'


class YabsServerMeter(YabsServerOneShot.YabsServerOneShot):
    type = 'YABS_SERVER_METER'
    description = 'Test for switcher'

    input_parameters = (YabsServerOneShot.SpecResourceID, SwitcherBases, SwitcherRevision, SwitcherPatch, SwitcherArguments, LaunchYabs, LaunchRankBs, ExperimentIdForB2B)

    use_separated_bases = True

    def join_new_mysql_archive(self):
        new_mysql_archive_res = sdk2.Resource["YABS_MYSQL_ARCHIVE_CONTENTS"].find(
            task=sdk2.Task[self.ctx['subtask_get_sql_archive']],
            state=State.READY
        ).first()
        new_mysql_archive_res.next_id = self.ctx.get('global_mysql_archive_id')
        return new_mysql_archive_res.id

    def get_fresh_dbe_and_formula_tables(self):
        subtask = self.create_subtask(
            task_type='YABS_SERVER_GET_SQL_ARCHIVE',
            description='restore dbe and formula tables for switcher',
            input_parameters={
                'bin_db_list': 'formula dbe',
            }
        )
        self.ctx['subtask_get_sql_archive'] = subtask.id
        self.wait_tasks(
            tasks=[subtask.id],
            statuses=[self.Status.SUCCESS, self.Status.FAILURE],
            wait_all=True
        )

    def run_make_bin_bases_switcher(self, cl, setup_ctx):
        def _create_subtask(description='make all bases', input_parameters=None):
            default_input_parameters = self.get_common_mbb_params()
            default_input_parameters.update({
                'bin_db_list': bin_db_list,
                'cs_import_ver': setup_ctx['cs_import_ver'],
                'do_not_restart': True,
                'glue_reduce': True,
                'reuse_existing_bases': True,
            })
            default_input_parameters.update(input_parameters or {})
            return self.create_subtask(
                task_type='YABS_SERVER_MAKE_BIN_BASES',
                description=description,
                priority=("SERVICE", "NORMAL"),  # subtasks either run in YABS pool or are forced to BACKGROUND:HIGH
                input_parameters=default_input_parameters
            )

        bin_db_set = set()
        for role in self.iter_roles():
            bin_db_set.update(setup_ctx.get('tags_common_{}'.format(role)))
            for shard in self.ctx['_load_shard_list']:
                bin_db_set.update(setup_ctx.get('tags_shard_{}_{}'.format(shard, role)))

        bin_db_list = ' '.join(list(bin_db_set))

        switcher_db_set = set(['dbe', 'formula'])
        non_switcher_db_set = bin_db_set - switcher_db_set

        switcher_db_list = ' '.join(switcher_db_set)
        non_switcher_db_list = ' '.join(non_switcher_db_set)

        gen_bin_bases_flags_id = self.ctx.get(YabsServerOneShot.GenBinBasesFlagsID.name)
        raw_attrs = cl.resource[gen_bin_bases_flags_id].attribute.read()
        attrs = {a['name']: a['value'] for a in raw_attrs}
        add_base_options = attrs.get('add_options')

        options_list = add_base_options.strip() if add_base_options is not None else ''

        options_list_switcher = 'run_switcher ' + options_list

        input_non_switcher_parameters = {
            'mysql_archive_contents': self.ctx['global_mysql_archive_id'],
            'bin_db_list': non_switcher_db_list,
            'options_list': options_list,
            'reuse_existing_bases': True,
        }
        subtask_non_switcher = _create_subtask('make all bases (non switcher)', input_non_switcher_parameters)

        input_switcher_parameters = {
            'mysql_archive_contents': self.ctx['new_mysql_archive_id'],
            'bin_db_list': switcher_db_list,
            'options_list': options_list_switcher,
            'reuse_existing_bases': False,
            'switcher_revision': self.ctx.get('switcher_revision'),
            'switcher_argruments': self.ctx.get('switcher_argruments')
        }
        subtask_switcher_base = _create_subtask('make all bases (switcher BASE)', input_switcher_parameters)

        input_switcher_parameters['arcadia_patch'] = self.ctx['switcher_patch']
        subtask_switcher_patched = _create_subtask('make all bases (switcher PATCH)', input_switcher_parameters)

        tasks = [subtask_non_switcher.id, subtask_switcher_base.id, subtask_switcher_patched.id]
        self.ctx['subtask_make_bin_bases_non_switcher'] = subtask_non_switcher.id
        self.ctx['subtask_make_bin_bases_switcher_base'] = subtask_switcher_base.id
        self.ctx['subtask_make_bin_bases_switcher_patched'] = subtask_switcher_patched.id

        self.wait_tasks(
            tasks=tasks,
            statuses=[self.Status.SUCCESS, self.Status.FAILURE],
            wait_all=True
        )

    def get_ft_shoot_settings(self):
        ft_settings_resource_id = self.ctx.get('global_ft_shoot_settings_resource')
        if ft_settings_resource_id:
            ft_settings_path = self.sync_resource(ft_settings_resource_id)
            if ft_settings_path:
                with open(ft_settings_path) as ft_settings_file:
                    ft_settings = json.load(ft_settings_file)
                    return ft_settings
        return None

    def run_shoot_switcher(self, cl):

        shard_list = self.ctx['_load_shard_list']

        mbb_set_non_switcher = set(cl.task[self.ctx['subtask_make_bin_bases_non_switcher']].context.read().get('bin_base_res_ids'))
        mbb_set_switcher_base = set(cl.task[self.ctx['subtask_make_bin_bases_switcher_base']].context.read().get('bin_base_res_ids'))
        mbb_set_switcher_patched = set(cl.task[self.ctx['subtask_make_bin_bases_switcher_patched']].context.read().get('bin_base_res_ids'))

        mbb_list_base = list(mbb_set_non_switcher | mbb_set_switcher_base)
        mbb_list_patched = list(mbb_set_non_switcher | mbb_set_switcher_patched)

        if not mbb_list_base or not mbb_list_patched:
            raise SandboxTaskFailureError("Something wrong with MAKE_BIN_BASES results.")

        subtask_list = []

        self.ctx['_load_tasks_base'] = {'bs': {}, 'yabs': {}, 'bsrank': {}}
        self.ctx['_load_tasks_patched'] = {'bs': {}, 'yabs': {}, 'bsrank': {}}
        self.ctx['_func_tasks_base'] = {'bs': {}, 'yabs': {}, 'bsrank': {}}
        self.ctx['_func_tasks_patched'] = {'bs': {}, 'yabs': {}, 'bsrank': {}}

        logging.info('Force experiment_id %s', self.ctx.get('experiment_id'))
        for meta_mode in self.ctx['load_modes']:
            dolbilka_plan_id = self.ctx['global_dolbilka_plan_' + meta_mode]

            if self.ctx.get('global_request_log_' + meta_mode):
                request_log_id = self.ctx['global_request_log_' + meta_mode]
            else:
                request_log_id = find_last_ready_resource(
                    'YABS_SERVER_REQUEST_LOG_GZ',
                    attrs={"testenv_switch_trigger": meta_mode + '_func'}
                ).id
                self.ctx['global_request_log_' + meta_mode] = request_log_id

            request_log_resource = sdk2.Resource[request_log_id]
            cachedaemon_dump_id = request_log_resource.cachedaemon_dump_res_id

            for shard in shard_list:
                for bases_list, part in [(mbb_list_base, 'base'), (mbb_list_patched, 'patched')]:
                    dolbilka_plan_resource = sdk2.Resource[dolbilka_plan_id]
                    cachedaemon_dump_id = dolbilka_plan_resource.cachedaemon_dump_res_id
                    load_task_class = sdk2.Task['YABS_SERVER_STAT_PERFORMANCE_BEST_2']
                    subtask = load_task_class(
                        load_task_class.current,
                        description='shoot {}, shard {}, {} switcher'.format(meta_mode, shard, part.upper()),
                        priority=self.priority,
                        owner=self.owner,
                        **{
                            'binary_base_resources': bases_list,
                            'server_resource': self.ctx['local_bs_release_tar_id'],
                            'shoot_plan_resource': dolbilka_plan_id,
                            'cache_daemon_stub_resource': cachedaemon_dump_id,
                            'meta_mode': meta_mode,
                            'specify_cluster_set_config': True,
                            'stat_shards': [int(shard)],
                            'stat_shoot_threads': 480,
                            'store_plan_in_memory': True,
                            'request_timeout': 600,
                            'stat_shoot_sessions': 3,
                            'stat_shoot_threads': 480,
                            'stat_shoot_request_limit': 500000,
                            'shoot_request_limit': 100000,
                            'failure_limit': 2,
                            'other_break_limit': 2,
                            'abandoned_limit': 10,
                            'subtasks_with_corrections_count': 0,
                            'subtasks_without_corrections_count': 15,
                            'stat_store_request_log': True,

                            'generic_disk_space': 100,
                            'use_tmpfs': True,

                        }
                    ).enqueue()
                    subtask_list.append(subtask.id)
                    self.ctx['_load_tasks_{}'.format(part)][meta_mode][shard] = subtask.id

                    custom_task_params = {
                        'cache_daemon_stub_resource': cachedaemon_dump_id,
                        'stat_shards': [int(shard)],
                        'maximum_bad_response_ratio': 0.3,
                        'ignored_ext_tags': ['bigb_balancer', 'bigb_rf_balancer'],
                    }
                    if self.ctx.get('experiment_id'):
                        queryargs_update_dict = subtask.Parameters.queryargs_update_dict
                        queryargs_update_dict.update({EXP_ID_ARG: str(self.ctx['experiment_id'])})
                        custom_task_params['queryargs_update_dict'] = queryargs_update_dict

                    ft_shoot_task_id = _run_shoot_task(
                        parent_task=sdk2.Task.current,
                        baseline_task_id=self.ctx['local_func_shoot_task_id_' + meta_mode],
                        task_type=sdk2.Task['YABS_SERVER_B2B_FUNC_SHOOT_2'],
                        binary_base_resources=bases_list,
                        description='shoot {}, shard {}, {} switcher'.format(meta_mode, shard, part.upper()),
                        other_params=custom_task_params
                    )
                    subtask_list.append(ft_shoot_task_id)
                    self.ctx['_func_tasks_{}'.format(part)][meta_mode][shard] = ft_shoot_task_id

        self.ctx['subtasks_shoot'] = subtask_list

        self.wait_tasks(
            tasks=subtask_list,
            statuses=[self.Status.SUCCESS, self.Status.FAILURE],
            wait_all=True
        )

    def create_load_report_switcher(self, cl):
        report = 'Load test results:'
        local_rps_dict_base = self.ctx.get('local_rps_dict_base')
        local_rps_dict_patched = self.ctx.get('local_rps_dict_patched')
        if local_rps_dict_base and local_rps_dict_patched:
            load_shards = self.ctx['_load_shard_list']
            for meta_mode in self.ctx['load_modes']:
                report += '<br/>'
                for shard in load_shards:
                    try:
                        rps_base = local_rps_dict_base[meta_mode][shard]
                        rps_patched = local_rps_dict_patched[meta_mode][shard]
                        rps_diff = 100.0 * rps_patched / rps_base - 100.0
                        report += '{}/{}: {:.1f} RPS = {:.1f} {:+.1f}%<br/>'.format(
                            meta_mode, shard, rps_patched, rps_base, rps_diff
                        )
                    except Exception as e:
                        raise TaskError('Cannot build performance report: %s', e.message)
        return report

    def run_diff_switcher(self, cl):
        shard_list = self.ctx['_load_shard_list']

        subtask_list = []
        subtask_cmp = {}
        cmp_task_class = sdk2.Task['YABS_SERVER_B2B_FUNC_SHOOT_CMP']
        for meta_mode in self.ctx['load_modes']:
            for shard in shard_list:
                diff_subtask = cmp_task_class(
                    cmp_task_class.current,
                    description='func shoot diff {} {}'.format(meta_mode, shard),
                    owner=self.owner,
                    priority=self.priority,
                    pre_task=self.ctx['_func_tasks_base'][meta_mode][shard],
                    test_task=self.ctx['_func_tasks_patched'][meta_mode][shard],
                )
                diff_subtask.enqueue()
                subtask_list.append(diff_subtask.id)
                subtask_cmp.setdefault(meta_mode, {})[shard] = diff_subtask.id

        self.ctx['subtasks_cmp'] = subtask_cmp
        self.wait_subtasks(subtask_list)

    def create_func_report_switcher(self, cl):
        def html_hyperlink(link, text, shard):
            return '<a href="{link}" target="_blank">{text} {shard}</a>'.format(link=link, text=text, shard=shard)

        def startrek_hyperlink(link, text, shard):
            return '(({link} {text} {shard}))'.format(link=link, text=text, shard=shard)

        html_report = 'FUNC:<br/>'
        startrek_report = 'FUNC\n'

        report_template = '{meta_mode_link}: {tests_status}'
        unknown_tests_status_text = 'Tests status is unknown'

        shard_list = self.ctx['_load_shard_list']
        for shard in shard_list:
            for meta_mode in self.ctx['load_modes']:
                cmp_task_id = self.ctx['subtasks_cmp'][meta_mode][shard]
                cmp_ctx = cl.task[cmp_task_id].context.read()
                cmp_task_url = 'https://sandbox.yandex-team.ru/task/{task_id}/view'.format(task_id=cmp_task_id)
                short_report_text = cmp_ctx.get('short_report_text')
                short_report_link = cmp_ctx.get('short_report_link')

                html_report += report_template.format(
                    meta_mode_link=html_hyperlink(cmp_task_url, meta_mode, shard),
                    tests_status=(html_hyperlink(short_report_link, short_report_text, shard)
                                  if short_report_text
                                  else unknown_tests_status_text),
                ) + '<br />'
                startrek_report += report_template.format(
                    meta_mode_link=startrek_hyperlink(cmp_task_url, meta_mode, shard),
                    tests_status=(startrek_hyperlink(short_report_link, short_report_text, shard)
                                  if short_report_text
                                  else unknown_tests_status_text),
                ) + '\n'

        return html_report, startrek_report

    def get_switcher_revision(self, cl):
        raw_attrs = cl.resource[self.ctx['local_bs_release_tar_id']].attribute.read()
        attrs = {a['name']: a['value'] for a in raw_attrs}
        self.ctx['switcher_revision'] = attrs.get('base_revision')

    def iter_roles(self):
        if self.ctx.get('launch_rank_bs'):
            yield 'bs'
            yield 'bsrank'
        if self.ctx.get('launch_yabs'):
            yield 'yabs'

    def on_execute(self):
        self.set_info('This task is no longer supported, contact <a href="https://staff.yandex-team.ru/igorock", target="_blank">igorock@</a> for more details', do_escape=False)
        raise NotImplementedError('This task is deprecated, use AB-experiment instead')

        cl = rest.Client()

        if not self.ctx.get('local_bs_release_tar_id'):
            logging.info("Get spec")
            self.ctx['launch_load'] = True
            self.put_spec_into_ctx(cl)

            self.ctx['load_modes'] = list(self.iter_roles())

            self.ctx['_child_description'] = 'Test of switcher'
            self.ctx['_load_shard_list'] = self.get_shard_list(cl, load=True)

        # check input parameters
        if not any([self.ctx.get('switcher_revision'), self.ctx.get('switcher_patch'), self.ctx.get('switcher_argruments')]):
            raise SandboxTaskFailureError('There are no parameters for running experiment switcher')

        if self.ctx.get('switcher_argruments') and not self.ctx.get('switcher_revision'):
            raise SandboxTaskFailureError("Can't run experiment switcher with additional arguments without a certain revision")

        if self.ctx.get('local_bs_release_tar_id') and not self.ctx.get('switcher_revision'):
            self.get_switcher_revision(cl)

        if not self.ctx.get('subtask_get_sql_archive'):
            logging.info("Restore tables for dbe and formula")
            self.get_fresh_dbe_and_formula_tables()

        if not self.ctx.get('subtask_make_bin_bases_non_switcher') and self.ctx.get('subtask_get_sql_archive'):

            if self.ctx.get('local_setup_ya_make_id'):
                subtask_setup = self.ctx['local_setup_ya_make_id']
                setup_ctx = cl.task[subtask_setup].context.read()

            logging.info("Run make bin bases")
            self.ctx['new_mysql_archive_id'] = self.join_new_mysql_archive()
            logging.info("Run with new mysql archive %s", self.ctx['new_mysql_archive_id'])
            self.run_make_bin_bases_switcher(cl, setup_ctx)

        if self.ctx.get('subtask_make_bin_bases_non_switcher') and not self.ctx.get('subtasks_shoot') and self.ctx['load_modes']:
            logging.info("Run shoot")
            self.run_shoot_switcher(cl)

        if self.ctx.get('subtasks_shoot') and not self.ctx.get('subtasks_cmp'):
            rps_dict_base = {'bs': {}, 'yabs': {}, 'bsrank': {}}
            rps_dict_patched = {'bs': {}, 'yabs': {}, 'bsrank': {}}
            for meta_mode in self.ctx['load_modes']:
                for shard in self.ctx['_load_shard_list']:
                    rps_dict_base[meta_mode][shard] = cl.task[self.ctx['_load_tasks_base'][meta_mode][shard]].context.read().get('rps_hl_median')
                    rps_dict_patched[meta_mode][shard] = cl.task[self.ctx['_load_tasks_patched'][meta_mode][shard]].context.read().get('rps_hl_median')
            self.ctx['local_rps_dict_base'] = rps_dict_base
            self.ctx['local_rps_dict_patched'] = rps_dict_patched
            report = self.create_load_report_switcher(cl)
            self.set_info(report, do_escape=False)

            self.run_diff_switcher(cl)

        if self.ctx.get('subtasks_cmp'):
            logging.info("Create func report")
            report, st_report = self.create_func_report_switcher(cl)
