# -*- coding: utf-8 -*-

import logging
from datetime import timedelta

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.task import Status
from sandbox.common.urls import get_task_link, get_resource_link
from sandbox.projects.common.build.parameters import ArcadiaPatch, ArcadiaUrl, YtStore, YtProxy, YtDir, \
    YtTokenVaultOwner, YtTokenVaultName
from sandbox.projects.common.constants import SEMI_DISTBUILD_BUILD_SYSTEM, RELEASE_BUILD_TYPE, YMAKE_BUILD_SYSTEM
from sandbox.projects.yabs.YabsServerNewRuntime.YabsNewRuntimeDeploy import YabsNewRuntimeDeploy
from sandbox.projects.yabs.qa.resource_types import BS_RELEASE_TAR
from sandbox.projects.yabs.release.version.version import get_version_from_arcadia_url
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.projects.yabs.YabsServerNewRuntime.utils.st_helper import StartrekHelper
from sandbox.projects.yabs.qa.utils.general import html_hyperlink

logger = logging.getLogger('YabsServerDeployOnNewRuntime')
logging.basicConfig(level=logging.INFO)
BS_PACKAGE = 'yabs/server/packages/yabs-server-bundle.json'


class YabsNewRuntimeBuild(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        disk_space = 2048
        environments = [sdk2.environments.PipEnvironment('startrek_client')]

    class Parameters(sdk2.Parameters):
        description = 'Build yabs_server for preproduction_yabs_frontend_server_bs_dev1_meta/stat'
        arcadia_url = ArcadiaUrl()
        patch = ArcadiaPatch()
        issue_queue = sdk2.parameters.String("Queue in which a new ticket will be created", default='BSNEWRUNTIME')
        st_token = sdk2.parameters.String("ST token vault name", default="st_token", required=True)
        responsible = sdk2.parameters.List('List of responsible', sdk2.parameters.Staff,
                                           description='People who can confirm the deployment',
                                           default=['wunder9l'])
        with sdk2.parameters.Group("Build cache parameters") as build_cache_parameters:
            yt_store = YtStore(default=True)
            yt_proxy = YtProxy(default="hahn")
            yt_dir = YtDir(default="//home/yabs-cs-preprod/igorock/build-cache-object-files")
            yt_token_vault_owner = YtTokenVaultOwner(default="wunder9l")
            yt_token_vault_name = YtTokenVaultName(default="yt_token")

        with sdk2.parameters.Output:
            issue = sdk2.parameters.Url("Link to created issue")
            bs_release_tar_resource = sdk2.parameters.Resource("Yabs server package", resource_type=BS_RELEASE_TAR)

    def _create_st_client(self):
        """
        Creates and returns Startrek client using self.author (from parent sdk2.Task) and self.Parameters.st_token
        """
        from startrek_client import Startrek

        token = sdk2.Vault.data(self.author, self.Parameters.st_token)
        return Startrek(useragent='YabsServerDeployOnNewRuntime task', token=token)

    def get_issue_summary(self):
        return 'Выкладка на экспериментальные группировки preproduction_yabs_frontend_server_bs_dev1_meta/stat'

    def make_aradia_url(self):
        parsed_url = Arcadia.parse_url(self.Parameters.arcadia_url)
        url = 'https://a.yandex-team.ru/' + parsed_url.path
        if parsed_url.revision:
            url += '?rev={}'.format(parsed_url.revision)
        return url

    def get_issue_description(self):
        arcadia_url = self.make_aradia_url()
        text = 'Будет собран и выложен движок, собранный из (({} {}))'.format(arcadia_url, self.Parameters.arcadia_url)
        if self.Parameters.patch:
            text += '\nс наложенным патчем {}'.format(self.Parameters.patch)
        return text

    def get_version(self):
        return get_version_from_arcadia_url(self.Parameters.arcadia_url)

    def build_server(self):
        basic_version = self.get_version()
        resource_type = BS_RELEASE_TAR.name
        build_task_template = sdk2.Task["BUILD_YABS_SERVER"]
        subtask_parameters = dict(
            kill_timeout=timedelta(minutes=20).total_seconds(),
            packages_to_build={"yabs/server/packages/yabs-server-bundle.json": resource_type},
            packages_to_build_debug={},
            tools_to_build={},
            checkout_arcadia_from_url=self.Parameters.arcadia_url,
            arcadia_patch=self.Parameters.patch,
            build_system=SEMI_DISTBUILD_BUILD_SYSTEM,
            fallback_build_system=YMAKE_BUILD_SYSTEM,
            sanitize=None,
            max_retries=1,
            build_type=RELEASE_BUILD_TYPE,
            parallel_build=True,
            yt_store=self.Parameters.yt_store,
            yt_proxy=self.Parameters.yt_proxy,
            yt_dir=self.Parameters.yt_dir,
            yt_token_vault_owner=self.Parameters.yt_token_vault_owner,
            yt_token_vault_name=self.Parameters.yt_token_vault_name,
            yt_put=False,
            yt_replace_result=False,
            # yt_store_codec=YtStoreCodec(),
            # yt_max_cache_size=YtMaxCacheSize(),
            compute_md5=True,
            compute_version=True,
        )
        logger.debug("Running BUILD_YABS_SERVER with parameters: %s", subtask_parameters)

        subtask = build_task_template(
            self,
            description="BS_NEW_RUNTIME BUILD [{build_type}] {resource_type} {basic_version} + patch[{patch}]".format(
                build_type=RELEASE_BUILD_TYPE,
                resource_type=str(resource_type),
                basic_version=basic_version,
                patch=self.Parameters.patch
            ),
            hints=list(self.hints),
            **subtask_parameters
        ).enqueue()
        logger.info('Created subtask: {}'.format(subtask))
        return subtask

    def _create_deploy_task_draft(self, resource_id):
        logger.info('Creating deploy task')
        version = self.get_version()
        sub_task = YabsNewRuntimeDeploy(
            self,
            description="Deploy {major}-{minor} with patch ({patch}) [(({resource_url} {resource_id}))] on new runtime group".format(
                major=version.major,
                minor=version.minor,
                patch=self.Parameters.patch,
                resource_url=get_resource_link(resource_id),
                resource_id=resource_id
            )
        )
        # modify task parameters
        sub_task.Requirements.tasks_resource = self.Requirements.tasks_resource
        sub_task.Parameters.issue = self.Context.issue_key
        sub_task.Parameters.st_token = self.Parameters.st_token
        sub_task.Parameters.major_version = version.major
        sub_task.Parameters.minor_version = version.minor
        sub_task.Parameters.resource_with_server = resource_id
        sub_task.Parameters.comment = 'Deployment of {}-{} with patch {}'.format(version.major, version.minor, self.Parameters.patch)
        sub_task.save()
        return sub_task.id

    def make_report(self, task, deploy_task_id, resource):
        return '''=====yabs_server was built successfully
        **Base version**: {base_version} ({arcadia_url})

        **Patch**: {patch}

        **(({task} Successful build task))**

        **(({resource} BS_RELEASE_TAR))**
        -----
        **(({deploy_task} You can run deploy task after confirmation))**
        '''.format(
            base_version=self.get_version(),
            arcadia_url=self.Parameters.arcadia_url,
            patch=self.Parameters.patch,
            task=get_task_link(task.id),
            resource=get_resource_link(resource.id),
            deploy_task=get_task_link(deploy_task_id)
        )

    def report_successful_build(self, st_helper, task, deploy_task_id, resource):
        report = self.make_report(task, deploy_task_id, resource)
        st_helper.comment(self.Context.issue_key, report, summonees=self.Parameters.responsible)

    def report_failure_build(self, st_helper, broken_task):
        report = '=====yabs_server was NOT build, all attempts failed: (({} {}))'\
            .format(get_task_link(broken_task.id), broken_task.id)
        st_helper.comment(self.Context.issue_key, report, summonees=self.author)

    def send_build_result_message(self):
        st_helper = StartrekHelper(self._create_st_client())
        logger.info('Checking results of build subtask')
        build_task = sdk2.Task[self.Context.build_subtask]
        logger.debug('Task: {}'.format(build_task))
        logger.info('BUILD_YABS_SERVER has status {}, resource of BS_RELEASE_TAR is {}'
                    .format(build_task.status, build_task.Parameters.bs_release_tar_resource))
        if build_task.status != Status.SUCCESS or not build_task.Parameters.bs_release_tar_resource:
            self.report_failure_build(st_helper, build_task)
            error = 'Build task failed' if build_task.status != Status.SUCCESS else 'Can not find resource with BS_RELEASE_TAR'
            error += ', {}'.format(get_task_link(self.Context.build_subtask))
            raise TaskFailure(error)
        resource = build_task.Parameters.bs_release_tar_resource
        self.Parameters.bs_release_tar_resource = resource
        deploy_task_id = self._create_deploy_task_draft(resource.id)
        self.report_successful_build(st_helper, build_task, deploy_task_id, resource)

    def get_issue_url(self, issue_key):
        return 'https://st.yandex-team.ru/%s' % issue_key

    def on_execute(self):
        with self.memoize_stage.create_issue_stage:
            st_helper = StartrekHelper(self._create_st_client(), self.Parameters.issue_queue)
            issue = st_helper.create_issue(self.author, self.author, self.get_issue_summary(), self.get_issue_description())
            if not issue:
                raise TaskFailure("Can't create issue, check the logs")
            self.Parameters.issue = self.get_issue_url(issue.key)
            self.set_info(html_hyperlink(link=self.get_issue_url(issue.key), text='See created issue'), do_escape=False)
            self.Context.issue_key = issue.key
        with self.memoize_stage.build_stage:
            self.Context.build_subtask = self.build_server().id
            logger.info('Waiting for build')
            raise sdk2.WaitTask(self.Context.build_subtask,
                                Status.Group.FINISH + Status.Group.BREAK,
                                wait_all=True)
        with self.memoize_stage.check_build_result:
            self.send_build_result_message()
