# -*- coding: utf-8 -*-

import logging

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.task import Status
from sandbox.common.urls import get_task_link
from sandbox.projects.release_machine.core.const import ReleaseStatus
from sandbox.projects.yabs.YabsServerNewRuntime.utils.check_requirements import check_new_runtime_requirements
from sandbox.projects.yabs.YabsServerNewRuntime.utils.st_helper import StartrekHelper
from sandbox.projects.yabs.qa.resource_types import BS_RELEASE_TAR
from sandbox.projects.yabs.release.tasks.CreateNannySnapshots import FilterType

logger = logging.getLogger('YabsServerDeployOnGroups')
logging.basicConfig(level=logging.DEBUG)


class YabsNewRuntimeDeploy(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        disk_space = 2048
        environments = [sdk2.environments.PipEnvironment('startrek_client')]

    class Parameters(sdk2.Parameters):
        issue = sdk2.parameters.String("Issue, created for deployment", required=True)
        st_token = sdk2.parameters.String("ST token vault name", default="st_token", required=True)
        with sdk2.parameters.Group("Release info") as release_info:
            major_version = sdk2.parameters.Integer("Major")
            minor_version = sdk2.parameters.Integer("Minor")

        with sdk2.parameters.Group("Updates") as updates:
            resource_with_server = sdk2.parameters.Resource('Resource with yabs-server', resource_type=BS_RELEASE_TAR,
                                                            required=True)
            comment = sdk2.parameters.String("Snapshot comment")

        with sdk2.parameters.Group("Nanny deployment parameters") as services_filter:
            recipe_name = sdk2.parameters.String("Recipie to deploy", required=True,
                                                 default='bsfront_universal_by_segment.yaml')
            dashboard_id = sdk2.parameters.String("Dashboard identifier", required=True, default='bsfront_production')
            service_ids = sdk2.parameters.List("List of Nanny services to update",
                                               default=['preproduction_yabs_frontend_server_bs_dev1_' + suffix for suffix in ['meta', 'stat']])
            nanny_oauth_token_name = sdk2.parameters.String("Nanny OAuth token", default='nanny_token',
                                                            description="If empty, token for robot-srch-releaser will be used")

        with sdk2.parameters.Output:
            created_snapshots = sdk2.parameters.Dict("Created snapshots")

    def create_snapshot(self):
        subtask_parameters = {
            'component_name': 'yabs_server',
            'release_type': ReleaseStatus.stable,
            'comment': self.Parameters.comment,
            'sandbox_files_dict': {'yabs-server-bundle.tar.gz': self.Parameters.resource_with_server.id},
            'static_files_dict': {},
            'filter_type': FilterType.service_id.value,
            'dashboard_id': self.Parameters.dashboard_id,
            'service_ids': self.Parameters.service_ids,
            'nanny_oauth_token_name': self.Parameters.nanny_oauth_token_name
        }
        logger.debug('Running subtask with parameters: %s', subtask_parameters)
        snapshot_task = sdk2.Task['CREATE_NANNY_SNAPSHOTS'](
            self,
            description='Create snapshot with yabs-server[BS_RELEASE_TAR={resource_id}] for {services}'.format(
                resource_id=self.Parameters.resource_with_server.id,
                services=self.Parameters.service_ids
            ),
            hints=list(self.hints),
            **subtask_parameters
        ).enqueue()
        logger.info('Created subtask (CREATE_NANNY_SNAPSHOTS): {}'.format(snapshot_task))
        return snapshot_task

    def check_snapshot_task_result(self, snapshot_task_id):
        snapshot_task = sdk2.Task[snapshot_task_id]
        logger.info('Obtaining snapshots from task[{}] => {}'.format(snapshot_task_id, snapshot_task))
        if not snapshot_task:
            return 'Can not get task with snapshot {}'.format(snapshot_task_id)
        if snapshot_task.status != Status.SUCCESS:
            return 'Task with snapshots was not succeeded {}.Status={}. See (({} details))'\
                .format(snapshot_task, snapshot_task.status, get_task_link(snapshot_task_id))
        logger.info('Prepared snapshots: {}'.format(snapshot_task.Parameters.created_snapshots))
        return None

    def check_deploy_task_result(self, depoy_task_id):
        deploy_task = sdk2.Task[depoy_task_id]
        logger.info('Obtaining deploy results from task[{}] => {}'.format(depoy_task_id, deploy_task))
        if not deploy_task:
            return 'Can not get deploy task {}'.format(depoy_task_id)
        if deploy_task.status != Status.SUCCESS:
            return 'Deploy task was not succeeded {}.Status={}. See (({} details))' \
                .format(deploy_task, deploy_task.status, get_task_link(depoy_task_id))
        return None

    def deploy_prepared_snapshots(self, snapshot_task):
        subtask_parameters = {
            'snapshots_to_deploy': snapshot_task.Parameters.created_snapshots,
            'component_name': 'yabs_server',
            'release_type': ReleaseStatus.stable,
            'major_version': self.Parameters.major_version,
            'minor_version': self.Parameters.minor_version,
            'recipe_name': self.Parameters.recipe_name,
            'dashboard_id': self.Parameters.dashboard_id,
            'nanny_oauth_token_name': self.Parameters.nanny_oauth_token_name,
            'notify': False,
        }
        logger.debug('Running subtask with parameters: %s', subtask_parameters)
        deploy_task = sdk2.Task['YABS_SERVER_DEPLOY_NANNY_SNAPSHOTS'](
            self,
            description='Deploy snapshots with yabs-server[BS_RELEASE_TAR={resource_id}] for {services}'.format(
                resource_id=self.Parameters.resource_with_server.id,
                services=self.Parameters.service_ids
            ),
            hints=list(self.hints),
            **subtask_parameters
        ).enqueue()
        logger.info('Created subtask (YABS_SERVER_DEPLOY_NANNY_SNAPSHOTS): {}'.format(deploy_task))
        return deploy_task

    def _create_st_client(self):
        """
        Creates and returns Startrek client using self.author (from parent sdk2.Task) and self.Parameters.st_token
        """
        from startrek_client import Startrek

        token = sdk2.Vault.data(self.author, self.Parameters.st_token)
        return Startrek(useragent='YabsServerDeployOnGroups task', token=token)

    def send_comment(self, message):
        st_helper = StartrekHelper(self._create_st_client())
        st_helper.comment(self.Parameters.issue, message)

    def comment_and_fail(self, message):
        self.send_comment(message)
        raise TaskFailure(message)

    def on_execute(self):
        with self.memoize_stage.check_requirements:
            is_legit, errors = check_new_runtime_requirements(self._create_st_client(), self.Parameters.issue, logger)
            if not is_legit:
                error_message = '\n'.join(['* {}'.format(e) for e in errors])
                formatted_message = 'Not all requirements are met:\n\n{}'.format(error_message)
                self.comment_and_fail(formatted_message)
        with self.memoize_stage.create_snapshot:
            snapshot_task = self.create_snapshot()
            logger.debug('Saving snapshot_task.id ({}) into Context'.format(snapshot_task.id))
            self.Context.snapshot_task = snapshot_task.id
            raise sdk2.WaitTask(snapshot_task,
                                Status.Group.FINISH + Status.Group.BREAK,
                                wait_all=True)
        with self.memoize_stage.deploy_stage:
            logger.debug('Start deploy stage')
            error = self.check_snapshot_task_result(self.Context.snapshot_task)
            if error:
                self.comment_and_fail(error)
            snapshot_task = sdk2.Task[self.Context.snapshot_task]
            self.Parameters.created_snapshots = snapshot_task.Parameters.created_snapshots
            deploy_task = self.deploy_prepared_snapshots(snapshot_task)
            self.Context.deploy_task = deploy_task.id
            raise sdk2.WaitTask(deploy_task,
                                Status.Group.FINISH + Status.Group.BREAK,
                                wait_all=True)
        with self.memoize_stage.check_deploy_task_result:
            error = self.check_deploy_task_result(self.Context.deploy_task)
            if error:
                self.comment_and_fail(error)
            self.send_comment('Deployment successfully scheduled: see (({task_link} deploy task))'.format(
                task_link=get_task_link(self.Context.deploy_task)))
