import logging

from sandbox import sdk2

from sandbox.common.errors import Wait, TemporaryError, TaskFailure
from sandbox.common.utils import get_task_link
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sandboxsdk.parameters import (
    ResourceSelector,
    SandboxStringParameter,
    SandboxIntegerParameter,
    SandboxBoolParameter,
)
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.projects.common.utils import get_or_default
from sandbox.projects.common.yabs.server.db.task.cs import (
    InputSpec as RequiredInputSpec,
    MySQLArchiveContents as RequiredMySQLArchiveContents,
    SettingsArchive,
    CSSettingsPatch,
)
from sandbox.projects.common.yabs.server.util import send_startrek_report
from sandbox.projects.yabs.qa.utils import task_run_type
from sandbox.projects.yabs.qa.utils.arcadia import get_arcadia_file_extension, get_arcanum_url
from sandbox.projects.yabs.qa.utils.general import html_hyperlink, startrek_hyperlink

from sandbox.projects.yabs.qa.spec.constants import META_MODES
from sandbox.projects.yabs.qa.tasks.YabsServerTestOneShot.stages import (
    get_spec_data,
    init_cs_settings,
    launch_ammo_generation_tasks,
    launch_print_oneshot_tables_task,
    get_oneshot_bases_and_tables,
    launch_cs_import_test_stability_task,
    launch_base_producing_tasks,
    get_binary_bases,
    launch_chkdb_task,
    launch_chkdb_cmp_task,
    get_ammo_overrides,
    check_base_changes_after_oneshot_application,
    create_report,
)
from sandbox.projects.yabs.qa.pipeline_test_framework.stages import (
    get_shard_map,
    launch_ft_shoot_tasks,
    launch_ft_validation_tasks,
    launch_ft_stability_tasks,
    launch_stat_load_shoot_tasks,
    launch_meta_load_shoot_tasks,
    launch_ft_cmp_tasks,
    launch_ft_validation_cmp_tasks,
    launch_stat_load_cmp_tasks,
    launch_meta_load_cmp_tasks,
)

from .spec import CUSTOM_REQUEST_SPEC


GROUP_TOP = 'SPEC and oneshots'
LOAD_SPEC_PARAM = 'local_rps_dict'
PRE_SHOOT_KEY = 'pre_local_func_shoot_task_id'
PRE_VALIDATION_KEY = 'pre_local_validation_task_id'
GET_RESOURCES_ERRORS = (ValueError, IndexError, KeyError)
STABILITY_RUNS = 4


class CreateSpec(SandboxBoolParameter):
    name = 'create_spec'
    description = 'create new YABS_SERVER_HOUSE_SPEC'
    group = GROUP_TOP


class ReleaseSpec(SandboxBoolParameter):
    name = 'release_spec'
    description = 'release new YABS_SERVER_HOUSE_SPEC. DO NOT TOUCH, IF YOU ARE NOT SURE'
    default_value = False
    do_not_copy = True


class SpecResourceID(ResourceSelector):
    name = 'SpecResource'
    description = 'YABS_SERVER_HOUSE_SPEC (overwrites other params)'
    group = GROUP_TOP
    do_not_copy = True


class OneShotPath(SandboxStringParameter):
    name = 'oneshot_path'
    description = 'oneshot_path (arcadia path to oneshot)'
    group = GROUP_TOP


class OneShotArgs(SandboxStringParameter):
    name = 'oneshot_args'
    description = 'oneshot_args (arguments for oneshot)'
    group = GROUP_TOP


class YTOneshot(SandboxBoolParameter):
    name = 'is_yt_oneshot'
    description = 'Apply oneshot to YT'
    default_value = False
    group = GROUP_TOP


class SettingsChange(SandboxBoolParameter):
    name = 'is_settings_change'
    description = 'Test cs settings change'
    default_value = False
    group = GROUP_TOP


class OneShotBases(SandboxStringParameter):
    name = 'oneshot_bases'
    description = 'oneshot_bases (binary bases affected by oneshot, comma-separated)'
    group = GROUP_TOP
    multiline = True


class OneShotTables(SandboxStringParameter):
    name = 'oneshot_tables'
    description = 'oneshot_tables (tables found in oneshot SQL query, comma-separated)'
    group = GROUP_TOP


class OneShotHosts(SandboxStringParameter):
    name = 'oneshot_hosts'
    description = 'oneshot_hosts (comma-separated, as in bsint GUI)'
    group = GROUP_TOP
    default_value = 'bsdb'


class OneShotTicket(SandboxStringParameter):
    name = 'oneshot_ticket'
    description = 'oneshot_ticket (startrek ticket id)'
    group = GROUP_TOP


class StartrekTokenVaultName(SandboxStringParameter):
    name = 'vault_name'
    default_value = 'robot-yabs-cs-b2b-startrek-token'
    description = 'Startrek token vault name to send comment'
    group = GROUP_TOP


class CSSettings(SandboxStringParameter):
    name = 'cs_settings'
    description = 'cs_settings The URL or JSON with CS Settings (will be passed to --settings-spec)'
    group = GROUP_TOP
    required = False


GROUP_SHOOTING_OPTIONS = 'Custom shooting options (roles and requests from YQL)'


def _create_shooting_param_for_role(custom_role):
    class EnableCustomRoleShooting(SandboxBoolParameter):
        name = '{role}_role_shooting'.format(role=custom_role)
        description = 'Enable {role} role shooting'.format(role=custom_role)
        default_value = True

    return EnableCustomRoleShooting


PARAMS_FOR_ENABLE_SHOOTING_ROLE = {meta_mode: _create_shooting_param_for_role(meta_mode) for meta_mode in META_MODES}


class CustomRequest(SandboxStringParameter):
    """
    Example:
    SELECT GraphID as requestID
    FROM {//logs/bs-proto-accessstat-log/1h}
    WHERE Request like "GET /code/165%";
    """
    group = GROUP_SHOOTING_OPTIONS
    required = False
    multiline = True


class CustomRequestYabs(CustomRequest):
    name = 'custom_request_yabs'
    description = 'Custom request yabs'


class CustomRequestBs(CustomRequest):
    name = 'custom_request_bs'
    description = 'Custom request bs'


class CustomRequestBsrank(CustomRequest):
    name = 'custom_request_bsrank'
    description = 'Custom request bsrank'


class DaysInterval(SandboxIntegerParameter):
    name = 'days_interval'
    description = "Interval in days to get requests"
    default_value = '1'
    group = GROUP_SHOOTING_OPTIONS


class LogsInterval(SandboxStringParameter):
    name = 'logs_interval'
    description = "Logs interval: 1d or 1h"
    default_value = "1h"
    group = GROUP_SHOOTING_OPTIONS


class CacheDaemon(SandboxIntegerParameter):
    name = 'cache_daemon'
    description = 'CACHE_DAEMON (default_value = 845513904)'
    default_value = '845513904'
    group = GROUP_SHOOTING_OPTIONS


GROUP_CHILDREN = 'Children'


class LaunchLoad(SandboxBoolParameter):
    name = 'launch_load'
    description = 'Launch load subtasks'
    default_value = False
    group = GROUP_CHILDREN


class LaunchMetaLoad(SandboxBoolParameter):
    name = 'launch_meta_load'
    description = 'Launch meta load subtasks'
    default_value = False
    group = GROUP_CHILDREN


GROUP_TESTENV_GLOBALS = 'TestEnv global resources'


class BuildFlagsResID(ResourceSelector):
    name = 'BuildFlags'
    description = 'YABS_SERVER_BUILD_FLAGS'
    group = GROUP_TESTENV_GLOBALS


class ShardMapID(ResourceSelector):
    name = 'ShardMap'
    description = 'YABS_SERVER_TESTENV_SHARD_MAP'
    group = GROUP_TESTENV_GLOBALS


class GenBinBasesFlagsID(ResourceSelector):
    name = 'GenBinBasesFlags'
    description = 'YABS_SERVER_GEN_BIN_BASES_FLAGS'
    group = GROUP_TESTENV_GLOBALS


GROUP_BUILD_BANNER_SERVER = 'Settings for BUILD_BANNER_SERVER (new server is built only in "create new YABS_SERVER_HOUSE_SPEC" mode)'


class FindReleased(SandboxBoolParameter):
    name = 'find_released'
    description = 'find recent BUILD_BANNER_SERVER resource (READY, released=stable)'
    default_value = False
    group = GROUP_BUILD_BANNER_SERVER


class CheckBinaryBasesChanges(SandboxBoolParameter):
    name = 'check_binary_bases_changes'
    description = 'Checking changes in binary bases after oneshot application'
    default_value = True
    group = GROUP_TOP


class RunImportTestStability(SandboxBoolParameter):
    name = 'run_import_test_stability'
    description = 'Run cs import many times and compare result identity'
    default_value = False
    group = GROUP_TOP


class MySQLArchiveContents(RequiredMySQLArchiveContents):
    required = False


class InputSpec(RequiredInputSpec):
    required = False


class ReusableStages(sdk2.parameters.CheckGroup):
    name = 'reusable_stages'
    default_value = []
    choices = [
        (s, s)
        for s in [
            'get_spec_data',
            'get_shard_map',
            'init_cs_settings',
            'launch_print_oneshot_tables_task',
            'launch_ammo_generation_tasks',
            'get_oneshot_bases_and_tables',
            'launch_cs_import_test_stability_task',
            'launch_base_producing_tasks',
            'get_binary_bases',
            'launch_chkdb_task',
            'launch_chkdb_cmp_task',
            'launch_ft_shoot_tasks',
            'launch_stat_load_shoot_tasks',
            'launch_meta_load_shoot_tasks',
            'launch_ft_stability_tasks',
            'launch_ft_validation_tasks',
            'launch_ft_cmp_tasks',
            'launch_ft_validation_cmp_tasks',
            'launch_stat_load_cmp_tasks',
            'launch_meta_load_cmp_tasks',
        ]
    ]


class YabsServerOneShot(SandboxTask):
    type = 'YABS_SERVER_ONE_SHOT'
    description = ''
    environment = [
        PipEnvironment('retrying'),
        PipEnvironment('jsondiff', version='1.2.0'),
        PipEnvironment('startrek_client'),
    ]

    input_parameters = [
        BuildFlagsResID, ShardMapID, GenBinBasesFlagsID,
        CreateSpec, SpecResourceID, OneShotPath, OneShotArgs, YTOneshot, SettingsChange, OneShotBases, OneShotTables,
        OneShotHosts, OneShotTicket, StartrekTokenVaultName, LaunchLoad, LaunchMetaLoad, FindReleased,
        CSSettings, SettingsArchive, CSSettingsPatch, ReleaseSpec, CheckBinaryBasesChanges, InputSpec, RunImportTestStability, MySQLArchiveContents, ReusableStages] + \
        PARAMS_FOR_ENABLE_SHOOTING_ROLE.values() + \
        [CustomRequestYabs, CustomRequestBs, CustomRequestBsrank, DaysInterval, LogsInterval]

    def _get_current_roles(self):
        return [meta_mode
                for meta_mode in META_MODES
                if get_or_default(self.ctx, PARAMS_FOR_ENABLE_SHOOTING_ROLE[meta_mode])]

    def check_input_parameters(self):
        if not self._get_current_roles():
            raise TaskFailure("Please select at least one role to be shot.")

        if self.ctx.get('oneshot_path'):
            oneshot_type = get_arcadia_file_extension(self.ctx['oneshot_path'])
            if oneshot_type not in {'.sql', '.pl', '.py'}:
                raise SandboxTaskFailureError(
                    "Unsupported oneshot type: \"{oneshot_type}\". Only SQL, Perl and Python oneshots are supported"
                    .format(oneshot_type=oneshot_type)
                )

        if not any((self.ctx.get('is_yt_oneshot'),
                    self.ctx.get('is_settings_change'))) and \
           not all((self.ctx.get('oneshot_bases'),
                    self.ctx.get('oneshot_tables'),
                    self.ctx.get('oneshot_hosts'))):
            raise TaskFailure("Please, fill the fields oneshot_bases, oneshot_tables and oneshot_hosts")

    def custom_request_flag(self):
        return any([self.ctx.get('custom_request_{}'.format(role)) for role in self._get_current_roles()])

    def is_oneshot_test(self):
        return bool(self.ctx.get('oneshot_path'))

    def is_content_system_settings_change_test(self):
        return bool(self.ctx.get(SettingsChange.name))

    def on_execute(self):
        try:
            self._on_execute()
        except (Wait, TemporaryError):
            raise
        except BaseException as err:
            logging.exception(err.message)
            try:
                self._report_error_to_startrek(err)
            except Exception:
                logging.warning("Failed to report error to Startrek", exc_info=True)
                self.set_info("Failed to report error to Startrek")
            raise err

    def add_tag(self, tag):
        self.tags = list(set(self.tags + [tag]))

    def _on_execute(self):
        self.check_input_parameters()

        with self.memoize_stage.first_run(commit_on_entrance=False):
            self.ctx['__reusable_stages'] = self.ctx.get(ReusableStages.name, [])
            if self.ctx.get('oneshot_path'):
                oneshot_arcanum_url = get_arcanum_url(self.ctx['oneshot_path'])
                self.set_info(
                    'Oneshot: {}'.format(html_hyperlink(link=oneshot_arcanum_url, text=oneshot_arcanum_url)),
                    do_escape=False
                )

            if self.ctx.get('oneshot_ticket'):
                self.set_info(
                    'Startrek: <a href="https://st.yandex-team.ru/{ticket}">{ticket}</a>'
                    .format(ticket=self.ctx.get('oneshot_ticket')),
                    do_escape=False
                )

            if self.is_oneshot_test():
                self.add_tag(task_run_type.SANDBOX_TASK_TAGS['oneshot_test'])
                self.ctx['_child_description'] = 'oneshot ' + self.ctx['oneshot_ticket']
            elif self.is_content_system_settings_change_test():
                self.add_tag(task_run_type.SANDBOX_TASK_TAGS['content_system_settings_change_test'])
                self.ctx['_child_description'] = 'cs settings change ' + self.ctx['oneshot_ticket']
            else:
                self.add_tag(task_run_type.SANDBOX_TASK_TAGS['create_oneshot_spec'])
                self.ctx['_child_description'] = 'create oneshot spec'

            if self.custom_request_flag() and self.ctx.get('launch_load'):
                self.ctx['launch_load'] = False
                self.set_info("Stat load tasks are disabled because custom requests are set")

        meta_modes = self._get_current_roles()

        # New-style stages

        get_spec_data(self, spec_resource_id=self.ctx.get(SpecResourceID.name), stat_load=self.ctx[LaunchLoad.name], meta_load=self.ctx[LaunchMetaLoad.name])
        get_shard_map(self, ft_shards=('A', ), stat_load_shards=('A', 'B'), meta_load_shards=('A', 'B'), stat_load=self.ctx[LaunchLoad.name], meta_load=self.ctx[LaunchMetaLoad.name])

        init_cs_settings(self, cs_settings=self.ctx[CSSettings.name])
        launch_ammo_generation_tasks(
            self,
            ammo_generation_config={
                meta_mode: self.ctx.get('custom_request_{}'.format(meta_mode))
                for meta_mode in meta_modes
            },
            test_description=self.ctx['_child_description'],
            cache_daemon_resource=get_or_default(self.ctx, CacheDaemon),
            days_interval=self.ctx[DaysInterval.name],
            logs_interval=self.ctx[LogsInterval.name],
            custom_request_spec=CUSTOM_REQUEST_SPEC,
        )

        launch_print_oneshot_tables_task(self, oneshot_path=self.ctx[OneShotPath.name], is_yt_oneshot=self.ctx[YTOneshot.name])
        get_oneshot_bases_and_tables(
            self,
            is_yt_oneshot=self.ctx[YTOneshot.name],
            oneshot_bases=self.ctx[OneShotBases.name],
            oneshot_tables=self.ctx[OneShotTables.name],
            cs_settings_patch_resource_id=self.ctx[CSSettingsPatch.name],
        )

        launch_cs_import_test_stability_task(
            self,
            run_import_test_stability=self.ctx[RunImportTestStability.name],
            cs_settings_patch_resource_id=self.ctx[CSSettingsPatch.name],
        )

        launch_base_producing_tasks(
            self,
            cs_settings_patch_resource_id=self.ctx[CSSettingsPatch.name],
            is_content_system_settings_change_test=self.is_content_system_settings_change_test(),
            is_yt_oneshot=self.ctx[YTOneshot.name],
            oneshot_path=self.ctx[OneShotPath.name],
            oneshot_args=self.ctx[OneShotArgs.name],
            oneshot_hosts=self.ctx[OneShotHosts.name],
        )
        get_binary_bases(self)

        launch_chkdb_task(self, test_description=self.ctx['_child_description'])
        launch_chkdb_cmp_task(self, test_description=self.ctx['_child_description'])
        check_base_changes_after_oneshot_application(self, check_binary_bases_changes=self.ctx[CheckBinaryBasesChanges.name])

        ammo_overrides_map = get_ammo_overrides(self)
        shoot_baseline_tasks = bool(ammo_overrides_map)

        launch_ft_shoot_tasks(self, test_description=self.ctx['_child_description'], meta_modes=meta_modes, ammo_overrides=ammo_overrides_map, shoot_baseline_tasks=shoot_baseline_tasks)
        launch_stat_load_shoot_tasks(self, test_description=self.ctx['_child_description'], stat_load=self.ctx[LaunchLoad.name], meta_modes=meta_modes)
        launch_meta_load_shoot_tasks(self, test_description=self.ctx['_child_description'], meta_load=self.ctx[LaunchMetaLoad.name], meta_modes=meta_modes)
        launch_ft_stability_tasks(self, test_description=self.ctx['_child_description'], meta_modes=meta_modes, ammo_overrides=ammo_overrides_map)
        launch_ft_validation_tasks(self, test_description=self.ctx['_child_description'], shoot_baseline_tasks=shoot_baseline_tasks)

        launch_ft_cmp_tasks(self, test_description=self.ctx['_child_description'])
        launch_stat_load_cmp_tasks(self, test_description=self.ctx['_child_description'], stat_load=self.ctx[LaunchLoad.name])
        launch_meta_load_cmp_tasks(self, test_description=self.ctx['_child_description'], meta_load=self.ctx[LaunchMetaLoad.name])
        launch_ft_validation_cmp_tasks(self, test_description=self.ctx['_child_description'])

        html_report, startrek_report = create_report(self)

        self.set_info(html_report, do_escape=False)
        self._send_startrek_report(startrek_report)

    def _send_startrek_report(self, st_report):
        ticket = self.ctx.get(OneShotTicket.name)
        if ticket:
            startrek_token = self.get_vault_data(self.ctx.get('vault_name'))
            send_startrek_report(startrek_token, ticket, st_report, self.id)

    def _report_error_to_startrek(self, err):
        task_url = get_task_link(self.id)
        if isinstance(err, TaskFailure):
            fmt = "Oneshot task {task_link} FAILED:\n{msg}"
        else:
            fmt = "EXCEPTION in the oneshot task {task_link}:\n{msg}"
        report = fmt.format(
            task_link=startrek_hyperlink(task_url, self.id),
            msg=err,
        )
        self._send_startrek_report(report)
