#!/usr/bin/env python
# encoding=utf-8

from sandbox import sdk2
from datetime import datetime, timedelta
from sandbox.sandboxsdk import environments
from email.header import Header
from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText
from email.mime.base import MIMEBase
import re
import random
import logging
import copy
import json
import hashlib
import struct
import time
import requests
import smtplib


class YabsServerSSPPagesModeration(sdk2.Task):

    result_stat = {
        "AffectedApps": 0,
        "AffectedHits": 0,
        "AffectedCost": 0,
        "AffectedAppsBan": 0,
        "AffectedAppsUnban": 0
    }

    email_notification_texts_array = []
    email_notification_intersection = []

    start_process_time = 0

    class Parameters(sdk2.Task.Parameters):
        yql_token_owner = sdk2.parameters.String("YQL Token Owner Name", default="robot-yabs-google", required=True)
        yql_token_vault_name = sdk2.parameters.String("YQL Robot Token Vault Name", default="robot_google_yql_token", required=True)

        solomon_key = sdk2.parameters.YavSecret("Solomon Key", default="sec-01egr7r5qh453d0tq0q4bht99p", required=True)

        sspid_list = sdk2.parameters.String("List SSPID", default="", required=True)

        yt_logging_cluster = sdk2.parameters.String("Logging YT Cluster ", default="hahn", required=True)
        yt_log_table = sdk2.parameters.String("Log table", default="hahn", required=True)

        yt_stat_cluster = sdk2.parameters.String("Stat YT Cluster", default='hahn', required=True)
        table_log_final_stat = sdk2.parameters.String("Intermediate working table for stat by PageToken", default='', required=True)

        table_appid_to_bundle = sdk2.parameters.String("Table with associations appid <=> bundle", default='//home/yabs-rt/mirage/tmp-appid_to_bundle', required=False)
        debug_refresh_appid_mobile_bundle_table = sdk2.parameters.Bool("[Debug] Refresh convert appid to bundle table", default=False, required=False)

        yt_apply_cluster_finish_table = sdk2.parameters.String("Apply YT Cluster (finish table)", default='markov', required=True)
        yt_read_cluster_finish_table = sdk2.parameters.String("Readonly YT Cluster (finish table)", default='hahn', required=True)
        yt_finish_table_name = sdk2.parameters.String("Apply table", default='hahn', required=True)

        solomon_project = sdk2.parameters.String("Solomon Project", default="yabs_debug", required=True)
        solomon_cluster = sdk2.parameters.String("Solomon Cluster", default="yabs_server_frontend_debug", required=True)
        solomon_service = sdk2.parameters.String("Solomon Service", default="sys", required=True)

        custom_exceptions = sdk2.parameters.String("Exceptions for specified pages (<SSPID>:<PageToken>:<ShowProbability>)", default="", required=False)

        execute_ban_action = sdk2.parameters.Bool("Do ban action", default=True, required=True)
        execute_unban_action = sdk2.parameters.Bool("Do unban action", default=False, required=True)
        execute_update_custom_exceptions_only = sdk2.parameters.Bool("Update custom exceptions only", default=False, required=True)
        execute_ssp_sdk_intersection = sdk2.parameters.Bool("Do monitor intersection ssp/sdk", default=False, required=False)

        reasons_list = sdk2.parameters.String("Reasons list to check", default="ctr_high,cpm_high,bids_rate,win_rate,impressions_rate,sspprice_more_than_cost,sspprice_more_than_bid", required=True)

        rows_limit_to_apply_ban = sdk2.parameters.Integer("[Ban] Rows limit to apply", default=1000, required=False)
        rows_limit_to_apply_unban = sdk2.parameters.Integer("[Unban] Rows limit to apply", default=1000, required=False)
        delta_show_probability_unban = sdk2.parameters.Integer("[Unban] ShowProbability delta for unban (M)", default=50000, required=False)
        candidates_limit = sdk2.parameters.Integer("Candiates limit to process", default=1000, required=False)

        hits_reduction_limit = sdk2.parameters.Integer("Hits reduction limit", default=1000000, required=False)
        daily_money_loss_limit = sdk2.parameters.Integer("Daily money loss limit (after dividing on 1M)", default=100, required=False)

        # sort_method = sdk2.parameters.String("Sort method before slice (random/shows_probability)", default="show_probability", required=False)
        with sdk2.parameters.RadioGroup('Sort method before slice (random/shows_probability)', required=True) as sort_method:
            sort_method.values['random'] = sort_method.Value(value='random')
            sort_method.values['shows_probability'] = sort_method.Value(value='shows_probability', default=True)

        skip_generate_new_data = sdk2.parameters.Bool("[Manual] Do not process logs (reuse preaggregated data)", default=False, required=False)
        debug_fixed_timestamp = sdk2.parameters.Integer("[Manual] Fix timestamp", default=0, required=False)
        debug_allow_fix_pagetoken_md5 = sdk2.parameters.Bool("[Manual] Fix incorrect PageTokenMD5", default=False, required=True)
        debug_real_ban_intersection_ssp_sdk = sdk2.parameters.Bool("[DEBUG] Real ban candidates intersection SSP-SDK", default=False, required=False)

        email_notifications = sdk2.parameters.String("Send email notifications to:", default='mirage@yandex-team.ru', required=True)

        custom_ban_condition = sdk2.parameters.String("Custom YQL ban condition", default="", required=False)
        custom_ban_condition_name = sdk2.parameters.String("Custom YQL ban condition name", default="", required=False)
        custom_ban_condition_abs_value = sdk2.parameters.Integer("Custom YQL condition prob value", default="10000", required=False)

        custom_ban_intersection_ssp_sdk_abs_value = sdk2.parameters.Integer("Custom YQL condition prob value", default="10000", required=False)

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yql', version='1.2.91')
        )

    def md5unit64(self, string):
        digest = hashlib.md5(string.encode('utf-8')).digest()
        ar = struct.unpack(">IIII", digest)
        hig = ar[1] ^ ar[3]
        low = ar[0] ^ ar[2]
        return (hig << 32 | low)

    def kill_current_query(self):
        logging.info("Killing current query")
        if self.current_query:
            if self.current_query.abort():
                print('Operation %s aborted successfully, status: %s' % (
                    self.current_query.operation_id,
                    self.current_query.status
                ))
            else:
                print('Failed to abort operation %s' % self.current_query.operation_id)
        else:
            logging.info("No query to kill")

    def on_terminate(self):
        self.kill_current_query()

    def on_before_timeout(self, seconds):
        self.kill_current_query() 

    def yt_connect(self, yt_cluster):
        import yt.wrapper as yt

        logging.info("Try connect to " + str(yt_cluster))
        try:
            cfg = {
                "tabular_data_format": yt.JsonFormat(control_attributes_mode="row_fields"),
                "detached": False,
                "token": self.yql_token,
                "proxy": {"url": yt_cluster},

            }
            ytc = yt.YtClient(config=cfg)
            logging.info("Succesfully connected to {}".format(yt_cluster))

            return ytc
        except:
            logging.info("Can`t connect to {}".format(yt_cluster))
            return None

    def exec_yql_query(self, db, token, query):
        from yql.api.v1.client import YqlClient

        query = re.sub(r'\\', '', query)

        client = YqlClient(db=db, token=token)
        
        self.current_query = client.query(query, syntax_version=1)
        self.current_query.run()
        self.current_query.wait_progress()

        query = self.current_query
        self.current_query = None

        if not query.is_success:
            errors = []
            if query.errors:
                errors = [str(error) for error in query.errors]
            raise Exception("YQL request failed. Status: {}. Errors: {}".format(query.status, ' - '.join(errors)))

        result = []
        for table in query.get_results():
            table.fetch_full_data()

            columns = []
            for column_name, column_type in table.columns:
                columns.append(column_name)

            for row in table.rows:
                result.append(dict([(columns[i], value) for i, value in enumerate(row)]))

        return json.loads(json.dumps(result))

    def read_yt_table(self, yt_cluster, query):
        logging.info("Start reading yt table")

        yt_res = self.exec_yql_query(yt_cluster, self.yql_token, query)
        logging.info("OUT: {}\n".format(json.dumps(yt_res)))

        return yt_res

    def get_query_ssp_sdk_intersection(self, day_delta=1):
        date_for_log = self.get_process_time('%Y-%m-%d', day_delta)
        query = '''
	    $sdk_pages = (
		select `PageID`, `Name`
		from `//home/yabs/dict/Page`
		where `OptionsMobile` = true and `OptionsApp` = true and `OptionsSsp` = false
	    );

	    $sdk_event_stat = (
		select
		    cast(event.`pageid` as int64) as pageid,
		    sum_if(1, event.`countertype` == "1") as impressions,
		    sum_if(1, event.`countertype` == "2") as clicks,
		    sum(cast(event.`eventcost` as double)/1000000) as cost
		from `//logs/bs-chevent-log/1d/{date_for_log}` as event
		    inner join $sdk_pages as pages on pages.`PageID` = cast(event.`pageid` as int64)
		group by event.`pageid`
	    );

	    $sdk_dsp_stat = (
		select
		    cast(dsp.`pageid` as int64) as pageid,
		    sum_if(1, dsp.`countertype` == "0") as bids
		from hahn.`//logs/bs-dsp-log/1d/{date_for_log}` as dsp
		    inner join $sdk_pages as pages on pages.`PageID` = cast(dsp.`pageid` as int64)
		group by dsp.`pageid`
	    );

	    $sdk_rtb_stat = (
		select
		    cast(rtb.`pageid` as int64) as pageid,
		    count(*) as hits
		from hahn.`//logs/bs-dsp-log/1d/{date_for_log}` as rtb
		    inner join $sdk_pages as pages on pages.`PageID` = cast(rtb.`pageid` as int64)
		group by rtb.`pageid`
	    );

	    $sdk_stat = (
		select
		    dsp.`pageid` as pageid,
		    pages.`Name` as bundle,
		    dsp.`bids` as bids,
		    event.`impressions` as impressions,
		    event.`clicks` as clicks,
		    event.`cost` as cost,
                    rtb.`hits` as hits
		from
                    $sdk_rtb_stat as rtb
		    left join $sdk_dsp_stat as dsp ON rtb.pageid = dsp.pageid
		    left join $sdk_event_stat as event ON rtb.pageid = event.pageid
		    left join $sdk_pages as pages ON dsp.pageid = pages.PageID
	    );

	    $appid_to_bundle = (
		-- select store, app_id, bundle
		-- from (
		--        select "gplay" as store, app_id, bundle
		--        from seneca-sas.`//home/direct/extdata-mobile/gplay/latest`
	        --	union all
		--        select "itunes" as store, String::ReplaceAll(app_id, "id", "") as app_id, bundle
		--        from senecasas.`//home/direct/extdata-mobile/itunes/latest`
		-- )
		-- group by store, app_id, bundle

		select app_id, bundle, store
		from hahn.`{table_appid_to_bundle}`
                group by app_id, bundle, store
	    );

	    $ssp_stat = (
		SELECT
		    stat_table.`rtb_dsp.sspid` as sspid,
		    stat_table.`rtb_ssp.pagetoken` as pagetoken,
		    stat_table.`rtb_dsp.hits` as hits,
		    cast(stat_table.`rtb_event.cost` as double)/1000000 as cost,
		    stat_table.`rtb_event.impressions` as impressions,
		    stat_table.`rtb_event.clicks` as clicks,
		    stat_table.`rtb_dsp.bids` as bids,
		    if(conv.`bundle` is not null and conv.`bundle` != "", conv.`bundle`, stat_table.`rtb_ssp.pagetoken`) as final_bundle
		FROM
		    `{table_log_final_stat}` as stat_table
		    left join $appid_to_bundle as conv ON conv.`app_id` = stat_table.`rtb_ssp.pagetoken`
		WHERE
		    stat_table.`timestamp` = '{timestamp}'
	    );

            $ssp_sdk_intersection = (
	    select
		sdk_pageid,
		bundle,
		sspid,
		ssp_pagetoken,

                min(ssp.`bids`) as ssp_bids,
		min(sdk.`bids`) as sdk_bids,

                min(ssp.`impressions`) as ssp_impressions,
		min(sdk.`impressions`) as sdk_impressions,

                min(ssp.`clicks`) as ssp_clicks,
		min(sdk.`clicks`) as sdk_clicks,

		min(ssp.`hits`) as ssp_hits,
		min(sdk.`hits`) as sdk_hits,

		min(ssp.`cost`) as ssp_cost,
		min(sdk.`cost`) as sdk_cost
	    from
		$ssp_stat as ssp
		inner join $sdk_stat as sdk on sdk.`bundle` = ssp.`final_bundle`
	    group by
		sdk.`pageid` as sdk_pageid,
		sdk.`bundle` as bundle,
		ssp.`sspid` as sspid,
		ssp.`pagetoken` as ssp_pagetoken
            );

            select *
            from $ssp_sdk_intersection
            where sdk_hits > 1000 and ssp_hits > 1000

        '''.format(timestamp=self.start_process_time, table_log_final_stat=self.Parameters.table_log_final_stat, date_for_log=date_for_log, table_appid_to_bundle=self.Parameters.table_appid_to_bundle)

        return query

    def get_query_pagetoken_statistic(self, day_delta = 1):
        date_for_log = self.get_process_time('%Y-%m-%d', day_delta)
        query = '''
    PRAGMA AllowDotInAlias;

    \$rtb_bidreqid_slice = (
        SELECT
            bidreqid,
            sspid,
            pageid,
            queryargs,
            pagetoken
        FROM
            `//logs/bs-rtb-log/1d/{log_date}` as rtb
        WHERE
            cast(sspid as Int64) in ({sspid_list})
            AND rtb.declinereasons = ""
    );

    \$rtb_dsp_slice = (
            SELECT
                bidreqid_translation.sspid as sspid,
                bidreqid_translation.pagetoken as pagetoken,
                count(DISTINCT dsp.bidreqid) AS hits,
                sum(bids) AS bids,
                sum(won_bids) AS won_bids,
                sum(block_impressions) AS block_impressions
            FROM \$rtb_bidreqid_slice
                AS bidreqid_translation
            LEFT JOIN (
                SELECT
                    bidreqid,
                    sum(if(countertype == '0', 1, 0)) AS bids,
                    sum(if(countertype == '0' AND win == '1', 1, 0)) AS won_bids,
                    sum(if(countertype == '1', 1, 0)) AS block_impressions
                FROM `logs/bs-dsp-log/1d/{log_date}`
                WHERE dspid = '1'
                GROUP BY
                bidreqid
            ) AS dsp
            ON dsp.bidreqid = bidreqid_translation.bidreqid
            GROUP BY
                bidreqid_translation.sspid,
                bidreqid_translation.pagetoken
    );

    \$rtb_ssp_slice = (
            SELECT
                bidreqid_translation.sspid as sspid,
                bidreqid_translation.pagetoken as pagetoken,

                sum(cast(ssp.sspprice as Int64)) as sspprice,
                sum(cast(ssp.partnerprice as Int64)) as partnerprice,

                count(distinct bidreqid_translation.bidreqid) as wins
            FROM
                \$rtb_bidreqid_slice as bidreqid_translation
                LEFT JOIN `logs/bs-ssp-log/1d/{log_date}` as ssp ON bidreqid_translation.bidreqid = ssp.bidreqid
            WHERE
                ssp.win = '1'
            GROUP BY
                bidreqid_translation.sspid,
                bidreqid_translation.pagetoken
    );

    \$rtb_event_slice = (
            SELECT
                bidreqid_translation.sspid as sspid,
                bidreqid_translation.pagetoken as pagetoken,
                sum(chevent.cost) AS cost,
                sum(chevent.impressions) AS impressions,
                sum(chevent.clicks) AS clicks
            FROM \$rtb_bidreqid_slice
                AS bidreqid_translation
            JOIN (
                SELECT
                    rtbbidreqid,
                    sum(CAST(eventcost AS Int64)) AS cost,
                    sum(if(countertype == '1', 1, 0)) AS impressions,
                    sum(if(countertype == '2', 1, 0)) AS clicks
                    FROM `logs/bs-chevent-log/1d/2022-04-27`
                    WHERE fraudbits = '0' AND sspid != '0'
                    GROUP BY
                        rtbbidreqid
            )
                AS chevent
            ON bidreqid_translation.bidreqid = chevent.rtbbidreqid
            GROUP BY
                bidreqid_translation.sspid,
                bidreqid_translation.pagetoken
    );

    INSERT INTO `{table_log_final_stat}`
    SELECT
        '{timestamp}' as `timestamp`,
        rtb_dsp.sspid as `rtb_dsp.sspid`,
        rtb_dsp.pagetoken as `rtb_dsp.pagetoken`,
        COALESCE(rtb_dsp.hits, 0) as `rtb_dsp.hits`,
        COALESCE(rtb_dsp.bids, 0) as `rtb_dsp.bids`,
        COALESCE(rtb_dsp.won_bids, 0) as `rtb_dsp.won_bids`,
        COALESCE(rtb_dsp.block_impressions, 0) as `rtb_dsp.block_impressions`,
        rtb_ssp.sspid as `rtb_ssp.sspid`,
        rtb_ssp.pagetoken as `rtb_ssp.pagetoken`,
        COALESCE(rtb_ssp.sspprice, 0) as `rtb_ssp.sspprice`,
        COALESCE(rtb_ssp.partnerprice, 0) as `rtb_ssp.partnerprice`,
        COALESCE(rtb_ssp.wins, 0) as `rtb_ssp.wins`,
        rtb_event.sspid as `rtb_event.sspid`,
        rtb_event.pagetoken as `rtb_event.pagetoken`,
        COALESCE(rtb_event.cost, 0) as `rtb_event.cost`,
        COALESCE(rtb_event.impressions, 0) as `rtb_event.impressions`,
        COALESCE(rtb_event.clicks, 0) as `rtb_event.clicks`
    FROM
        \$rtb_dsp_slice as rtb_dsp
        LEFT JOIN \$rtb_ssp_slice as rtb_ssp ON rtb_dsp.sspid = rtb_ssp.sspid AND rtb_dsp.pagetoken = rtb_ssp.pagetoken
        LEFT JOIN \$rtb_event_slice as rtb_event ON rtb_dsp.sspid = rtb_event.sspid AND rtb_dsp.pagetoken = rtb_event.pagetoken;

        '''.format(log_date = date_for_log, table_log_final_stat = self.Parameters.table_log_final_stat, timestamp = self.start_process_time, sspid_list = self.Parameters.sspid_list)

        return query

    MIN_HITS_LEVEL = 1000
    MIN_WINS_LEVEL = 100
    MIN_BIDS_LEVEL = 1000
    MIN_CLICKS_LEVEL = 10
    MIN_IMPRESSIONS_LEVEL = 100
    MAX_DIFFERENCE_LEVEL_HIT_BID = 1000
    MAX_DIFFERENCE_LEVEL_WIN_IMP = 80
    MAX_DIFFERENCE_LEVEL_BID_WIN = 500
    MAX_ALLOWED_CPM = 50
    MAX_ALLOWED_CTR = 0.2
    MAX_COEFF_DIFF_SSPPRICE = 1
    MIN_HIT_COST_PRICE = 100

    def get_query_candidates_for_throttling(self):

        CUSTOM_CONDITION_SELECT = ''
        CUSTOM_CONDITION_WHERE = ''

        if self.Parameters.custom_ban_condition != "" and self.Parameters.custom_ban_condition_name != "" and self.Parameters.custom_ban_condition_abs_value != "0":
            CUSTOM_CONDITION_SELECT = ', (' + str(self.Parameters.custom_ban_condition) + ') as ' + str(self.Parameters.custom_ban_condition_name)
            CUSTOM_CONDITION_WHERE = 'OR (' + str(self.Parameters.custom_ban_condition) + ')'

        query = '''
    SELECT
        \`rtb_dsp.sspid\` as sspid,
        \`rtb_ssp.pagetoken\` as pagetoken,
        \`rtb_dsp.hits\` as hits,
        \`rtb_event.cost\` as cost,
        (\`rtb_dsp.hits\` > {MIN_HITS_LEVEL} AND \`rtb_dsp.hits\` > {MAX_DIFFERENCE_LEVEL_HIT_BID} * \`rtb_dsp.bids\` AND (cast(`rtb_event.cost` as double)/`rtb_dsp.hits`) < {MIN_HIT_COST_PRICE}) as bids_rate,
        (\`rtb_ssp.wins\` > {MIN_WINS_LEVEL} AND `rtb_dsp.hits\` > {MIN_HITS_LEVEL} AND \`rtb_ssp.wins\` > {MAX_DIFFERENCE_LEVEL_WIN_IMP} * \`rtb_dsp.block_impressions\` AND (cast(`rtb_event.cost` as double)/`rtb_dsp.hits`) < {MIN_HIT_COST_PRICE}) as impressions_rate,
        (\`rtb_ssp.wins\` > {MIN_WINS_LEVEL} AND \`rtb_dsp.bids\` > {MIN_BIDS_LEVEL} AND \`rtb_dsp.bids\` > {MAX_DIFFERENCE_LEVEL_BID_WIN} * \`rtb_ssp.wins\` AND (cast(`rtb_event.cost` as double)/`rtb_dsp.hits`) < {MIN_HIT_COST_PRICE}) as win_rate,
        (\`rtb_ssp.wins\` > {MIN_WINS_LEVEL}  AND \`rtb_ssp.sspprice\` > {MAX_COEFF_DIFF_SSPPRICE} * \`rtb_ssp.partnerprice\`) as sspprice_more_than_bid,
        (\`rtb_ssp.wins\` > {MIN_WINS_LEVEL}  AND \`rtb_ssp.sspprice\` > 30*\`rtb_event.cost\`) as sspprice_more_than_cost,
        (\`rtb_event.clicks\` > {MIN_CLICKS_LEVEL} AND \`rtb_event.clicks\`/cast(\`rtb_event.impressions\` as double) > {MAX_ALLOWED_CTR}) as ctr_high,
        (\`rtb_event.impressions\` > {MIN_IMPRESSIONS_LEVEL} AND cast(\`rtb_event.cost\` as double)/\`rtb_event.impressions\`/1000 > {MAX_ALLOWED_CPM}) as cpm_high
        {CUSTOM_CONDITION_SELECT}
    FROM
        `{table_log_final_stat}` as stat_table
    WHERE
        `timestamp` = '{timestamp}'
        AND (
            (\`rtb_dsp.hits\` > {MIN_HITS_LEVEL} AND \`rtb_dsp.hits\` > {MAX_DIFFERENCE_LEVEL_HIT_BID} * \`rtb_dsp.bids\` AND (cast(`rtb_event.cost` as double)/`rtb_dsp.hits`) < {MIN_HIT_COST_PRICE})
            OR (\`rtb_ssp.wins\` > {MIN_WINS_LEVEL} AND `rtb_dsp.hits\` > {MIN_HITS_LEVEL} AND \`rtb_ssp.wins\` > {MAX_DIFFERENCE_LEVEL_WIN_IMP} * \`rtb_dsp.block_impressions\` AND (cast(`rtb_event.cost` as double)/`rtb_dsp.hits`) < {MIN_HIT_COST_PRICE})
            OR (\`rtb_ssp.wins\` > {MIN_WINS_LEVEL} AND \`rtb_dsp.bids\` > {MIN_BIDS_LEVEL} AND \`rtb_dsp.bids\` > {MAX_DIFFERENCE_LEVEL_BID_WIN} * \`rtb_ssp.wins\` AND (cast(`rtb_event.cost` as double)/`rtb_dsp.hits`) < {MIN_HIT_COST_PRICE})
            OR (\`rtb_event.clicks\` > {MIN_CLICKS_LEVEL} AND \`rtb_event.clicks\`/cast(\`rtb_event.impressions\` as double) > {MAX_ALLOWED_CTR})
            OR (\`rtb_ssp.wins\` > {MIN_WINS_LEVEL} AND \`rtb_ssp.sspprice\` > {MAX_COEFF_DIFF_SSPPRICE} * \`rtb_ssp.partnerprice\`)
            OR (\`rtb_event.impressions\` > {MIN_IMPRESSIONS_LEVEL} AND cast(\`rtb_event.cost\` as double)/\`rtb_event.impressions\`/1000 > {MAX_ALLOWED_CPM})
            {CUSTOM_CONDITION_WHERE}
        )
        AND \`rtb_ssp.pagetoken\` IS NOT NULL
        AND \`rtb_ssp.pagetoken\` != ''
        AND cast(\`rtb_dsp.sspid\` as Int64) IN ({sspid_list})
    ORDER BY \`hits\` DESC
    LIMIT {limit}
    '''.format(
            table_log_final_stat = self.Parameters.table_log_final_stat,
            timestamp = self.start_process_time,
            limit = self.Parameters.candidates_limit,
            sspid_list = self.Parameters.sspid_list,
            MIN_HITS_LEVEL = self.MIN_HITS_LEVEL,
            MIN_WINS_LEVEL = self.MIN_WINS_LEVEL,
            MIN_BIDS_LEVEL = self.MIN_BIDS_LEVEL,
            MIN_CLICKS_LEVEL = self.MIN_CLICKS_LEVEL,
            MIN_IMPRESSIONS_LEVEL = self.MIN_IMPRESSIONS_LEVEL,
            MAX_DIFFERENCE_LEVEL_HIT_BID = self.MAX_DIFFERENCE_LEVEL_HIT_BID,
            MAX_DIFFERENCE_LEVEL_BID_WIN = self.MAX_DIFFERENCE_LEVEL_BID_WIN,
            MAX_DIFFERENCE_LEVEL_WIN_IMP = self.MAX_DIFFERENCE_LEVEL_WIN_IMP,
            MAX_ALLOWED_CPM = self.MAX_ALLOWED_CPM,
            MAX_ALLOWED_CTR = self.MAX_ALLOWED_CTR,
            MAX_COEFF_DIFF_SSPPRICE = self.MAX_COEFF_DIFF_SSPPRICE,
            MIN_HIT_COST_PRICE = self.MIN_HIT_COST_PRICE,
            CUSTOM_CONDITION_SELECT = CUSTOM_CONDITION_SELECT,
            CUSTOM_CONDITION_WHERE = CUSTOM_CONDITION_WHERE
        )

        return query

    def get_current_state_for_candidates(self, yt_cluster, candidates):
        where_condition = []

        candidates_by_ssp = {}
        for row in candidates:
            if 'pagetoken' not in row or row["pagetoken"] == "" or '"' in row["pagetoken"]:
                logging.info("Skip incorrect page_token: {}".format(row["pagetoken"]))
                continue

            if int(row["sspid"]) not in candidates_by_ssp.keys():
                candidates_by_ssp[int(row["sspid"])] = []

            candidates_by_ssp[int(row["sspid"])].append(str(row["pagetoken"]))

        for sspid in candidates_by_ssp.keys():
            where_condition.append("(SSPID = {sspid} AND PageToken IN ('{page_token_list}'))".format(sspid = sspid, page_token_list = "', '".join(candidates_by_ssp[sspid])))

        if not (len(where_condition) > 0):
            return None

        query_current_state = '''
            SELECT
                SSPID, PageToken, PageTokenMD5, ShowProbability
            FROM
                `{final_table_show_probability}`
            WHERE
                {where_condition}
        '''.format(where_condition = " OR ".join(where_condition), final_table_show_probability = self.FINAL_TABLE_SHOW_PROBABILITY)

        response = self.read_yt_table(yt_cluster, query_current_state)
        return response

    def get_unban_candidates(self):

        query = '''
            \$final_stat_table = (
                SELECT
                    cast(\`rtb_dsp.sspid\` as Int64) as SSPID,
                    \`rtb_dsp.pagetoken\` as PageToken,
                    `timestamp`
                FROM `{table_log_final_stat}`
                WHERE `timestamp` = '{timestamp}'
            );

            SELECT
                current_state.SSPID as SSPID,
                current_state.PageToken as PageToken,
                current_state.PageTokenMD5 as PageTokenMD5,
                current_state.ShowProbability as ShowProbability
            FROM
                `{final_table_show_probability}` as current_state
                LEFT JOIN \$final_stat_table as stat_table ON stat_table.SSPID = current_state.SSPID AND stat_table.PageToken = current_state.PageToken
            WHERE
                stat_table.`timestamp` IS NULL
                AND current_state.SSPID in ({sspid_list})
                AND current_state.ShowProbability < {max_show_probability}
                AND current_state.ShowProbability >= 0

        '''.format(final_table_show_probability = self.FINAL_TABLE_SHOW_PROBABILITY, table_log_final_stat = self.Parameters.table_log_final_stat, timestamp = self.start_process_time, sspid_list = self.Parameters.sspid_list, max_show_probability = self.MAX_SHOWS_PROBABILITY)

        response = self.read_yt_table(self.Parameters.yt_stat_cluster, query)
        return response

    REASON2COEFF = {
        'sspprice_more_than_cost': 0.5,
        'ctr_high': 0.5,
        'cpm_high': 0.5,
    }

    REASON2FIXED = {
        'bids_rate': 50000,
        'win_rate': 50000,
        'impressions_rate': 50000,
        'sspprice_more_than_bid': 50000,
    }

    MAX_SHOWS_PROBABILITY = 1000000

    def process_candidates(self, candidates):

        if not self.check_array_length(candidates):
            return None

        current_state = self.get_current_state_for_candidates(self.Parameters.yt_read_cluster_finish_table, candidates)

        current_state_dict = {}
        for row in current_state:
            current_state_dict[ '-'.join([str(row["SSPID"]), str(row["PageToken"])]) ] = row

        reasons_dict = dict((reason, 1) for reason in self.Parameters.reasons_list.split(","))

        min_coeff_value = None
        min_fixed_value = None

        rows_to_apply = []
        logging.info("Process candidates")

        for candidate in candidates:
            kk = '-'.join([str(candidate["sspid"]), str(candidate["pagetoken"])])

            current_shows_probability_value = 0
            try:
                current_shows_probability_value = current_state_dict[ kk ]["ShowProbability"]
            except:
                current_shows_probability_value = self.MAX_SHOWS_PROBABILITY

            min_coeff_value = 1
            min_fixed_value = self.MAX_SHOWS_PROBABILITY

            for attr in candidate.keys():

                if attr not in reasons_dict or not candidate[attr]:
                    continue

                if attr in self.REASON2COEFF and self.REASON2COEFF[attr] < min_coeff_value:
                    min_coeff_value = self.REASON2COEFF[attr]

                if attr in self.REASON2FIXED and self.REASON2FIXED[attr] < min_fixed_value:
                    min_fixed_value = self.REASON2FIXED[attr]

            logging.info(json.dumps(candidate))
            logging.info(json.dumps(min_fixed_value))
            logging.info(json.dumps(min_coeff_value))

            new_value = int( min([min_fixed_value, current_shows_probability_value*min_coeff_value]) or 1 )

            logging.info("New value = " + str(new_value))

            row_to_apply = {}

            if kk in current_state:
                row_to_apply = copy.deepcopy(current_state_dict[kk])
            else:
                row_to_apply = {
                    "SSPID": int(candidate["sspid"]),
                    "PageToken": str(candidate["pagetoken"]),
                    "PageTokenMD5": int(self.md5unit64(candidate["pagetoken"])),
                    "ShowProbability": int(self.MAX_SHOWS_PROBABILITY)
                }

            row_to_apply["ShowProbability"] = int(new_value)
            row_to_apply["_hits"] = candidate["hits"]

            if "cost" in candidate and candidate["cost"] is not None:
                row_to_apply["_cost"] = candidate["cost"]/1000000

            if "YtHash" in row_to_apply:
                del row_to_apply["YtHash"]

            if kk not in current_state and row_to_apply["ShowProbability"] != self.MAX_SHOWS_PROBABILITY \
                or kk in current_state and int(current_state_dict[kk]["ShowProbability"]) > int(row_to_apply["ShowProbability"]):
                rows_to_apply.append(row_to_apply)

        return rows_to_apply

    def filter_candidates_for_apply(self, candidates):
        logging.info("Start filter the candidates by list of allowed SSPID: {}".format(self.Parameters.sspid_list))
        logging.info("Before: {}".format(len(candidates)))

        if not self.check_array_length(candidates):
            return None

        allowed_sspid_dict = dict((int(sspid), 1) for sspid in self.Parameters.sspid_list.split(","))
        candidates = [ rec for rec in candidates if int(rec["SSPID"]) in allowed_sspid_dict.keys() ]

        if self.Parameters.sort_method == 'random':
            random.shuffle(candidates)
        elif self.Parameters.sort_method == 'shows_probability':
            candidates = sorted( candidates, key = lambda x: x['ShowProbability'] )

        if candidates and type(candidates) == list:
            logging.info("After filter by SSPID: {}".format(len(candidates)))

            filtered_candidates = copy.deepcopy(candidates)

            if self.Parameters.daily_money_loss_limit > 0:
                filtered_candidates = []

                inc_sum = 0
                for item in candidates:
                    if "_cost" in item:
                        inc_sum += item["_cost"]

                    if inc_sum <= self.Parameters.daily_money_loss_limit:
                        filtered_candidates.append(item)
                    else:
                        break

                logging.info("After filter by money: {}".format(len(filtered_candidates)))

            candidates = copy.deepcopy(filtered_candidates)

            if self.Parameters.hits_reduction_limit > 0:
                filtered_candidates = []

                inc_sum = 0
                logging.info(len(candidates))
                for item in candidates:
                    inc_sum += item["_hits"]
                    logging.info("hits:{}, sum:{}".format(item["_hits"], inc_sum))
                    if inc_sum <= self.Parameters.hits_reduction_limit:
                        logging.info(item)
                        filtered_candidates.append(item)
                    else:
                        logging.info("break")
                        break

                logging.info("After filter by hits limit: {}".format(len(filtered_candidates)))

            for item in filtered_candidates:
                # TODO: add normalizing by current value for ShowProbability
                if "_cost" in item:
                    self.result_stat["AffectedCost"] += item["_cost"]
                    del item["_cost"]

                if "_hits" in item:
                    self.result_stat["AffectedHits"] += item["_hits"]
                    del item["_hits"]

            if self.Parameters.rows_limit_to_apply_ban >= 0:
                filtered_candidates = filtered_candidates[0:self.Parameters.rows_limit_to_apply_ban]

            logging.info("After filter by limit: {}".format(len(filtered_candidates)))

            return filtered_candidates
        else:
            logging.info("After: empty array")
            return None

    def log_results(self, candidates, stage=""):
        logging.info("Start logging the candidates")
        logging.info("Stage: {}".format(stage))

        log_candidates = copy.deepcopy(candidates)

        if not self.check_array_length(log_candidates):
            return None

        chunk_size = 1000
        chunked_items = [log_candidates[i:i + chunk_size] for i in xrange(0, len(log_candidates), chunk_size)]

        ytc = self.yt_connect(self.Parameters.yt_logging_cluster)

        def _row_add_stage(obj, stage, timestamp):
            obj["stage"] = stage
            obj["timestamp"] = timestamp
            return obj

        from yt.wrapper.errors import YtError, YtResponseError

        if not ytc.exists(self.Parameters.yt_log_table):
            ytc.create_table(self.Parameters.yt_log_table)

        for chunk in chunked_items:
            chunk = [ _row_add_stage(x, stage, self.log_timestamp) for x in chunk ]
            try:
                ytc.write_table(ytc.TablePath(self.Parameters.yt_log_table, append = True), chunk)
                # ytc.insert_rows(self.Parameters.yt_log_table, chunk)
                logging.info("Succefully insert {0} rows".format(len(chunk)))
            except YtResponseError as err:
                logging.error("YtResponseError: " + str(err.inner_errors[0]["message"]))
                raise err
            except YtError as err:
                logging.error("YtError: " + str(err.inner_errors[0]["message"]))
                raise err
            except:
                logging.info("Fail insert rows")
                raise
        logging.info("Finish logging the candidates")

    def sanity_check(self, candidates):
        for row in candidates:
            if 'PageToken' not in row or str(row['PageToken']) == "":
                logging.info(row)
                raise ValueError("PageToken is empty")

            if 'PageTokenMD5' not in row or str(row['PageTokenMD5']) == "" or str(row['PageTokenMD5']) != str(self.md5unit64(row['PageToken'])):
                logging.info(row)
                raise ValueError("PageTokenMD5 is empty or wrong")

            if 'SSPID' not in row or str(row['SSPID']) == "" or int(row["SSPID"]) == 0:
                logging.info(row)
                raise ValueError("SSPID is empty")

            if 'ShowProbability' not in row or int(row["ShowProbability"]) < -100 or int(row["ShowProbability"]) > self.MAX_SHOWS_PROBABILITY:
                logging.info(row)
                raise ValueError("ShowProbability is empty")

            for key in row.keys():
                if key not in ['PageToken', 'PageTokenMD5', 'SSPID', 'ShowProbability']:
                    logging.info(row)
                    raise ValueError("Uknown column: " + str(key))

        return None

    def upsert_values(self, candidates, double_check_update=False):
        from yt.wrapper.errors import YtError, YtResponseError
        logging.info("Start applying the results")

        logging.info("Start sanity checks")
        self.sanity_check(candidates)

        chunk_size = 1000
        chunked_items = [candidates[i:i + chunk_size] for i in xrange(0, len(candidates), chunk_size)]

        for chunk in chunked_items:
            self.log_results(chunk, "apply_action")

        self.result_stat["AffectedApps"] = len(candidates)

        if not double_check_update:
            return None

        ytc = self.yt_connect(self.Parameters.yt_apply_cluster_finish_table)

        for chunk in chunked_items:
            try:
                logging.info(chunk)
                ytc.insert_rows(self.Parameters.yt_finish_table_name, chunk, require_sync_replica=False)
                logging.info("Upsert: Succefully insert {0} rows".format(len(chunk)))
            except YtResponseError as err:
                logging.error("Upsert: YqlResponseError: " + str(err.inner_errors[0]["message"]))
                raise err
            except YtError as err:
                logging.error("Upsert: YqlError: " + str(err.inner_errors[0]["message"]))
                raise err
            except:
                logging.info("Upsert: fail insert rows")
                pass
        logging.info("Finish applying the results")

    def do_ban_action(self, custom_exceptions_dict):

        if self.Parameters.custom_ban_condition != "" and self.Parameters.custom_ban_condition_name != "" and self.Parameters.custom_ban_condition_abs_value != "0":
            self.REASON2FIXED[self.Parameters.custom_ban_condition_name] = self.Parameters.custom_ban_condition_abs_value

        if not self.Parameters.skip_generate_new_data:
            logging.info("Start preprocessing data")
            logging.info("Start clear statistic in cache (old table)")
            query_prepare_stat = self.get_query_pagetoken_statistic()

            logging.info(query_prepare_stat)
            response = self.read_yt_table(self.Parameters.yt_stat_cluster, query_prepare_stat)

        logging.info("Start selection data for processing")
        query_get_candidates = self.get_query_candidates_for_throttling()

        logging.info(query_get_candidates)
        response = self.read_yt_table(self.Parameters.yt_stat_cluster, query_get_candidates)

        self.log_results(response, "after_selection")

        if not self.check_array_length(response):
            logging.info("Incorrect response")
            return None

        logging.info("Start processing selected candidates")
        candidates = self.process_candidates(response)

        logging.info("Start logging processed candidates")
        self.log_results(candidates, "before_filter")

        if not self.check_array_length(candidates):
            return None

        logging.info("Start filtering candidates before apply")
        candidates = self.filter_candidates_for_apply(candidates)

        logging.info("Start logging filtered candidates")
        self.log_results(candidates, "after_filter")

        if custom_exceptions_dict is not None:
            candidates = self.intersect_candidates_with_exceptions(candidates, custom_exceptions_dict)

        if not self.check_array_length(candidates):
            logging.info("Skip candidates array length check")
            return None

        if candidates is not None and len(candidates) > 0:

            logging.info("Start applying new calculated values")
            res = self.upsert_values(candidates, True)

            list_apps = "\n\t".join(self.get_app_description(row) for row in candidates)
            self.email_notification_texts_array.append("\nReduced ShowProbability for:\n\n" + list_apps)
            self.result_stat["AffectedAppsBan"] = len(candidates)

            logging.info("Result = " + str(res))

            return candidates
        else:
            return None

    def do_intersection_monitoring(self, custom_exceptions_dict):
        logging.info("Start ssp-sdk intersection monitoring task")

        if self.Parameters.debug_refresh_appid_mobile_bundle_table:
            self.do_refresh_appid_bundle_table()

        query_intersection = self.get_query_ssp_sdk_intersection()
        logging.info(query_intersection)
        response = self.read_yt_table(self.Parameters.yt_stat_cluster, query_intersection)

        logging.info(response)

        for row in response:
            self.email_notification_intersection.append(self.get_app_intersection_description(row))

        candidates = []
        for candidate in response:
            row_to_apply = {
                "SSPID": int(candidate["sspid"]),
                "PageToken": str(candidate["ssp_pagetoken"]),
                "PageTokenMD5": int(self.md5unit64(candidate["ssp_pagetoken"])),
                "ShowProbability": int(self.Parameters.custom_ban_intersection_ssp_sdk_abs_value)
            }
            logging.info(row_to_apply)
            candidates.append(row_to_apply)

        if custom_exceptions_dict is not None:
            candidates = self.intersect_candidates_with_exceptions(candidates, custom_exceptions_dict)

        if not self.check_array_length(candidates):
            logging.info("Skip candidates array length check")
            return None

        if candidates is not None and len(candidates) > 0:

            logging.info("Start ban intersection ssp-sdk candidates")
            res = self.upsert_values(candidates, self.Parameters.debug_real_ban_intersection_ssp_sdk)

            list_apps = "\n\t".join(self.get_app_description(row) for row in candidates)
            self.email_notification_texts_array.append("\nBanned intersection SSP-SDK for:\n\n" + list_apps)

            self.result_stat["AffectedAppsBan"] = len(candidates)

            logging.info("Result (intersection ssp-sdk) = " + str(res))

            return candidates
        else:
            return None

    def do_refresh_appid_bundle_table(self):
        # tmp fix with refresh table
        query_refresh_table = '''
            insert into `{table_appid_to_bundle}`
            select "tmp-fix" as bundle, "tmp-fix" as appid, "tmp-fix" as store
            from `{table_appid_to_bundle}`
            limit 1
        '''.format(table_appid_to_bundle=self.Parameters.table_appid_to_bundle)

        response = self.read_yt_table(self.Parameters.yt_stat_cluster, query_refresh_table)

        return response

    def get_app_intersection_description(self, row):
        return ", ".join("{}={}".format(k,v) for k, v in sorted(row.items()))

    def get_app_description(self, row):
        return " : ".join([str(row["SSPID"]), str(row["PageToken"]), str(row["ShowProbability"])])

    def intersect_candidates_with_exceptions(self, candidates, custom_exceptions_dict):

        if not self.check_array_length(candidates):
            candidates = []

        candidates = [
            item for item in candidates if "-".join([str(item["SSPID"]), str(item["PageToken"])]) not in custom_exceptions_dict.keys()
        ]
        candidates = candidates + custom_exceptions_dict.values()

        return candidates

    def check_array_length(self, var):
        if var is None:
            logging.info("Error: undefined value")
            return None
        elif type(var) != list:
            logging.info("Error: type is not an array")
            return None
        elif len(var) == 0:
            logging.info("Error: empty array")
            return None

        return True

    UNBAN_INCREMENT_SHOW_PROBABILITY = 100000
    UNBAN_MAX_ALLOWED_INCREASE_COEFF = 10

    def do_unban_action(self, custom_exceptions_dict):
        logging.info("Start unban action")

        unban_candidates = self.get_unban_candidates()
        logging.info("Finished candidates select")
        logging.info(unban_candidates)

        if self.Parameters.rows_limit_to_apply_unban >= 0:
            logging.info("Splice array of candidates")
            unban_candidates = unban_candidates[0:self.Parameters.rows_limit_to_apply_unban]

        for item in unban_candidates:
            kk = "-".join([str(item["SSPID"]), str(item["PageToken"])])
            if kk in custom_exceptions_dict:
                # do not change custom exceptions
                item["ShowProbability"] = custom_exceptions_dict[kk]["ShowProbability"]
            else:
                delta = self.UNBAN_INCREMENT_SHOW_PROBABILITY
                if self.Parameters.delta_show_probability_unban >= 0:
                    delta = self.Parameters.delta_show_probability_unban

                item["ShowProbability"] = min(item["ShowProbability"]*self.UNBAN_MAX_ALLOWED_INCREASE_COEFF, item["ShowProbability"]+delta)

            item["ShowProbability"] = min(item["ShowProbability"], self.MAX_SHOWS_PROBABILITY)

            if self.Parameters.debug_allow_fix_pagetoken_md5:
                item["PageTokenMD5"] = int(self.md5unit64(item["PageToken"]))

        logging.info("Start applying process")
        res = self.upsert_values(unban_candidates, True)

        list_apps = "\n\t".join(self.get_app_description(row) for row in unban_candidates)
        self.email_notification_texts_array.append("\n\nIncreased ShowProbability for:\n\n" + list_apps)

        if res is not None:
            self.result_stat["AffectedAppsUnban"] = res

        logging.info("Finished applying process")
        logging.info(res)

        return unban_candidates

    def get_process_time(self, frmt='%Y-%m-%d', days=1):
        start_time = datetime.now() - timedelta(days=days)
        return start_time.strftime(frmt)

    def parse_custom_exceptions(self, custom_exceptions):
        res = {}

        if custom_exceptions is not None and custom_exceptions != "":
            for block in str(custom_exceptions).split(","):
                sspid, bundle, show_prob = str(block).split(":")

                if not(int(sspid) > 0 and str(bundle) != "" and int(show_prob) > -100 and int(show_prob) < self.MAX_SHOWS_PROBABILITY):
                    raise ValueError("Incorrect custom exceptions: " + str(block))

                kk = "-".join([str(sspid), str(bundle)])
                res[kk] = {
                    "SSPID": int(sspid),
                    "PageToken": str(bundle),
                    "PageTokenMD5": int(self.md5unit64(bundle)),
                    "ShowProbability": int(show_prob)
                }
        return res

    def apply_custom_exceptions(self, custom_exceptions_dict):
        logging.info("Start process and apply custom exceptions")

        custom_exceptions_to_apply = custom_exceptions_dict.values()

        logging.info("Data to apply:")
        logging.info(custom_exceptions_to_apply)

        res = self.upsert_values(custom_exceptions_to_apply, True)

        logging.info("Result apply:" + str(res))

        return res

    def on_execute(self):
        logging.info("Started main process!")

        self.FINAL_TABLE_SHOW_PROBABILITY = '//home/yabs/dict/SSPPageTokenShowProbability'

        self.start_process_time = self.get_process_time("%Y%m%d%H%M%S")
        self.log_timestamp = self.get_process_time("%Y%m%d%H%M%S")

        if self.Parameters.debug_fixed_timestamp > 0:
            self.start_process_time = self.Parameters.debug_fixed_timestamp

        self.yql_token = sdk2.task.Vault.data(self.Parameters.yql_token_owner, self.Parameters.yql_token_vault_name)

        custom_exceptions_dict = {}

        if self.Parameters.custom_exceptions != "":
            custom_exceptions_dict = self.parse_custom_exceptions(self.Parameters.custom_exceptions)
            logging.info("Custom exceptions length = {}".format(len(custom_exceptions_dict.keys())))

        if self.Parameters.execute_update_custom_exceptions_only:
            logging.info("Apply custom exceptions")
            logging.info(self.Parameters.custom_exceptions)

            res = self.apply_custom_exceptions(custom_exceptions_dict)

            logging.info("Result: " + str(res))
        else:
            logging.info("Try do action")
            banned_records = []

            if self.Parameters.execute_ban_action:
            	logging.info("Do ban action")
                banned_records = self.do_ban_action(custom_exceptions_dict)
		self.log_results(banned_records, "result_ban_action")

            if self.Parameters.execute_unban_action:
            	logging.info("Do unban action")
                unbanned_records = self.do_unban_action(custom_exceptions_dict)
		self.log_results(unbanned_records, "result_unban_action")

        self.send_stat_to_solomon(self.Parameters.solomon_key.data()['token'])

        if len(self.email_notification_texts_array) > 0:
            self.send_email("\n\n".join(self.email_notification_texts_array))

        if self.Parameters.execute_ssp_sdk_intersection:
            self.do_intersection_monitoring(custom_exceptions_dict)

            if len(self.email_notification_intersection) > 0:
                self.send_email("\n".join(self.email_notification_intersection))

	return None

    def send_email(self, body_txt, format="text"):
        frm = 'Yandex RTB Reporter <ssp-int@yandex-team.ru>'
        to = self.Parameters.email_notifications.split(',')

        date_for_log = self.get_process_time('%Y-%m-%d')

        subject = "Yandex SSP Automoderation {process_date}".format(process_date=date_for_log)

        msg = MIMEMultipart('alternative')
        msg.set_charset('utf8')
        msg['From'] = frm
        msg['To'] = ', '.join(to)
        msg['Cc'] = ', '.join('mirage@yandex-team.ru')
        msg['Subject'] = Header(subject, 'UTF-8').encode()

        if format == "csv":
            body = MIMEText("", 'plain', 'utf8')
            msg.attach(body)

            attachment = MIMEBase('application', 'csv')

            filename = "data-%s.csv" % date_for_log
            header = 'Content-Disposition', 'attachment; filename="%s"' % filename
            attachment.add_header(*header)

            attachment.set_payload(body_txt)
            msg.attach(attachment)
        else:
            body = MIMEText(body_txt, 'plain', 'utf8')
            msg.attach(body)

        try:
            srv = smtplib.SMTP('yabacks.yandex.ru', port=25)
            srv.sendmail(frm, to, msg.as_string())
            srv.quit()
        except smtplib.SMTPException:
            raise Exception('No e-mails was sent. Internal SMTP exception occured')

    def send_stat_to_solomon(self, solomon_token):

        if not (self.Parameters.solomon_project != "" and self.Parameters.solomon_cluster != "" and self.Parameters.solomon_service != ""):
            logging.info("Skipped sending data to Solomon")
            return None

        SOLOMON_URL = 'https://solomon.yandex.net/api/v2/push?project={project}&cluster={cluster}&service={service}'.format(
            project=self.Parameters.solomon_project,
            cluster=self.Parameters.solomon_cluster,
            service=self.Parameters.solomon_service
        )

        logging.info("Sending data to solomon")

        headers = {
            'Content-type': 'application/json',
            'Authorization': 'OAuth {token}'.format(token=solomon_token)
        }

        data = {
            "sensors": [
                {
                    "labels": {"SSPAutomoderation": "AffectedApps"},
                    "ts": int(time.time()),
                    "value": int(self.result_stat["AffectedApps"]),
                },
                {
                    "labels": {"SSPAutomoderation": "AffectedHits"},
                    "ts": int(time.time()),
                    "value": int(self.result_stat["AffectedHits"]),
                },
                {
                    "labels": {"SSPAutomoderation": "AffectedCost"},
                    "ts": int(time.time()),
                    "value": int(self.result_stat["AffectedCost"]),
                },
                {
                    "labels": {"SSPAutomoderation": "AffectedAppsBan"},
                    "ts": int(time.time()),
                    "value": int(self.result_stat["AffectedAppsBan"]),
                },
                {
                    "labels": {"SSPAutomoderation": "AffectedAppsUnban"},
                    "ts": int(time.time()),
                    "value": int(self.result_stat["AffectedAppsUnban"]),
                }
            ]

        }
        response = requests.post(SOLOMON_URL, data=json.dumps(data), headers=headers)
        response.raise_for_status()
        logging.info("Data to solomon: " + json.dumps(data))
        logging.info("Sent all to solomon")
