import logging
import smtplib
from datetime import datetime, timedelta
from email.header import Header
from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText
from sandbox.sandboxsdk import environments
from sandbox import sdk2


MOSCOW_UTC_OFFSET = 3


class YabsServerSSPReport(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        yql_token_vault_name = sdk2.parameters.String("YQL Token vault name", default="yql_token", required=True)
        db = sdk2.parameters.String("Yt Cluster", default="hahn", required=True)
        recipients = sdk2.parameters.String("EMail recipients (by comma)", default="ssp-int@yandex-team.ru", required=True)
        process_delay = sdk2.parameters.Integer("Process delay (in hours)", default=30, required=True)
        utc_offset = sdk2.parameters.Integer("Offset from UTC time (in hours)", default=MOSCOW_UTC_OFFSET, required=True)
        sspid = sdk2.parameters.Integer("SSPID", required=True)

    class Requirements(sdk2.Task.Requirements):
        environments = [environments.PipEnvironment('yql', version='1.2.91'), environments.PipEnvironment('yandex-yt')]

    def GetProcessDate(self):
        log_frmt = '%Y-%m-%dT%H:00:00'
        frmt = '%Y-%m-%d'

        process_date = (datetime.utcnow() + timedelta(hours=self.Parameters.utc_offset - self.Parameters.process_delay)).replace(hour=0, minute=0, second=0)
        process_date_msk = process_date + timedelta(hours=MOSCOW_UTC_OFFSET - self.Parameters.utc_offset)

        return process_date_msk.strftime(log_frmt), (process_date_msk + timedelta(hours=23)).strftime(log_frmt), process_date.strftime(frmt)

    def GetRequest(self):
        return '''
            use hahn;

            select count(*) as Hits
            from range(`logs/bs-rtb-log/1h`, `{process_date_start}`, `{process_date_end}`) as Log
            where cast(Log.sspid as Uint64) == {sspid};

            select count(*) as Shows
            from range(`logs/bs-dsp-log/1h`, `{process_date_start}`, `{process_date_end}`) as Log
            where cast(Log.sspid as Uint64) == {sspid} and cast(Log.countertype as Uint32) == 1;

            select sum(cast(ssppricecur as Double)) / 1e6 as Cost
            from range(`logs/bs-ssp-log/1h`, `{process_date_start}`, `{process_date_end}`) as Log
            where cast(Log.sspid as Uint64) == {sspid};
        '''.format(
            process_date_start=self.process_date_start,
            process_date_end=self.process_date_end,
            sspid=self.Parameters.sspid
        )

    def SendEmail(self, report):
        frm = 'Yandex RTB Reporter <ssp-int@yandex-team.ru>'
        to = self.Parameters.recipients.split(',')
        subject = "Yandex Daily RTB Stats {process_date}".format(process_date=self.process_date)
        body_txt = """
            Date: {process_date}
            Requests: {hits}
            Impressions: {shows}
            Revenue: {cost}
        """.format(process_date=self.process_date, hits=report[0]["Hits"], shows=report[1]["Shows"], cost=report[2]["Cost"])

        msg = MIMEMultipart('alternative')
        msg.set_charset('utf8')
        msg['From'] = frm
        msg['To'] = ', '.join(to)
        msg['Subject'] = Header(subject, 'UTF-8').encode()

        body = MIMEText(body_txt, 'plain', 'utf8')
        msg.attach(body)

        try:
            srv = smtplib.SMTP('yabacks.yandex.ru', port=25)
            srv.sendmail(frm, to, msg.as_string())
            srv.quit()
        except smtplib.SMTPException:
            raise Exception('No e-mails was sent. Internal SMTP exception occured')

    def on_execute(self):
        self.yql_token = sdk2.task.Vault.data(self.author, self.Parameters.yql_token_vault_name)

        logging.info("Started!")

        self.process_date_start, self.process_date_end, self.process_date = self.GetProcessDate()
        logging.info("Gonna run for {0} UTC {1} (log from {2} to {3} MSK)".format(self.process_date, self.Parameters.utc_offset, self.process_date_start, self.process_date_end))

        request = self.GetRequest()
        logging.info("Prepared request " + request)

        from yql.api.v1.client import YqlClient
        client = YqlClient(db=self.Parameters.db, token=self.yql_token)
        query = client.query(request, syntax_version=1)
        query.run()
        query.wait_progress()

        if not query.is_success:
            raise Exception('\n'.join([str(err) for err in query.errors]))

        result = []
        for table in query.get_results():
            table.fetch_full_data()

            columns = []
            for column_name, column_type in table.columns:
                columns.append(column_name)

            for row in table.rows:
                result.append(dict([(columns[i], value) for i, value in enumerate(row)]))

        logging.info("RESULT: {}".format(result))

        logging.info("Sending email...")
        self.SendEmail(result)

        logging.info("Done!")
