# -*- coding: utf-8 -*-

import json
import logging
from sandbox import sdk2
import sandbox.common.types.resource as ctr
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.common.types.task import ReleaseStatus


YT_TOKEN_VAULT_NAME = "robot-yabs-yt-backup-token"


class YabsYtBackupRestoreBinary(sdk2.Resource):
    """ YABS_YT_BACKUP_RESTORE_BINARY """
    releasable = True
    any_arch = False
    auto_backup = True
    executable = True
    releasers = ['danilgrig', 'radolga', 'yabsstartrek']


def get_executable():
    attrs = {
        'released': ReleaseStatus.STABLE,
    }
    res = YabsYtBackupRestoreBinary.find(state=ctr.State.READY, attrs=attrs).first()
    res_data = sdk2.ResourceData(res)
    return str(res_data.path)


class YabsYtBackupRestore(sdk2.Task):
    """ YABS_YT_BACKUP_RESTORE """

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 4096
        disk_space = 4096

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Main params') as main_block:
            src_cluster = sdk2.parameters.String('Source YT cluster', required=True)
            dst_cluster = sdk2.parameters.String('Destination YT cluster', required=True)
            mode = sdk2.parameters.RadioGroup(
                'Work mode',
                choices=(
                    ("Dynamic -> Static (advanced_backup)", "advanced_backup"),
                    ("Static -> Dynamic (advanced_restore)", "advanced_restore"),
                    ("Dynamic -> Dynamic (clone)", "clone"),
                    ("Static -> Static (transfer)", "transfer"),
                    ("Backup", "backup"),
                    ("Restore", "restore"),
                ),
                sub_fields={
                    'backup': [
                        'tables_list',
                        'dump_query',
                        'ttl',
                        'erasure',
                        'overwrite_backup',
                    ],
                    'advanced_backup': [
                        'tables_dict',
                        'dump_query',
                        'ttl',
                        'erasure',
                        'overwrite_backup',
                    ],
                    'clone': [
                        'tables_dict',
                        'dump_query',
                        'overwrite_dst_table',
                        'erase_query',
                        'force_compaction',
                        'use_ssd_blobs',
                    ],
                    'transfer': [
                        'tables_dict',
                        'dump_query',
                        'overwrite_backup',
                    ],
                    'restore': [
                        'tables_list',
                        'origin_cluster',
                        'backup_date',
                        'overwrite_dst_table',
                        'erase_query',
                        'force_compaction',
                        'use_ssd_blobs',
                    ],
                    'advanced_restore': [
                        'tables_dict',
                        'dump_query',
                        'overwrite_dst_table',
                        'erase_query',
                        'force_compaction',
                        'use_ssd_blobs',
                    ],
                },
                default='clone',
                required=True,
            )
            origin_cluster = sdk2.parameters.String('Origin backups YT cluster')
            tables_list = sdk2.parameters.List('Dynamic tables list')
            tables_dict = sdk2.parameters.Dict('Input-output tables dict')
            backup_date = sdk2.parameters.String('Backup date (yyyy-mm-dd)')
            dump_query = sdk2.parameters.String('SQL-like predicate to select rows from src tables')
            erase_query = sdk2.parameters.String('SQL-like predicate to erase rows from dst table before restore')
            ttl = sdk2.parameters.Integer('TTL in days for backup tables')
            delete_tmp = sdk2.parameters.Bool('Delete tmp dumps', default=True)
            overwrite_dst_table = sdk2.parameters.Bool('Overwrite dst table if it exist', sub_fields={'false': ['erase_query']})
            overwrite_backup = sdk2.parameters.Bool('Overwrite static table if it exist')
            force_compaction = sdk2.parameters.Bool('Force compaction after restore')
            erasure = sdk2.parameters.Bool('Convert backup to erasure')
            use_ssd_blobs = sdk2.parameters.Bool('Add ssd_blobs to restored table')

        with sdk2.parameters.Group('Specific params. Don\'t change if you not sure') as specific_block:
            use_transfer_queue = sdk2.parameters.Bool('Use special queue for TM', sub_fields={'true': ['queue_name']})
            queue_name = sdk2.parameters.String('Queue name for TM')
            use_custom_pool = sdk2.parameters.Bool('Use special pool for TM (sic!)', sub_fields={'true': ['custom_tm_pool']})
            custom_tm_pool = sdk2.parameters.String('Special pool name for TM')
            use_tables_list_input = sdk2.parameters.Bool('Tables list in one field (hack for move)', sub_fields={'true': ['tables_list_input']})
            tables_list_input = sdk2.parameters.String('List of dynamic tables', multiline=True)
            copy_attributes = sdk2.parameters.List('List of attributes to save/copy/restore')
            tablet_count = sdk2.parameters.Integer('tablet_count for reshard')
            job_count = sdk2.parameters.Integer('job_count for DumpRestoreClient')
            user_slots = sdk2.parameters.Integer('user_slots for DumpRestoreClient')
            max_failed_job_count = sdk2.parameters.Integer('max_failed_job_count for DumpRestoreClient')
            yt_token_vault = sdk2.parameters.String('YT token secret name', default=YT_TOKEN_VAULT_NAME,)

    def on_execute(self):
        vault_name = YT_TOKEN_VAULT_NAME
        if self.Parameters.yt_token_vault:
            vault_name = self.Parameters.yt_token_vault
        yt_token = sdk2.Vault.data(vault_name)

        params = dict(self.Parameters)
        params.update(
            yt_token=str(yt_token),
            task_id=self.id,
        )

        descr = "{} from {} to {} : ".format(
            self.Parameters.mode,
            self.Parameters.src_cluster,
            self.Parameters.dst_cluster,
        )

        if self.Parameters.use_tables_list_input:
            descr += "{}".format(self.Parameters.tables_list_input.replace('\n', ', '))
        else:
            if self.Parameters.mode in ('backup', 'restore'):
                descr += "{}".format(list(self.Parameters.tables_list))
            else:
                descr += "{}".format(dict(self.Parameters.tables_dict))

        if self.Parameters.mode == 'restore':
            descr += " from backup {}/{}".format(
                self.Parameters.origin_cluster,
                self.Parameters.backup_date,
            )

        if self.Parameters.dump_query:
            descr += ' by DumpQuery "{}"'.format(self.Parameters.dump_query)

        if self.Parameters.erase_query:
            descr += ' by EraseQuery "{}"'.format(self.Parameters.erase_query)

        self.set_info(descr)

        exe_path = get_executable()
        command = [
            exe_path,
            json.dumps(params)
        ]

        with sdk2.helpers.ProcessLog(self, logger="yt_backup_restore") as pl:
            pl.logger.propagate = 1
            retcode = sp.Popen(command, stdout=pl.stdout, stderr=sp.STDOUT).wait()
            if retcode == 0:
                logging.info("Task processed successfully")
            else:
                raise Exception("Task execution failed")


__TASK__ = YabsYtBackupRestore
