import logging
from itertools import chain
import os

from sandbox import sdk2
from sandbox import common
from subprocess import Popen, PIPE


def get_binary_path():
    resource = sdk2.Resource.find(
        attrs=dict(resource_name='yabs-yt-upserter')
    ).order(-sdk2.Resource.id).first()

    if resource:
        logging.info('Got resource %i' % resource.id)
        return str(sdk2.ResourceData(resource).path)
    else:
        raise common.errors.TaskError('Binary resource does not exist')


class YabsYtUpserter(sdk2.Task):
    """Upserts data to YT"""
    class Parameters(sdk2.Task.Parameters):
        cluster_name = sdk2.parameters.String(
            'Cluster name', required=True,
            default_value='hahn'
        )

        key_columns = sdk2.parameters.List(
            'Key columns', required=True
        )

        value_columns = sdk2.parameters.List(
            'Value columns', required=True
        )

        input_table = sdk2.parameters.String(
            'Input table path', required=True
        )

        output_table = sdk2.parameters.String(
            'Output table path', required=True
        )

        yt_token_vault_name = sdk2.parameters.String(
            label='YT token vault name',
            required=True,
        )

    def on_execute(self):
        yt_token = sdk2.Vault.data(self.Parameters.yt_token_vault_name)

        key_columns = self.Parameters.key_columns
        value_columns = self.Parameters.value_columns

        output_table_path = self.Parameters.output_table

        binary_path = get_binary_path()

        def prepare_list_args(label, columns):
            return list(chain(*map(lambda c: (label, c), columns)))

        args = [binary_path] + ['--proxy', self.Parameters.cluster_name] + \
            prepare_list_args('--key-column', key_columns) + \
            prepare_list_args('--value-column', value_columns) + \
            ['--input', self.Parameters.input_table, '--output', output_table_path]

        logging.info('launch binary: %s' % str(args))

        os.environ['YT_TOKEN'] = yt_token

        p = Popen(args, stderr=PIPE, stdout=PIPE)
        out, err = p.communicate()
        status = p.returncode
        logging.info('binary output: %s' % out)
        logging.error('binary error: %s' % err)
        assert status == 0


__TASK__ = YabsYtUpserter
