import time
import logging
import requests
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry

from sandbox import sdk2
from sandbox.projects.common import solomon
from sandbox.common.errors import TaskFailure
from sandbox.common.types.task import Status
from sandbox.common.types.notification import Transport, JugglerStatus

TEMPLATE_ITS_ENABLE_TESTID_URL = '{adminka_url}/api/v1/its/enable_testid/{testid}'
TEMPLATE_ITS_DISABLE_TESTID_URL = '{adminka_url}/api/v1/its/disable_testid/{testid}'
TEMPLATE_ITS_TESTID_STATUS_URL = '{adminka_url}/api/v1/its/testid_status/{testid}'


def requests_retry_session(retries=5, backoff_factor=0.5, status_forcelist=(500, 502, 504)):
    session = requests.Session()
    retry = Retry(
        total=retries,
        read=retries,
        connect=retries,
        backoff_factor=backoff_factor,
        status_forcelist=status_forcelist,
    )
    adapter = HTTPAdapter(max_retries=retry)
    session.mount('https://', adapter)
    return session


def post_request(token, url):
    requests_retry_session().post(url, headers={'Authorization': 'OAuth {}'.format(token)}, timeout=30)


def get_request(token, url):
    return requests_retry_session().get(url, headers={'Authorization': 'OAuth {}'.format(token)}, timeout=30)


class YabsCheckAutoMartyWorkability(sdk2.Task):

    class Requirements(sdk2.Requirements):  # https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
        cores = 1  # 1 cores or less
        ram = 2048  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):
        kill_timeout = 6 * 3600

        experiments = sdk2.parameters.JSON(
            "Test experiment",
            default={'control': 349776, 'experiment': 349778},  # https://st.yandex-team.ru/EXPERIMENTS-68236
            required=True,
        )
        adminka_url = sdk2.parameters.String("Ab adminka url", default="https://ab.test.yandex-team.ru", required=True)
        check_time_timit = sdk2.parameters.Integer("Time limit", default=5 * 3600, required=True)
        yav_secret = sdk2.parameters.YavSecret("Yav secret with ab_token", default="sec-01ftg092w1vcehdmph3b36tmq0", required=True)

    def on_save(self):
        self.Parameters.notifications = [
            sdk2.Notification(
                [Status.FAILURE, Status.Group.BREAK],
                ["host=yabs_ab_sandbox&service=YABS_CHECK_AUTO_MARTY_WORKABILITY"],
                Transport.JUGGLER,
                check_status=JugglerStatus.CRIT
            ),
            sdk2.Notification(
                [Status.SUCCESS],
                ["host=yabs_ab_sandbox&service=YABS_CHECK_AUTO_MARTY_WORKABILITY"],
                Transport.JUGGLER,
                check_status=JugglerStatus.OK
            )
        ]

    def enable_testid(self, testid):
        logging.info('enable testid {}'.format(testid))
        post_request(
            self.Parameters.yav_secret.data()['ab_token'],
            TEMPLATE_ITS_ENABLE_TESTID_URL.format(adminka_url=self.Parameters.adminka_url, testid=testid),
        )

    def disable_testid(self, testid):
        logging.info('disable testid {}'.format(testid))
        post_request(
            self.Parameters.yav_secret.data()['ab_token'],
            TEMPLATE_ITS_DISABLE_TESTID_URL.format(adminka_url=self.Parameters.adminka_url, testid=testid),
        )

    def is_disabled_testid(self, testid):
        logging.info('check testid {}'.format(testid))
        r = get_request(
            self.Parameters.yav_secret.data()['ab_token'],
            TEMPLATE_ITS_TESTID_STATUS_URL.format(adminka_url=self.Parameters.adminka_url, testid=testid),
        )
        j = r.json()
        return j['status'] == 'disabled'

    def run(self, control, experiment):
        self.enable_testid(control)
        self.enable_testid(experiment)

        if self.is_disabled_testid(control) or self.is_disabled_testid(experiment):
            raise Exception('Can not enable experiment and control testids')

        self.Context.success = False
        self.Context.check_start = int(time.time())
        logging.info('check start time: {}'.format(self.Context.check_start))
        while (time.time() - self.Context.check_start) < self.Parameters.check_time_timit:
            if self.is_disabled_testid(experiment):
                self.Context.success = True
                break
            time.sleep(2 * 60)
        self.Context.check_end = int(time.time())
        logging.info('check end time: {}'.format(self.Context.check_end))
        self.Context.check_time = self.Context.check_end - self.Context.check_start

        self.disable_testid(control)
        if not self.Context.success:
            self.disable_testid(experiment)

    def push_results_to_solomon(self):
        sensors = {
            'success': int(self.Context.success),
            'time': self.Context.check_time,
        }
        logging.info('pushing to Solomon: {}'.format(sensors))

        solomon.push_to_solomon_v2(
            token=self.Parameters.yav_secret.data()['solomon_token'],
            params={
                'project': 'bsyeti',
                'cluster': 'sandbox_tasks',
                'service': 'sandbox_tasks',
            },
            sensors=[
                {
                    'labels': {
                        'task': 'YABS_CHECK_AUTO_MARTY_WORKABILITY',
                        'sensor': sensor,
                    },
                    'ts': self.Context.check_start,
                    'value': value,
                }
                for sensor, value in sensors.items()
            ],
        )

    def on_execute(self):
        control = self.Parameters.experiments['control']
        experiment = self.Parameters.experiments['experiment']
        logging.info('control = {}, experiment = {}'.format(control, experiment))

        self.run(control, experiment)
        self.push_results_to_solomon()
        if not self.Context.success:
            raise TaskFailure('Automarty wasnt triggered before timeout')
