import time
import logging
import json
import requests
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry

from sandbox import sdk2
from sandbox.projects.common import solomon
from sandbox.common.types.task import Status
from sandbox.common.types.notification import Transport, JugglerStatus

HOUR = 60 * 60
DAY = 24 * HOUR
CHECK_STEP_TIME_SECONDS = 60
QUEUE_ID = 59

ADMINKA_PRESETS = {
    'prod': {
        'url': 'https://ab.yandex-team.ru',
        'dimension': 4004,
        'aspect': 'automarty',
    },
    'test': {
        'url': 'https://ab.test.yandex-team.ru',
        'dimension': 3994,
        'aspect': 'automarty',
    }
}

CONTROL = json.dumps([{"HANDLER": "BIGB_MAIN", "CONTEXT": {"MAIN": {"BIGB_MAIN": {"BSParameters": [{"Condition": {}, "Settings": {}}]}}}, "TESTID": ["<replace_token>"]}])  # noqa
TEST = json.dumps([{"HANDLER": "BIGB_MAIN", "CONTEXT": {"MAIN": {"BIGB_MAIN": {"BSParameters": [{"Condition": {"PageLabels": ["exp_group_rtb"]}, "Settings": {}}]}}}, "TESTID": ["<replace_token>"]}])  # noqa

TEMPLATE_ITS_ENABLE_TESTID_URL = '{adminka_url}/api/v1/its/enable_testid/{testid}'
TEMPLATE_ITS_DISABLE_TESTID_URL = '{adminka_url}/api/v1/its/disable_testid/{testid}'
TEMPLATE_ITS_TESTID_STATUS_URL = '{adminka_url}/api/v1/its/testid_status/{testid}'
TEMPLATE_CREATE_TESTID = '{adminka_url}/api/testid'
BASE_CREATE_TESTID_DATA = {
    'type': "ABT",
    'queue_id': QUEUE_ID,
    'replace_token': '<replace_token>',
}
TEMPLATE_CREATE_EXPERIMENT = '{adminka_url}/api/task'
BASE_CREATE_EXPERIMENT_DATA = {
    'title': 'Test Automarty Experiment',
    'goal': 'Check Automarty Workability',
    'queue_id': QUEUE_ID,
    'type': 'ABT',
    'mnogomernost': 1,
    'is_technical': 0,
    'staff_only': 0,
    'apriori_bad': 0,
    'has_new_serp_el': 0,
    'salt_dependent': 0,
    'planned_to_production': 1,
    'launch_by_offline': 0,
    'expectations': 'Green metrics',
    'regions': [],
    'state': 'DRAFT',
    'duration': 100,
    'percent': 2,
    'hash_type': 32768,
}
TEMPLATE_SET_TAG_EXPERIMENT = '{adminka_url}/api/tag'
BASE_SET_TAG_EXPERIMENT_DATA = 'name=auto_marty_heartbeat&target_type=0&target_id={task}'
TEMPLATE_TAG_INFO = '{adminka_url}/api/tag/?name=auto_marty_heartbeat'
TEMPLATE_STATE_EXPERIMENT = '{adminka_url}/api/task/{task}'
TEMPLATE_CHECK_EXPERIMENT = '{adminka_url}/api/task/{task}'
TEMPLATE_VERSION_CONFIG = '{adminka_url}/api/config/225/history?action=TAG&tag=production&limit=1'
TEMPLATE_AUTOFILL_CONFIG = '{adminka_url}/api/v1/config/autofill/225'
TEMPLATE_STOP_EXPERIMENT = '{adminka_url}/api/v1/task_change_request/request'
BASE_STOP_EXPERIMENT_DATA = {
    "queue_id": QUEUE_ID,
    "is_removing_testids": True,
    "is_removing_all_testids": True,
}


def log(anything):
    logging.info(anything)
    return anything


def requests_retry_session(retries=5, backoff_factor=0.5, status_forcelist=(500, 502, 504)):
    session = requests.Session()
    retry = Retry(
        total=retries,
        read=retries,
        connect=retries,
        backoff_factor=backoff_factor,
        status_forcelist=status_forcelist,
    )
    adapter = HTTPAdapter(max_retries=retry)
    session.mount('https://', adapter)
    return session


def get_request(token, url):
    return requests_retry_session().get(
        url, headers={'Authorization': 'OAuth {}'.format(token)}, timeout=30)


def post_request(token, url, data=None, is_json=True):
    return requests_retry_session().post(
        url, headers={
            'Authorization': 'OAuth {}'.format(token),
            'Content-Type': 'application/json' if is_json else 'application/x-www-form-urlencoded',
        }, timeout=30, data=data)


def patch_request(token, url, data=None):
    return requests_retry_session().patch(
        url, headers={
            'Authorization': 'OAuth {}'.format(token),
            'Content-Type': 'application/json'
        }, timeout=30, data=data)


class YabsCheckAutoMartyWorkability2(sdk2.Task):

    # https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
    class Requirements(sdk2.Requirements):
        cores = 1  # 1 cores or less
        ram = 2048  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Parameters):
        adminka_preset = sdk2.parameters.String(
            "Adminka preset (prod/test)", default="test", required=True)
        yav_secret = sdk2.parameters.YavSecret(
            "Yav secret with ab & solomon tokens", default="sec-01ftg092w1vcehdmph3b36tmq0", required=True)

    def on_save(self):
        self.Parameters.notifications = [
            sdk2.Notification(
                [Status.FAILURE, Status.Group.BREAK],
                ["host=yabs_ab_sandbox&service=YABS_CHECK_AUTO_MARTY_WORKABILITY_2"],
                Transport.JUGGLER,
                check_status=JugglerStatus.CRIT
            ),
            sdk2.Notification(
                [Status.SUCCESS],
                ["host=yabs_ab_sandbox&service=YABS_CHECK_AUTO_MARTY_WORKABILITY_2"],
                Transport.JUGGLER,
                check_status=JugglerStatus.OK
            )
        ]

    def on_execute(self):
        preset = self.Parameters.adminka_preset
        ab_token = self.Parameters.yav_secret.data()['ab_token']
        solomon_token = self.Parameters.yav_secret.data()['solomon_token']

        clear_its(preset, ab_token)

        experiment = run_experiment(preset, ab_token)
        start, duration, status = wait_for_automarty(
            preset, ab_token, experiment)
        push_results_to_solomon(solomon_token, "automarty_{preset}".format(
            preset=preset), start, duration, status)
        close_experiment(preset, ab_token, experiment)


def wait_for_automarty(preset, token, experiment):
    logging.info('Wait for automarty')
    start = time.time()
    success = False
    while True:
        for testid in experiment['testids']:
            if is_disabled_testid(preset, token, testid):
                success = True
                break
        if success:
            break
        time.sleep(CHECK_STEP_TIME_SECONDS)
    return start, int(time.time() - start), success


def push_results_to_solomon(token, sensor_prefix, start_time, duration, success=True):
    sensors = {
        '{}_success'.format(sensor_prefix): int(success),
        '{}_time_{}'.format(sensor_prefix, ((int(start_time) + 3 * HOUR) // (6 * HOUR)) % 4): duration,
    }
    logging.info('pushing to Solomon: {}'.format(sensors))
    solomon.push_to_solomon_v2(
        token=token,
        params={
            'project': 'bsyeti',
            'cluster': 'sandbox_tasks',
            'service': 'sandbox_tasks',
        },
        sensors=[
            {
                'labels': {
                    'task': 'YABS_CHECK_AUTO_MARTY_WORKABILITY_2',
                    'sensor': sensor,
                },
                'ts': start_time,
                'value': value,
            }
            for sensor, value in sensors.items()
        ],
    )


def is_disabled_testid(preset, token, testid):
    logging.info('Check testid {}'.format(testid))
    result = log(get_request(
        token,
        TEMPLATE_ITS_TESTID_STATUS_URL.format(
            adminka_url=ADMINKA_PRESETS[preset]['url'],
            testid=testid
        ),
    ).json())
    return result['status'] == 'disabled'


def clear_its(preset, token):
    logging.info('Clear its')
    tags = log(get_request(token, TEMPLATE_TAG_INFO.format(
        adminka_url=ADMINKA_PRESETS[preset]['url']))).json()
    for tag in tags:
        if tag['target_type'] != 0:
            continue
        exp = log(get_request(token, TEMPLATE_CHECK_EXPERIMENT.format(
            adminka_url=ADMINKA_PRESETS[preset]['url'], task=tag['target_id']))).json()
        if time.strftime('%Y-%m-%dT%H:%M:%S', time.localtime(time.time() - DAY)) < exp['time_closed']:
            continue
        for testid in exp['testids']:
            if is_disabled_testid(preset, token, testid):
                log(post_request(token, TEMPLATE_ITS_ENABLE_TESTID_URL.format(
                    adminka_url=ADMINKA_PRESETS[preset]['url'], testid=testid)))


def run_experiment(preset, token):
    logging.info('Run Experiment')
    experiment = create_experiment(preset, token)
    change_experiment_state(preset, token, experiment, 'INBOX')
    set_auto_marty_heartbeat_tag(preset, token, experiment)
    change_experiment_state(preset, token, experiment, 'QUEUED')
    change_experiment_state(preset, token, experiment, 'RUNNING')
    return experiment


def set_auto_marty_heartbeat_tag(preset, token, experiment):
    logging.info('Set auto_marty_heartbeat tag')
    return log(post_request(
        token,
        TEMPLATE_SET_TAG_EXPERIMENT.format(
            adminka_url=ADMINKA_PRESETS[preset]['url']
        ),
        data=BASE_SET_TAG_EXPERIMENT_DATA.format(task=experiment['ticket']),
        is_json=False,
    ).json())


def create_experiment(preset, token):
    logging.info('Create Experiment')
    control = create_testid(preset, token, CONTROL, 'control')
    test = create_testid(preset, token, TEST, 'test')

    data = dict(BASE_CREATE_EXPERIMENT_DATA)
    data.update({
        'testids': '{},{}'.format(control['testid'], test['testid']),
        'aspect': ADMINKA_PRESETS[preset]['aspect'],
        'dimention_id': ADMINKA_PRESETS.get(preset, 'test')['dimension'],
    })
    return log(post_request(
        token,
        TEMPLATE_CREATE_EXPERIMENT.format(
            adminka_url=ADMINKA_PRESETS[preset]['url']
        ),
        data=json.dumps(data)
    ).json())


def create_testid(preset, token, params, title):
    logging.info('Create testid')
    data = dict(BASE_CREATE_TESTID_DATA)
    data.update({
        'title': title,
        'params': params,
    })
    return log(post_request(
        token,
        TEMPLATE_CREATE_TESTID.format(
            adminka_url=ADMINKA_PRESETS[preset]['url']
        ),
        data=json.dumps(data),
    ).json())


def change_experiment_state(preset, token, experiment, state):
    logging.info('Change experiment state to {}'.format(state))
    if state == 'RUNNING':
        autofill_config(preset, token)
    elif state == 'STOPPED':
        data = dict(BASE_STOP_EXPERIMENT_DATA)
        data.update({
            'task_ticket': experiment['ticket']
        })
        log(post_request(
            token,
            TEMPLATE_STOP_EXPERIMENT.format(
                adminka_url=ADMINKA_PRESETS[preset]['url']
            ),
            data=json.dumps(data),
        ))
        autofill_config(preset, token)
    else:
        tmp = patch_request(
            token,
            TEMPLATE_STATE_EXPERIMENT.format(
                adminka_url=ADMINKA_PRESETS[preset]['url'],
                task=experiment['ticket'],
            ),
            data=json.dumps({'state': state})
        )
        logging.info(tmp)

    wait_experiment_state(preset, token, experiment, state)


def close_experiment(preset, token, experiment):
    logging.info('Close experiment')
    change_experiment_state(preset, token, experiment, 'STOPPED')
    change_experiment_state(preset, token, experiment, 'CLOSED')


def check_experiment_state(preset, token, experiment, state):
    return get_request(
        token,
        TEMPLATE_CHECK_EXPERIMENT.format(
            adminka_url=ADMINKA_PRESETS[preset]['url'],
            task=experiment['ticket']
        )
    ).json()['state'] == state


def wait_experiment_state(preset, token, experiment, state):
    logging.info('Wait {} experiment state'.format(state))
    while not check_experiment_state(preset, token, experiment, state):
        time.sleep(CHECK_STEP_TIME_SECONDS)


def autofill_config(preset, token):
    logging.info('Autofill config')
    config_version = log(get_request(
        token,
        TEMPLATE_VERSION_CONFIG.format(
            adminka_url=ADMINKA_PRESETS[preset]['url']
        )
    ).json())[0]['version']

    log(post_request(
        token,
        TEMPLATE_AUTOFILL_CONFIG.format(
            adminka_url=ADMINKA_PRESETS[preset]['url']
        ),
        data=json.dumps({
            'version': config_version,
            'set_tag': True,
            'update_from_db': True,
        })
    ).json())
