import logging
import datetime

from sandbox import sdk2
from sandbox.common.types.task import ReleaseStatus
from sandbox.projects.yabs.base_bin_task import BaseBinTask
from sandbox.projects.tank.load_resources.resources import AMMO_FILE


YQL_QUERY = '''
SELECT
    Yson::LookupString(context, "path") as uri
FROM range("logs/deploy-logs/30min", "{range_from}")
where
    project = "about-the-advertiser" and
    stage = "about-the-advertiser" and
    workload = "about_the_advertiser" and
    Yson::Contains(context, "status") and
    Yson::LookupInt64(context, "status") == 200
LIMIT {limit};
'''
TIME_FMT = "%Y-%m-%dT%H:%M:00"


class AboutAdvertiserAmmo(AMMO_FILE):
    pass


class AboutAdvertiserMakeAmmo(BaseBinTask):

    class Requirements(sdk2.Requirements):  # https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
        cores = 1
        ram = 8192  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(BaseBinTask.Parameters):
        resource_attrs = BaseBinTask.Parameters.resource_attrs(default={"task_type": "ABOUT_ADVERTISER_MAKE_AMMO"})
        release_version = BaseBinTask.Parameters.release_version(default=ReleaseStatus.STABLE)
        kill_timeout = 3600

        yt_proxy = sdk2.parameters.String("YT proxy", default="hahn", required=True)
        yt_secret = sdk2.parameters.YavSecret("Secret with YT token", default="sec-01fhtx823mx1gvhc3btd4q6p2k", required=True)
        ammo_limit = sdk2.parameters.Integer("Ammo limit", default=1000, required=True)
        time_limit = sdk2.parameters.Integer("Logs time limit (hours)", default=4, required=True)

    def on_execute(self):
        from yql.api.v1.client import YqlClient
        from yql.client.operation import YqlOperationShareIdRequest

        client = YqlClient(db=self.Parameters.yt_proxy, token=self.Parameters.yt_secret.data()['YT_TOKEN'])
        request = client.query(
            YQL_QUERY.format(
                range_from=(datetime.datetime.now() - datetime.timedelta(hours=self.Parameters.time_limit)).strftime(TIME_FMT),
                limit=self.Parameters.ammo_limit,
            ),
            syntax_version=1,
            title='MAKE AMMO YQL'
        )
        request.run()
        uris = []
        for t in request.get_results():
            t.fetch_full_data()
            for row in t.rows:
                uris.append(row[0])
        logging.info('downloaded')
        if not uris:
            share_req = YqlOperationShareIdRequest(request.operation_id)
            share_req.run()
            error_message = "No data was downloaded\n"
            error_message += "Operation: https://yql.yandex-team.ru/Operations/{}".format(share_req.json)
            raise Exception(error_message)

        logging.info('creating resource AboutAdvertiserAmmo')
        resource = sdk2.ResourceData(AboutAdvertiserAmmo(
            self,
            description="AboutAdvertiser ammo file",
            path='ammo.list',
            size=len(uris),
            ammo_limit=self.Parameters.ammo_limit,
            time_limit=self.Parameters.time_limit,
            ammo_label='about_the_advertiser',
        ))
        with open(str(resource.path), 'w') as f:
            for uri in uris:
                f.write('{}\n'.format(uri))
        resource.ready()
        logging.info('resource created ')
