import os
import time
import json
import logging
import tarfile

from sandbox import sdk2
from sandbox.common.types.task import ReleaseStatus, Status
from sandbox.projects.yabs.base_bin_task import BaseBinTask
from sandbox.projects.common.ya_deploy import release_integration
from sandbox.projects.common import solomon


class YabsAboutAdvertiserBin(sdk2.Resource):
    executable = True
    releasable = True
    ttl = 30


class YabsAboutAdvertiserResourceUpdater(sdk2.Resource):
    executable = True
    releasable = True
    ttl = 30


class YabsAboutAdvertiserResource(sdk2.Resource):
    executable = False
    releasable = True
    ttl = 14
    ver = sdk2.resource.Attributes.String()


class YabsAboutAdvertiserSiteReviews(YabsAboutAdvertiserResource):
    pass


class YabsAboutAdvertiserCaesarData(YabsAboutAdvertiserResource):
    pass


class YabsAboutAdvertiserAntispamData(YabsAboutAdvertiserResource):
    pass


class YabsAboutAdvertiserDirectStats(YabsAboutAdvertiserResource):
    pass


class YabsAboutAdvertiserOrgReviews(YabsAboutAdvertiserResource):
    pass


class YabsAboutAdvertiserAppIds(YabsAboutAdvertiserResource):
    pass


class YabsAboutAdvertiserAppReviews(YabsAboutAdvertiserResource):
    pass


class YabsAboutAdvertiserExceptions(YabsAboutAdvertiserResource):
    pass


def add_row_to_data(table, row, data):
    if isinstance(table.key, str):
        _key, values = row[0], row[1:]
    else:
        keys, values = row[:len(table.key)], row[len(table.key):]
        for k in keys[:-1]:
            data = data.setdefault(k, dict())
        _key = keys[-1]
    if len(table.values) == 1:
        val = values[0]
    else:
        val = {k: v for k, v in zip(table.values, values) if v is not None}
    if not val:
        return
    data[_key] = val


class YabsAboutAdvertiserUpdateResourses(release_integration.ReleaseToYaDeployTask2, BaseBinTask):

    YP_TOKEN_YAV_SECRET_ID = "sec-01fhtx823mx1gvhc3btd4q6p2k"  # robot-about-adtiser secret

    class Requirements(sdk2.Requirements):  # https://wiki.yandex-team.ru/sandbox/clients/#client-tags-multislot
        cores = 1
        ram = 8192  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(BaseBinTask.Parameters):
        resource_attrs = BaseBinTask.Parameters.resource_attrs(default={"task_type": "YABS_ABOUT_ADVERTISER_UPDATE_RESOURSES"})
        release_version = BaseBinTask.Parameters.release_version(default=ReleaseStatus.STABLE)
        kill_timeout = 3600 * 2

        yt_proxy = sdk2.parameters.String("YT proxy", default="arnold", required=True)
        yav_secret = sdk2.parameters.YavSecret("Secret with YT token", default="sec-01fhtx823mx1gvhc3btd4q6p2k", required=True)
        resourse_to_update = sdk2.parameters.String("Resourse to update", required=True)
        result_released_ttl = sdk2.parameters.String(
            label="Result resource ttl after release",
            default_value="14",
        )

    def on_execute(self):
        from yql.api.v1.client import YqlClient
        from yql.client.operation import YqlOperationShareIdRequest

        from yabs.about_the_advertiser.prepare_data_lib import TABLES, get_yql_query, get_table_version

        for table in TABLES:
            if table.name == self.Parameters.resourse_to_update:
                break

        resource_class = globals()[table.cls]
        logging.info('downloading {table.name} {table.path} {table.cls}'.format(table=table))
        data = dict()
        client = YqlClient(db=self.Parameters.yt_proxy, token=self.Parameters.yav_secret.data()['YT_TOKEN'])
        request = client.query(get_yql_query(table), syntax_version=1, title=table.name + ' YQL')
        request.run()
        data = dict()
        for t in request.get_results():
            t.fetch_full_data()
            for row in t.rows:
                add_row_to_data(table, row, data)
        logging.info('downloaded')
        if not data:
            share_req = YqlOperationShareIdRequest(request.operation_id)
            share_req.run()
            error_message = "No data was downloaded\n"
            error_message += "Operation: https://yql.yandex-team.ru/Operations/{}".format(share_req.json)
            raise Exception(error_message)

        file_json = self.Parameters.resourse_to_update + ".json"
        file_tgz = self.Parameters.resourse_to_update + '.tar.gz'
        with open(file_json, "w") as f:
            json.dump(data, f)
        file_size = os.stat(file_json).st_size

        with tarfile.open(file_tgz, "w:gz") as tar:
            tar.add(file_json)

        logging.info('creating resource {}'.format(table.cls))
        resource = resource_class(
            self,
            description="{} {}".format(self.Parameters.resourse_to_update, table.path),
            path='resource.tar.gz',
            _keys=str(table.key),
            _values=str(table.values),
            table_version=get_table_version(table),
            release_version=self.Parameters.release_version,
            _length=len(data),
            _size_bytes=file_size,
        )
        resource_data = sdk2.ResourceData(resource)
        with open(file_tgz, 'rb') as f:
            resource_data.path.write_bytes(f.read())
        resource_data.ready()
        self.Context.resource_id = resource.id
        logging.info('resource {} created: {}'.format(table.cls, self.Context.resource_id))

        solomon.push_to_solomon_v2(
            token=self.Parameters.yav_secret.data()['solomon_token'],
            params={
                'project': 'about_the_advertiser',
                'cluster': 'sandbox_tasks',
                'service': 'sandbox_tasks',
            },
            sensors=[
                {
                    'labels': {
                        'base': self.Parameters.resourse_to_update,
                        'sensor': 'resource_size',
                    },
                    'ts': int(time.time()),
                    'value': file_size,
                }
            ],
        )

    def on_success(self, prev_status):
        AboutAdvertiserResourceRelease(
            self,
            release_type=self.Parameters.release_version,
            task_to_release=sdk2.Task[self.id],
            yav_secret=self.Parameters.yav_secret,
            resourse_type=self.Parameters.resourse_to_update,
            new_resource_id=self.Context.resource_id,
        ).enqueue()

    def on_release(self, additional_parameters):
        super(YabsAboutAdvertiserUpdateResourses, self).on_release(additional_parameters)
        self.mark_released_resources(
            additional_parameters[release_integration.RELEASE_TYPE_KEY],
            ttl=self.Parameters.result_released_ttl,
        )


class AboutAdvertiserResourceRelease(sdk2.Task):
    class Parameters(sdk2.Parameters):
        release_type = sdk2.parameters.String('Release type')
        task_to_release = sdk2.parameters.Task("Task to release")
        kill_timeout = 600
        yav_secret = sdk2.parameters.YavSecret("Secret with YT token", default="sec-01fhtx823mx1gvhc3btd4q6p2k", required=True)
        resourse_type = sdk2.parameters.String("Resourse type to release", required=True)
        new_resource_id = sdk2.parameters.Integer("New resource ID", required=True)
        threshold = sdk2.parameters.Float("Size change threshold", default=0.20)

    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def _validate_resource(self):
        logging.info('Validating resource {}:{}'.format(self.Parameters.resourse_type, self.Parameters.new_resource_id))
        new_resource = sdk2.Resource[self.Parameters.new_resource_id]
        last_released_resource = sdk2.Resource[self.Parameters.resourse_type].find(attrs={'released': 'stable'}).first()
        if getattr(last_released_resource, '_size_bytes', 0) == 0:
            logging.info("Size can not be validated")
            return True
        size_diff = float(new_resource._size_bytes) - float(last_released_resource._size_bytes)
        size_diff /= float(last_released_resource._size_bytes)
        logging.info("Size diff with last released resource #{}: {}".format(last_released_resource.id, size_diff))
        if abs(size_diff) > self.Parameters.threshold:
            logging.info("Validation failed, {} > {}".format(size_diff, self.Parameters.threshold))
            return False
        logging.info("Validated")
        return True

    def on_execute(self):
        logging.info('Waiting parent task to finish')
        with self.memoize_stage.wait_for_parent:
            raise sdk2.WaitTask(
                tasks=[self.Parameters.task_to_release],
                statuses=(Status.Group.FINISH + Status.Group.BREAK),
                wait_all=True
            )

        if self.Parameters.task_to_release.status == Status.SUCCESS:
            validated = self._validate_resource()
            if validated:
                logging.info('Releasing resource')
                self.server.release(
                    task_id=int(self.Parameters.task_to_release),
                    type=self.Parameters.release_type,
                    subject='Automatic release of ' + self.Parameters.description
                )
            solomon.push_to_solomon_v2(
                token=self.Parameters.yav_secret.data()['solomon_token'],
                params={
                    'project': 'about_the_advertiser',
                    'cluster': 'sandbox_tasks',
                    'service': 'sandbox_tasks',
                },
                sensors=[
                    {
                        'labels': {
                            'base': self.Parameters.resourse_type,
                            'sensor': 'resource_validated',
                        },
                        'ts': int(time.time()),
                        'value': int(validated),
                    }
                ],
            )


class UpdateYabsAboutAdvertiserSiteReviews(YabsAboutAdvertiserUpdateResourses):
    class Parameters(YabsAboutAdvertiserUpdateResourses.Parameters):
        resourse_to_update = sdk2.parameters.String("Resourse to update", required=True, default='YABS_ABOUT_ADVERTISER_SITE_REVIEWS')


class UpdateYabsAboutAdvertiserCaesarData(YabsAboutAdvertiserUpdateResourses):
    class Parameters(YabsAboutAdvertiserUpdateResourses.Parameters):
        resourse_to_update = sdk2.parameters.String("Resourse to update", required=True, default='YABS_ABOUT_ADVERTISER_CAESAR_DATA')


class UpdateYabsAboutAdvertiserAntispamData(YabsAboutAdvertiserUpdateResourses):
    class Parameters(YabsAboutAdvertiserUpdateResourses.Parameters):
        resourse_to_update = sdk2.parameters.String("Resourse to update", required=True, default='YABS_ABOUT_ADVERTISER_ANTISPAM_DATA')


class UpdateYabsAboutAdvertiserDirectStats(YabsAboutAdvertiserUpdateResourses):
    class Parameters(YabsAboutAdvertiserUpdateResourses.Parameters):
        resourse_to_update = sdk2.parameters.String("Resourse to update", required=True, default='YABS_ABOUT_ADVERTISER_DIRECT_STATS')


class UpdateYabsAboutAdvertiserOrgReviews(YabsAboutAdvertiserUpdateResourses):
    class Parameters(YabsAboutAdvertiserUpdateResourses.Parameters):
        resourse_to_update = sdk2.parameters.String("Resourse to update", required=True, default='YABS_ABOUT_ADVERTISER_ORG_REVIEWS')


class UpdateYabsAboutAdvertiserAppIds(YabsAboutAdvertiserUpdateResourses):
    class Parameters(YabsAboutAdvertiserUpdateResourses.Parameters):
        resourse_to_update = sdk2.parameters.String("Resourse to update", required=True, default='YABS_ABOUT_ADVERTISER_APP_IDS')


class UpdateYabsAboutAdvertiserAppReviews(YabsAboutAdvertiserUpdateResourses):
    class Parameters(YabsAboutAdvertiserUpdateResourses.Parameters):
        resourse_to_update = sdk2.parameters.String("Resourse to update", required=True, default='YABS_ABOUT_ADVERTISER_APP_REVIEWS')
