import datetime
import os
import logging
from dateutil.tz import tzlocal

from sandbox.common.types.resource import State
from sandbox.sdk2.helpers import subprocess, ProcessLog
from sandbox.sdk2 import (
    Resource,
    ResourceData,
    Task,
    parameters
)

from sandbox.projects.yabs.dropstat.resources import YabsDropStatBinaryPackage


class AdfoxDropStatTask(Task):
    _yt_client = None

    class Parameters(Task.Parameters):
        description = 'Send adfox dropstat table to logbroker'

        with parameters.Group('YT') as yt_params:
            table_path = parameters.String('Table path', required=True)
            work_yt_proxy = parameters.String('Data YT cluster', default='hahn', required=True)
            yt_token_secret = parameters.YavSecret(
                'YT token secret',
                description='Required key: yt_token',
                required=True
            )

        with parameters.Group('Logbroker') as lb_params:
            tvm_src_id = parameters.Integer('TVM source id', default=2018822, required=True)
            tvm_dst_id = parameters.Integer('Logbroker TVM id', default=2001059, required=True)
            tvm_secret = parameters.YavSecret('TVM secret', description='Required key: client_secret',
                                              required=True)

        with parameters.Group('YTPamp') as yt_pump_params:
            partition_count = parameters.Integer('Partitions count (how many session_ids in log)', default=13,
                                                 required=True)

    def on_prepare(self):
        self.yt_token = self.Parameters.yt_token_secret.data()['yt_token']

    def get_last_resource(self, resource_type, attrs_upd=None):
        if attrs_upd is None:
            attrs_upd = {}
        attrs = {'released': 'stable'}
        attrs.update(attrs_upd)
        res = Resource.find(resource_type=resource_type,
                            state=State.READY,
                            owner='YABS_DROP_STAT',
                            attrs=attrs).order(-Resource.id).first()
        return res

    @property
    def yt_client(self):
        if self._yt_client is None:
            import yt.wrapper as yt

            self._yt_client = yt.YtClient(proxy=self.Parameters.work_yt_proxy, token=self.yt_token)

        return self._yt_client

    def get_binary_path(self):
        bin_res = self.get_last_resource(YabsDropStatBinaryPackage)

        logging.info('Using binary resource #%s', bin_res.id)
        return str(ResourceData(bin_res).path)

    def on_execute(self):
        with open('secret', 'w') as f:
            f.write(self.Parameters.tvm_secret.data()['client_secret'])

        debug_path = '//home/yabs/dropstat/adfox/ytpump_debug'
        stats_path = '{}/{}_stats'.format(debug_path, self.id)
        stderr_path = '{}/{}_stderr'.format(debug_path, self.id)

        bin_path = os.path.join(self.get_binary_path(), 'ytpump')
        os.environ['YT_TOKEN'] = self.yt_token
        os.environ['YT_LOG_LEVEL'] = 'INFO'
        command = [
            bin_path,
            '--cluster', self.Parameters.work_yt_proxy,
            '--input', self.Parameters.table_path,
            '--group-by', 'request_session,request_id',
            '--session-by', 'session_id',
            '--ignore-fields', 'session_id,_rest,_stbx,_logfeller_timestamp',
            '--logbroker', 'lbkx.logbroker.yandex.net',
            '--topic', 'yabs-sb/adfox-undoevent-log',
            '--format', 'json',  # ???
            '--chunk-size', 8388608,
            '--tvm-src-id', self.Parameters.tvm_src_id,
            '--tvm-dst-id', self.Parameters.tvm_dst_id,
            '--tvm-secret', 'secret',
            '--partition-count', self.Parameters.partition_count,
            '--do-write', 1,
            '--stats-table', stats_path,
            '--stderr-table', stderr_path,
        ]

        with ProcessLog(self, logger='ytpump') as pl:
            subprocess.check_call(map(str, command), stdout=pl.stdout, stderr=pl.stderr)

        ts = datetime.datetime.now(tzlocal())
        ts += datetime.timedelta(days=10)
        for path in (stats_path, stderr_path):
            self.yt_client.set_attribute(path, 'expiration_time', ts.isoformat())
