import os

from sandbox import sdk2
from sandbox.projects.yabs.base_bin_task import BaseBinTask


REPORT_DEFAULT_TTL = 60


class YabsAnomalyAnalyzerReport(sdk2.Resource):
    """Resource for anomaly analyzer report"""
    releasable = True


class YABS_ANOMALY_ANALYZER(BaseBinTask):
    class Requirements(sdk2.Requirements):
        cores = 2
        ram = 8072
        disk_space = 8072

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(BaseBinTask.Parameters):
        description = 'Launch anomaly analyzer'

        with BaseBinTask.Parameters.version_and_task_resource() as version_and_task_resource:
            resource_attrs = sdk2.parameters.Dict('Filter resource by', default={'name': 'YabsAnomalyAnalyzer'})

        config_path = sdk2.parameters.String(
            'Path to config in arcadia like (yabs/analytics/anomaly_analyzer/config/...)',
            required=True
        )
        analyzer_name = sdk2.parameters.String(
            'Analyzer name',
            required=True
        )
        start_hour = sdk2.parameters.String(
            'Analysis start hour in "%Y-%m-%dT%H:00:00" format',
        )
        end_hour = sdk2.parameters.String(
            'Analysis end hour in "%Y-%m-%dT%H:00:00" format',
        )
        analyze_last_hour = sdk2.parameters.Bool(
            'Analyze last hour of time series',
            default=False
        )
        data_dir = sdk2.parameters.String(
            'Path to analyzer data directory on YT',
            required=True
        )
        model_name = sdk2.parameters.String(
            'Analyser model name',
            default='wow_std_diff'
        )
        disp_type = sdk2.parameters.String(
            'Analyser disp type',
            default='prev_week'
        )
        treshholds_multiplier = sdk2.parameters.Float(
            'Analyzer treshholds multiplier',
            default=1.
        )
        timestamp = sdk2.parameters.String(
            'Timestamp',
            default='1'
        )
        secret_name = sdk2.parameters.YavSecret(
            'yav secret id',
            default='sec-01dce6wh83dmf0t05gfj0amdhx'
        )
        alert_ymessenger_chats = sdk2.parameters.List(
            'Alert ymessenger chats to report anomalies',
            default=[]
        )
        alert_telegram_chats = sdk2.parameters.List(
            'Alert telegram chats to report anomalies',
            default=[]
        )
        prestable_services = sdk2.parameters.List(
            'Services to be treated as prestable',
            default=[]
        )
        stable_services = sdk2.parameters.List(
            'Services to be treated as stable',
            default=[]
        )
        proxy = sdk2.parameters.String(
            'YT cluster name',
            default='hahn'
        )
        debug = sdk2.parameters.Bool(
            'Disable pushing alerts to juggler',
            default=False
        )
        report = sdk2.parameters.Bool(
            'Generate html report (as sandbox resource)',
            default=False
        )
        yql_memory_limit = sdk2.parameters.Integer(
            'Memory limit for yql operations in GB',
            default=3
        )
        ttl_days = sdk2.parameters.Integer(
            'TTL limit for output tables in days',
            default=180
        )
        yt_pool = sdk2.parameters.String(
            'YT pool'
        )

    def on_execute(self):
        from yabs.analytics.anomaly_analyzer.src import run_anomaly_analyzer

        config_path = os.path.join(os.getcwd(), os.path.basename('config'))
        sdk2.svn.Arcadia.export(sdk2.svn.Arcadia.trunk_url(self.Parameters.config_path), config_path)

        config = {
            'arcadia_path': config_path,
            'analyzer_name': self.Parameters.analyzer_name,
            'analyse_start_hour': self.Parameters.start_hour,
            'analyse_end_hour': self.Parameters.end_hour,
            'analyze_last_hour': self.Parameters.analyze_last_hour,
            'analyser_data_dir': self.Parameters.data_dir.rstrip('/'),
            'model_name': self.Parameters.model_name,
            'disp_type': self.Parameters.disp_type,
            'treshholds_multiplier': self.Parameters.treshholds_multiplier,
            'timestamp': self.Parameters.timestamp,
            'yt_token': self.Parameters.secret_name.data()['yt_token'],
            'yql_token': self.Parameters.secret_name.data()['yql_token'],
            'solomon_token': self.Parameters.secret_name.data()['solomon_token'],
            'ymessenger_token': self.Parameters.secret_name.data()['yandex_messager_token'],
            'telegram_token': self.Parameters.secret_name.data()['telegram_token'],
            'alert_ymessenger_chats': self.Parameters.alert_ymessenger_chats,
            'alert_telegram_chats': self.Parameters.alert_telegram_chats,
            'prestable_services': self.Parameters.prestable_services,
            'stable_services': self.Parameters.stable_services,
            'proxy': self.Parameters.proxy,
            'debug': self.Parameters.debug,
            'report': self.Parameters.report,
            'yql_memory_limit': self.Parameters.yql_memory_limit,
            'ttl_days': self.Parameters.ttl_days,
            'yt_pool': self.Parameters.yt_pool
        }

        run_anomaly_analyzer(config)

        if self.Parameters.report:
            report_resource = YabsAnomalyAnalyzerReport(
                self,
                description="Anomaly analyzer HTML report",
                path="report",
                ttl=REPORT_DEFAULT_TTL
            )
            sdk2.ResourceData(report_resource).ready()
