# -*- coding: utf-8 -*-

import time
import json


class TrafficBackToBack():

    def __init__(
        self, artmon_token, time_difference_threshold,
        artmon_url, resource_name, repetitions_number
    ):
        from yabs.artmon.pylibs.client import ArtmonClient

        self.artmon_client = ArtmonClient(token=artmon_token)
        self.time_difference_threshold = time_difference_threshold
        self.artmon_url = artmon_url
        self.resource_name = resource_name
        self.repetitions_number = repetitions_number

    def b2b(self, first, second):
        import logging

        path = self.__get_resource_path()
        logging.info(path)
        requests = open(str(path), 'r')
        text = []
        for url in requests:
            if url.strip() == "" or url.startswith("##"):
                continue
            if url.startswith("#"):
                request_name = url.strip("#").strip(" ")
                continue
            if 'module=Traffic2' not in url:
                continue

            first_res = self.__get_information(url, first[0], first[1])
            first_msg = first_res.print_attrs()

            second_res = self.__get_information(url, second[0], second[1])
            second_msg = second_res.print_attrs()

            verdict = self.__compare_results(first_res, second_res)

            msg = '<{{{verdict}\n%%\n{name}\n{first_msg}\n{second_msg}\n{url}\n%%}}>'.format(
                first_msg=first_msg,
                second_msg=second_msg,
                verdict=verdict,
                name=request_name,
                url=url
            )
            text.append(msg)
            logging.info('{} processed'.format(request_name))
        return text

    def __get_resource_path(self):
        import sandbox.common.types.resource as ctr
        from sandbox import sdk2

        path = sdk2.ResourceData(
            sdk2.Resource[self.resource_name].find(state=ctr.State.READY).first()
        ).path
        return path

    def __get_information(self, url, module, module_type):
        traffic_2 = TrafficStat('{} {}'.format(module, module_type))
        for _ in range(self.repetitions_number):
            start_time = time.time()
            json_data = self.__get_json_from_url(url, module, module_type)
            work_time = (time.time() - start_time)
            traffic_2.fill_attrs(json_data, work_time)
        return traffic_2

    def __get_json_from_url(self, url, module, module_type):
        from yabs.artmon.pylibs.request import ArtmonRequest
        from yabs.artmon.pylibs.request import ArtmonHandler

        base_url = self.artmon_url
        if module_type == 'pre-prod':
            base_url = base_url.replace('bsadm', 'bsadm-pre')
        if module == 'Traffic3':
            url = url.replace('module=Traffic2', 'module=Traffic3')

        request = ArtmonRequest.from_url(
            url=url,
            handler=ArtmonHandler.DATA,
            strip_time_params=False,
            base_url=base_url
        )
        data = self.artmon_client.get_data(request)
        return json.loads(data)

    def __compare_results(self, first, second):
        if (first.shows == second.shows) and (first.clicks == second.clicks) and (int(first.cost) == int(second.cost)):
            verdict = 'OK.'
        else:
            verdict = 'Error, different results.'
        if (abs(first.time - second.time) / ((first.time + second.time) / 2)) > self.time_difference_threshold:
            time_verdict = 'Big time difference.'
        else:
            time_verdict = ''
        return '{} {}\n'.format(verdict, time_verdict)


class TrafficStat():
    def __init__(self, name):
        self.name = name
        self.shows = 0
        self.clicks = 0
        self.cost = 0
        self.avg_time = 0
        self.min_time = float('Inf')
        self.max_time = 0
        self.time = 0
        self.iteration = 0

    def fill_attrs(self, json_data, work_time):
        self.iteration += 1
        self.time += work_time
        self.min_time = min(self.min_time, work_time)
        self.max_time = max(self.max_time, work_time)
        self.avg_time = self.time / self.iteration
        for point in json_data['items']['rows']:
            if 'frac' in point.get('series_id', ''):
                continue
            if point.get('shows', False):
                self.shows += point['shows']
            elif point.get('clicks', False):
                self.clicks += point['clicks']
            elif point.get('cost', False):
                self.cost += point['cost']

    def print_attrs(self):
        from prettytable import PrettyTable

        pt = PrettyTable()
        pt.field_names = ['Name', 'Shows', 'Clicks', 'Cost', 'Avg time', 'Min time', 'Max time']
        pt.add_row([
            self.name,
            str(self.shows),
            str(self.clicks),
            str(self.cost),
            str(self.avg_time),
            str(self.min_time),
            str(self.max_time)
        ])
        table = str(pt)
        return table
