# -*- coding: utf-8 -*-
import json
import logging
from datetime import datetime

import pytz

ISSUE_REPORT_TEMPLATE = """
{stats_table}

See (({report_link} full report)).

Created by (({task_link} {task_type})).
"""
ISSUE_DESCRIPTION_TEMPLATE = u"""
Что-то пошло не так и на некоторых хостах обнаружены файлы, не прошедшие релизный процесс.
Подробнее можно почитать ((https://wiki.yandex-team.ru/BannernajaKrutilka/AuditRelease/ тут)).

{report_text}
"""

logger = logging.getLogger(__name__)


def create_issue_description(report_text):
    return ISSUE_DESCRIPTION_TEMPLATE.format(report_text=report_text)


def create_issue_report(report_link, stats_table, task_link, task_type):
    return ISSUE_REPORT_TEMPLATE.format(
        report_link=report_link,
        task_link=task_link,
        task_type=task_type,
        stats_table=stats_table,
    )


def create_or_update_issue(startrek_client, report_text, fails, date_tag=None, component_name=None, queue="BSAUDIT", assignee=None, followers=[]):
    if date_tag is None:
        today = datetime.now(tz=pytz.timezone("Europe/Moscow")).date()
        date_tag = today.strftime("%Y-%m-%d")

    issue_tags = [
        "audit_release",
        date_tag,
    ]
    if component_name is not None:
        issue_tags.append(component_name)

    search_query = " AND ".join([
        "Tags: {}".format(tag) for tag in issue_tags
    ] + [
        "Queue: {}".format(queue),
        "Resolution: empty()",
    ])
    created_issues = list(startrek_client.issues.find(search_query))
    if not created_issues:
        with open("audit_result.json", "w") as result_file:
            json.dump(fails, result_file, indent=2)
        return startrek_client.issues.create(
            queue=queue,
            summary="AuditRelease {component_name} {date_tag}".format(component_name=component_name, date_tag=date_tag),
            description=create_issue_description(report_text),
            assignee=assignee,
            tags=issue_tags,
            attachments=["audit_result.json"],
            followers=followers
        )
    issue = created_issues[0]
    state_changed = True
    state_similar_to = None

    for attachment in issue.attachments:
        attachment.download_to(".")
        with open(attachment.name, "r") as old_audit_result_file:
            old_fails = json.load(old_audit_result_file)
        if old_fails == fails:
            logger.info("Current state similar to %s created at %s", attachment.id, attachment.createdAt)
            state_similar_to = attachment.createdAt
            state_changed = False
            break

    if state_changed:
        with open("audit_result.json", "w") as result_file:
            json.dump(fails, result_file, indent=2)
        issue.comments.create(text=report_text, attachments=["audit_result.json"])
    else:
        issue.comments.create(text="Состояние тикета не изменилось с {}".format(state_similar_to))
        logger.info("Nothing was changed since %s check", state_similar_to)
    return issue
