import textwrap
from collections import defaultdict

from jinja2 import Environment


TH_WIDTH_PX = TD_WIDTH_PX = 30
TH_HEIGHT_PX = 200
HORIZONTAL_TD_PADDING_PX = HORIZONTAL_TH_PADDING_PX = 8
VERTICAL_TD_PADDING_PX = VERTICAL_TH_PADDING_PX = 4
FONT_SIZE_PX = 14


FAILS_REPORT_STYLE = {
    ".report": {
        "font-family": "Helvetica, Arial, sans-serif",
    },
    "table.report": {
        "border-collapse": "collapse",
    },
    "th": {
        "text-align": "left",
        "vertical-align": "bottom",
        "padding": "{}px {}px".format(VERTICAL_TD_PADDING_PX, HORIZONTAL_TD_PADDING_PX),
    },
    ".report th > div": {
        "width": "{}px".format(TH_WIDTH_PX),
    },
    ".report td > div": {
        "width": "{}px".format(TD_WIDTH_PX),
    },
    "span.rotated": {
        "border-bottom": "1px solid #ccc",
        "padding": "{}px {}px".format(VERTICAL_TD_PADDING_PX, HORIZONTAL_TD_PADDING_PX),
    },
    "span.status": {
        "font-weight": "700",
    },
    "span.status-FAIL": {
        "color": "red",
    },
    "span.status-OK": {
        "color": "green",
    },
    "td": {
        "border": "1px solid #ddd",
        "width": "{}px".format(TD_WIDTH_PX),
        "padding": "{}px {}px".format(VERTICAL_TD_PADDING_PX, HORIZONTAL_TD_PADDING_PX),
    },
    "tr:nth-child(even)": {
        "background-color": "#f2f2f2",
    },
}
RENDERED_FAILS_REPORT_STYLE = "\n".join(
    "{selector} {{\n{style}\n}}".format(
        selector=selector,
        style="\n".join(["{}: {};".format(key, value) for key, value in style.items()])
    ) for selector, style in FAILS_REPORT_STYLE.items()
)
FAILS_REPORT_TEMPLATE = '''
<html>
    <head>
        <style>
        {{ style }}
        </style>
    </head>
    <body>
        This report was created by <a href="{{ task_link }}" target="_blank">{{ task_type }}</a>.
        <h3>Monitoring fails by host</3>
        <table class="report">
            <thead>
                <th>Service</th>
                <th>Hostname FQDN</th>
                {% for file_name in file_names %}
                <th>
                    <div>{{ file_name|replace("-", "&#8209;") }}</div>
                </th>
                {% endfor %}
            </thead>
            <tbody>
            {% for service, host_problems in problems_by_service.items() %}
                {% for host, files in host_problems.items() %}
                <tr>
                    {% if loop.first %}
                    <td rowspan={{ loop.length }}><a href="https://nanny.yandex-team.ru/ui/#/services/catalog/{{ service }}/">{{ service }}</a></td>
                    {% endif %}
                    <td><pre>{{ host }}</pre></td>
                    {% for file_name in file_names -%}
                    <td>{% if file_name in files %}<span class="status status-FAIL">{{ files[file_name] }}</span>{% else %}<span class="status status-OK">OK</span>{% endif %}</td>
                    {%- endfor %}
                </tr>
                {% endfor %}
            {% endfor %}
            </tbody>
        </table>
    <body>
</html>
'''
SB_STYLE = '''
.table {
    white-space: normal;
}
.table table {
    border-collapse: collapse;
}
.table th {
    padding: 2px 8px;
    text-align: left;
    vertical-align: top;
    background-color: #f2f2f2;
    border: 1px solid #ddd;
}
.table tr {
    padding: 2px 8px;
    text-align: left;
    vertical-align: top;
}
.table td {
    padding: 2px 8px;
    border: 1px solid #ddd;
}
.table tr:nth-child(even) {
    background-color: #f2f2f2;
}
'''
APPROVED_FILES_REPORT_TEMPLATE = '''
<style>
{{ style }}
</style>
<div class="table">
    <h3>Approved releases</h3>
    <table>
        <theader>
            <tr>
                <th>Version</th>
                {% for file_name in file_names %}
                <th>{{ file_name }}</th>
                {%endfor%}
            </tr>
        </theader>
        <tbody>
        {% for (task_type, version, patch), md5 in approved_files_by_version %}
            <tr>
                <td>{{ version }} {{ "(patch: {})".format(patch) if patch else "" }}</td>
                {% for file_name in file_names %}
                <td>{{ md5[file_name] }}</td>
                {%endfor%}
            </tr>
        {% endfor %}
        </tbody>
    </table>
</div>
'''
SB_STATISTICS_REPORT_TEMPLATE = textwrap.dedent('''
<style>
{{ style }}
</style>
<div class="table">
    <h3>Monitoring results</h3>
    <table>
        <theader>
            <tr>
                <th>service</th>
                {% for file_name in file_names %}
                <th>{{ file_name }}</th>
                {% endfor %}
            </tr>
        </theader>
        <tbody>
        {% for service_name, stats in service_results.items() %}
            <tr>
                <td>
                    {{ service_name }}
                </td>
                {% for file_name in file_names %}
                <td>
                    {% for status, count in stats[file_name].items() -%}
                    <span style="color: {{ color_map[status] }}">{{ count }}</span>{% if not loop.last%}/{% endif %}
                    {%- endfor %}
                </td>
                {% endfor %}
            </tr>
        {% endfor %}
        </tbody>
        <tfooter>
            <td><b>Total</b></td>
            {% for file_name in file_names %}
                <td>
                    {% for status, count in file_results[file_name].items() -%}
                    <span style="color: {{ color_map[status] }}">{{ count }}</span>{% if not loop.last%}/{% endif %}
                    {%- endfor %}
                </td>
            {% endfor %}
        </tfooter>
    </table>
</div>
''')
ST_STATISTICS_REPORT_TEMPLATE = textwrap.dedent('''
=== Monitoring results
#|
|| **service** {% for file_name in file_names %}| **{{ file_name }}** {% endfor %}||
{% for service_name, stats in service_results.items() %}
|| ((https://nanny.yandex-team.ru/ui/#/services/catalog/{{ service_name }} {{ service_name }}))
{% for file_name in file_names -%}| {% for status, count in stats[file_name].items() -%} !!({{ color_map[status] }}){{ count }}!!{% if not loop.last %}/{% endif %}{%- endfor %}
{%- endfor %}
||
{% endfor %}
|| **Total** {% for file_name in file_names -%}
    | {% for status, count in file_results[file_name].items() -%}
    !!({{ color_map[status] }}){{ status }}!!:{{ count }}{% if not loop.last %} / {% endif %}
    {%- endfor %}
{%- endfor %}||
|#
''')


def create_approved_files_report(approved_files):
    approved_files_by_version = defaultdict(dict)
    for filename, md5_by_version in approved_files.items():
        for version, md5 in md5_by_version.items():
            approved_files_by_version[version][filename] = md5

    sorted_approved_files_by_version = []
    for version in reversed(sorted(approved_files_by_version.keys())):
        sorted_approved_files_by_version.append((version, approved_files_by_version[version]))

    env = Environment()
    template = env.from_string(APPROVED_FILES_REPORT_TEMPLATE)
    return template.render(
        file_names=approved_files.keys(),
        approved_files_by_version=sorted_approved_files_by_version,
    )


def create_statistics_report(stats, auditable_binaries, status_enum, color_map, template):
    status_list = [status.value for status in status_enum]
    file_results = {
        filename: defaultdict(int) for filename in auditable_binaries
    }
    service_results = {}

    for (service, filename, status), count in stats.items():
        file_results[filename][status.value] += count
        if service not in service_results:
            service_results[service] = {
                filename: defaultdict(int) for filename in auditable_binaries
            }
        service_results[service][filename][status.value] += count

    env = Environment()
    template = env.from_string(template)
    return template.render(
        file_results=file_results,
        service_results=service_results,
        statuses=status_list,
        file_names=auditable_binaries,
        color_map=color_map,
        style=SB_STYLE,
    )


def create_fails_report(problems_by_service, file_names, task_link, task_type):
    env = Environment()
    template = env.from_string(FAILS_REPORT_TEMPLATE)
    return template.render(problems_by_service=problems_by_service, file_names=file_names, task_link=task_link, task_type=task_type, style=RENDERED_FAILS_REPORT_STYLE)


def create_st_table_report(stats, auditable_binaries, status_enum, color_map):
    return create_statistics_report(stats, auditable_binaries, status_enum, color_map, ST_STATISTICS_REPORT_TEMPLATE)


def create_sb_table_report(stats, auditable_binaries, status_enum, color_map):
    return create_statistics_report(stats, auditable_binaries, status_enum, color_map, SB_STATISTICS_REPORT_TEMPLATE)
