import logging
import requests
import time
from collections import defaultdict
from itertools import cycle

from sandbox.projects.common.nanny.client import NannyApiException


logger = logging.getLogger(__name__)


BALANCERS_BY_PRJ = {
    "bs": ["admin-yabsl7balancer1.n.yandex-team.ru", "admin-yabsl7balancer2.n.yandex-team.ru"],
    "yabs": ["admin-yabsl7balancer3.n.yandex-team.ru", "admin-yabsl7balancer4.n.yandex-team.ru"],
    "bsrank": ["admin-yabsl7balancer5.n.yandex-team.ru", "admin-yabsl7balancer6.n.yandex-team.ru"],
    "metadsp": ["admin-yabsl7balancer14.n.yandex-team.ru", "admin-yabsl7balancer15.n.yandex-team.ru"],
}
EXCLUDE_SERVICES = [
]


def get_services(nanny_client, dashboard_id, filter_by_labels=(), exclude_services=(), use_conjunction=True):
    service_ids = nanny_client.get_dashboard_services(dashboard_id)
    if not isinstance(service_ids, list) or len(service_ids) == 0:
        logger.error("Expected non-empty list of service_ids, got this: %s", service_ids)
        raise NannyApiException("Expected non-empty list of service_ids")

    services = []
    for service_id in service_ids:
        if service_id in exclude_services:
            continue
        service_info = nanny_client.get_service(service_id)
        service_labels = [(label['key'], label.get('value')) for label in service_info['info_attrs']['content']['labels']]

        filter_function = all if use_conjunction else any
        if filter_function([
            label_key_value in service_labels
            for label_key_value in filter_by_labels
        ]):
            services.append(service_id)

    return services


def get_hosts_info(balancer_hostnames, retries=10, sleep_period=1):
    attempt = 0
    iter_balancer_hostnames = cycle(balancer_hostnames)
    last_exc = None
    while attempt < retries:
        url = "https://{}/hosts_info".format(iter_balancer_hostnames.next())
        try:
            r = requests.get(url)
            return r.json()["result"]
        except Exception as exc:
            logger.error("Got exception while trying to get hosts info from %s", url, exc_info=True)
            last_exc = exc
        attempt += 1
        time.sleep(sleep_period)
        sleep_period *= 2

    raise last_exc


def get_enabled_hosts_by_service(balancer_hostnames, services):
    results = get_hosts_info(balancer_hostnames)

    enabled_hosts_by_service = defaultdict(list)
    for result in results:
        if result["enabled"] and result["weight"] > 0 and result["group"] in services:
            enabled_hosts_by_service[result["group"]].append((result["host"], result["port"]))

    return enabled_hosts_by_service
