import logging
from abc import ABCMeta, abstractmethod

from .startrek import get_release_acceptor, get_release_requester
from .sandbox import get_task_releaser


logger = logging.getLogger(__name__)


class BaseRequirement(object):
    __metaclass__ = ABCMeta

    def __init__(self, expected=None):
        self.expected = expected

    @abstractmethod
    def check():
        pass


class StartrekRequirement(BaseRequirement):
    pass


class BuildTaskRequirement(BaseRequirement):
    pass


class IssueQueueRequirement(StartrekRequirement):
    def check(self, issue=None):
        if issue is None:
            return False, "Cannot find issue associated with release"
        return (
            issue.queue.key in self.expected,
            "Issue {issue} has incorrect queue {actual}, expected one of {expected}".format(
                issue=issue.key,
                actual=issue.queue.key,
                expected=self.expected,
            )
        )


class IssueTypeRequirement(StartrekRequirement):
    def check(self, issue=None):
        if issue is None:
            return False, "Cannot find issue associated with release"
        return (
            issue.type.key in self.expected,
            "Issue {issue} has incorrect type {actual}, expected one of {expected}".format(
                issue=issue.key,
                actual=issue.type.key,
                expected=self.expected,
            )
        )


class IssueAcceptedRequirement(StartrekRequirement):
    def check(self, issue=None):
        if issue is None:
            return False, "Cannot find issue associated with release"
        release_acceptor = get_release_acceptor(issue)
        return (
            release_acceptor is not None,
            "Issue {issue} was not accepted".format(
                issue=issue.key,
            )
        )


class IssueAcceptedNotByAuthorRequirement(StartrekRequirement):
    def check(self, issue=None):
        if issue is None:
            return False, "Cannot find issue associated with release"
        release_acceptor = get_release_acceptor(issue)
        release_requester = get_release_requester(issue)
        logger.info("IssueAcceptedNotByAuthorRequirement: release_acceptor={}, release_requester={}".format(release_acceptor, release_requester))
        approvers = set(filter(None, [release_acceptor, release_requester]))
        return (
            len(approvers) >= self.expected,
            "Release was requested by {release_requester}, accepted by {release_acceptor}. Requirement of {expected} number of people was NOT satisfied (actual={actual})".format(
                release_requester=release_requester,
                release_acceptor=release_acceptor,
                expected=self.expected,
                actual=len(approvers)
            )
        )


class BuildTaskReleasedRequirement(BuildTaskRequirement):
    def check(self, task_id=None, sandbox_client=None):
        release_type = self.expected
        task_releaser = get_task_releaser(task_id, sandbox_client, release_type)
        return (
            task_releaser is not None,
            "Task {task_id} was not released to {release_type}".format(
                task_id=task_id,
                release_type=release_type,
            )
        )


class BuildTaskTypeRequirement(BuildTaskRequirement):
    def check(self, task_id=None, sandbox_client=None):
        actual = sandbox_client.task[task_id].read()['type']
        return (
            actual in self.expected,
            "Task {task_id} has type {actual}, expected one of {expected}".format(
                task_id=task_id,
                actual=actual,
                expected=self.expected,
            )
        )


class ApproversCountRequirement(BaseRequirement):
    def check(self, issue=None, task_id=None, sandbox_client=None, **kwargs):
        if issue is None:
            return False, "Cannot find issue associated with release"
        release_acceptor = get_release_acceptor(issue)
        task_releaser = get_task_releaser(task_id, sandbox_client, 'testing')
        logger.info("release_acceptor={}, task_releaser={}".format(release_acceptor, task_releaser))
        approvers = set(filter(None, [release_acceptor, task_releaser]))
        return (
            len(approvers) >= self.expected,
            "Resource built by {task_id} has {actual} approvers: {approvers}, but expected at least {expected}".format(
                task_id=task_id,
                actual=len(approvers),
                approvers=approvers,
                expected=self.expected,
            )
        )


def check_requirements(requirements, issue, build_task_id, sandbox_client):
    success = True
    errors = []
    for requirement in requirements:
        if isinstance(requirement, StartrekRequirement):
            result, error = requirement.check(issue=issue)
        elif isinstance(requirement, BuildTaskRequirement):
            result, error = requirement.check(task_id=build_task_id, sandbox_client=sandbox_client)
        else:
            result, error = requirement.check(issue=issue, task_id=build_task_id, sandbox_client=sandbox_client)

        success = success and result
        if not result:
            errors.append(error)
    return success, errors
