import logging
from collections import namedtuple

import requests

from enum import Enum
from concurrent.futures import ThreadPoolExecutor, as_completed
from retrying import retry

logger = logging.getLogger(__name__)


EngineRole = namedtuple("EngineRole", ("prj", "port"))
HostStatus = namedtuple("HostStatus", ("engine_is_running", "md5", "exception"))
HostStatus.__new__.__defaults__ = (None, )


class EngineRoleEnum(Enum):
    bs = EngineRole("bs", 8001)
    metadsp = EngineRole("metadsp", 8001)
    bsrank = EngineRole("bsrank", 9001)
    yabs = EngineRole("yabs", 10001)


def yabs_server_is_running(host, server_port, timeout=5):
    try:
        r = requests.get("http://{}:{}/ping".format(host, server_port), timeout=timeout)
        logger.debug("Got ping response %d from %s:\n%s", r.status_code, r.url, r.content)
        r.raise_for_status()
        return True
    except requests.ConnectionError:
        return False
    except Exception as exc:
        logger.debug("Got exception %s while trying to ping %s:%s", exc, host, server_port)
        return False


def yabs_server_role_is_running(host, role, timeout=5):
    return yabs_server_is_running(host, role.value.port, timeout)


def detect_yabs_server_role(host, retries=3, timeout=5):
    for attempt in range(retries):
        for role in EngineRoleEnum:
            if yabs_server_role_is_running(host, role, timeout=timeout):
                logger.debug("Host '%s' has running instance of '%s'", host, role.name)
                return role
    return None


@retry(
    stop_max_delay=60 * 2 * 1000,
    wait_exponential_multiplier=2 * 1000,
    retry_on_exception=lambda exc: isinstance(exc, (requests.ConnectionError, requests.ReadTimeout)),
)
def get_hashes(host, port, timeout=120):
    r = requests.get("http://{}:{}/md5".format(host, port), timeout=timeout)
    r.raise_for_status()
    return r.json()


def check_host(host, port, engine_port=None, timeout=120):
    if engine_port is not None:
        role_desc = 'port {}'.format(engine_port)
        if not yabs_server_is_running(host, engine_port, timeout=timeout):
            return HostStatus(False, None)
    else:
        role = detect_yabs_server_role(host, timeout=timeout)
        if not role:
            return HostStatus(False, None)
        role_desc = role.name

    try:
        md5 = get_hashes(host, port, timeout=timeout)
    except Exception as e:
        logger.error("Cannot get md5 info from %s at %s", role_desc, host, exc_info=True)
        return HostStatus(True, None, e)

    return HostStatus(True, md5)


def iter_binary_hashes(host_roles, port, timeout=120):
    futures_map = {}
    with ThreadPoolExecutor(max_workers=32) as pool:
        for host, engine_port in host_roles:
            futures_map[pool.submit(check_host, host, port, engine_port, timeout)] = (host, engine_port)

        for future in as_completed(futures_map):
            host, engine_port = futures_map[future]
            yield host, engine_port, future.result()
