# -*- coding: utf-8 -*-

import logging
import traceback

from sandbox import sdk2


class TicketError(Exception):
    pass


class BaseAutoSupBsIssueProcessor(sdk2.Task):

    class Parameters(sdk2.Parameters):
        startrek_token = sdk2.parameters.String(
            'startrek token sandbox vault name',
            default='ROBOT_AUTOSUPBS_ST_TOKEN',
            required=True
        )

        tickets = sdk2.parameters.List(
            'tickets to be processed',
            default=['SUPBS-7423'],
            required=True
        )

        startrek_api_url = sdk2.parameters.String(
            'startrek api url',
            default='https://st-api.yandex-team.ru',
            required=True
        )

        remove_tag_after_processing = sdk2.parameters.Bool(
            'Remove tag after processing',
            default=False,
        )

        with remove_tag_after_processing.value[True]:
            tag_to_remove = sdk2.parameters.String(
                "Tag to remove after processing",
            )

        add_tag_after_processing = sdk2.parameters.Bool(
            'Add new tag after processing',
            default=False,
        )

        with add_tag_after_processing.value[True]:
            tag_to_add = sdk2.parameters.String(
                "Tag to add after processing",
            )

        ticket_example = sdk2.parameters.String(
            'example of correct parameters filling',
            default='',
            required=True
        )

        summon_assignee = sdk2.parameters.Bool(
            'Summon assignee after processing',
            default=False,
        )

    def on_execute(self):
        from startrek_client import Startrek

        startrek_token = sdk2.Vault.data(self.owner, self.Parameters.startrek_token)
        tickets = self.Parameters.tickets
        startrek_api_url = self.Parameters.startrek_api_url
        remove_tag_after_processing = self.Parameters.remove_tag_after_processing
        add_tag_after_processing = self.Parameters.add_tag_after_processing
        tag_to_add = self.Parameters.tag_to_add
        tag_to_remove = self.Parameters.tag_to_remove
        ticket_example = self.Parameters.ticket_example
        summon_assignee = self.Parameters.summon_assignee
        if self.Context.ticket:
            logging.info('Restart ticket: {}'.format(self.Context.ticket))
            tickets = [self.Context.ticket]

        logging.info("Common class tickets: {}".format(tickets))

        startrek = Startrek(
            useragent='autosupbs',
            base_url=startrek_api_url,
            token=startrek_token,
        )
        for ticket in tickets:
            logging.info('Ticket: {}'.format(ticket))
            issue = startrek.issues[ticket]
            comments = []
            summonees = []

            ticket_error_occured = False

            if summon_assignee and issue.assignee:
                summonees.append(issue.assignee)
            try:
                self.process_issue(issue, comments)
                logging.info('Processing {issue} completed succesfully'.format(issue=issue.key))
                for comment in comments:
                    comment_text = '\n'.join(comment)
                    issue.comments.create(text=comment_text.decode('unicode-escape'), summonees=summonees)
                logging.info('Comment created correctly at {issue}'.format(issue=issue.key))
            except TicketError as e:
                logging.error('ERROR: TicketError at {issue}:\n{e}\n{tb}'.format(
                    issue=issue.key,
                    e=str(e),
                    tb=traceback.format_exc()
                ))
                issue.comments.create(text='{}\n{}'.format(str(e), ticket_example), summonees=summonees)
                ticket_error_occured = True
            except Exception as e:
                logging.error('ERROR: Exception at {issue}:\n{e}\n{tb}'.format(
                    issue=issue.key,
                    e=str(e),
                    tb=traceback.format_exc()
                ))
                ticket_error_occured = True

            issue = startrek.issues[ticket]
            if not ticket_error_occured and remove_tag_after_processing:
                startrek.issues.update(
                    issue,
                    tags={'remove': [tag_to_remove]}
                )

            if not ticket_error_occured and add_tag_after_processing:
                startrek.issues.update(
                    issue,
                    tags={'add': [tag_to_add]}
                )

    def process_issue(self, issue, comments):
        raise NotImplementedError("Method process_issue must be implemented")
