# -*- coding: utf-8 -*-

from abc import abstractmethod

import logging
from textwrap import dedent
from datetime import datetime, timedelta

from queries import (
    audit_budget_versioned_processor_title1,
    audit_budget_versioned_processor_query1,
    audit_cpc_versioned_processor_title1,
    audit_cpc_versioned_processor_query1,
    audit_cpm_versioned_processor_title1,
    audit_cpm_versioned_processor_query1,
    audit_any_budget_processor_title1,
    audit_any_budget_processor_title2,
    audit_any_budget_processor_query1,
    audit_any_budget_processor_query2,
    audit_cpc_processor_title1,
    audit_cpc_processor_title2,
    audit_cpc_processor_query1,
    audit_cpc_processor_query2,
    audit_cpm_processor_title1,
    audit_cpm_processor_query1,
    audit_bids_processor_title1,
    audit_bids_processor_query1,
    audit_fix_cpm_processor_title1,
    audit_fix_cpm_processor_query1,
)


YAV_SECRET = "sec-01fx7jcsjevejnypw63tk26nj3"


class AuditProcessor():

    def __init__(self, yql_token):
        self.yql_token = yql_token

    @abstractmethod
    def process(self, args):
        pass

    def _is_yql_result_empty(self, query):
        for table in query.get_results():
            if len(table.rows) > 0:
                return False
        return True

    def _add_yql(self, title, body, args):
        query_body = body.format(**args)
        return '**{title}**\n{body}'.format(
            title=title,
            body=self._return_yql_link(query_body, query_title=title),
        )

    def _return_yql_link(self, query_body, query_title):
        query = self._run_yql(query_body, query_title)
        if not query.is_success:
            return 'Failed to run\n{}'.format(query.share_url)
        return query.share_url

    def _run_yql(self, query_body, query_title):
        from yql.api.v1.client import YqlClient

        yql_client = YqlClient(token=self.yql_token)
        yql_client.config.pass_token_to_yt = True
        query = yql_client.query(
            dedent(query_body).decode('utf-8'),
            title=query_title,
            syntax_version=1,
        )
        query.run()
        logging.info('Executing YQL query: {}'.format(query_body))
        query.wait_progress()
        return query


class AuditAnyBudgetProcessor(AuditProcessor):
    def process(self, args):
        message = []
        message.append(self._add_yql(
            audit_any_budget_processor_title1,
            audit_any_budget_processor_query1,
            args
        ))
        message.append(self._add_yql(
            audit_any_budget_processor_title2,
            audit_any_budget_processor_query2,
            args
        ))

        return message


class AuditBudgetVersionedProcessor(AuditProcessor):
    def process(self, args):
        message = []
        message.append(self._add_yql(
            audit_budget_versioned_processor_title1,
            audit_budget_versioned_processor_query1,
            args
        ))

        return message


class AuditCPCVersionedProcessor(AuditProcessor):
    def process(self, args):
        message = []
        message.append(self._add_yql(
            audit_cpc_versioned_processor_title1,
            audit_cpc_versioned_processor_query1,
            args
        ))

        return message


class AuditCPMVersionedProcessor(AuditProcessor):
    def process(self, args):
        message = []
        message.append(self._add_yql(
            audit_cpm_versioned_processor_title1,
            audit_cpm_versioned_processor_query1,
            args
        ))

        return message


class AuditCpcProcessor(AuditProcessor):
    def process(self, args):
        message = []
        message.append(self._add_yql(
            audit_cpc_processor_title1,
            audit_cpc_processor_query1,
            args
        ))
        message.append(self._add_yql(
            audit_cpc_processor_title2,
            audit_cpc_processor_query2,
            args
        ))

        return message


class AuditCpmProcessor(AuditProcessor):
    def process(self, args):
        message = []
        message.append(self._add_yql(
            audit_cpm_processor_title1,
            audit_cpm_processor_query1,
            args
        ))

        return message


class AuditFixCpmProcessor(AuditProcessor):
    def process(self, args):
        message = []
        message.append(self._add_yql(
            audit_fix_cpm_processor_title1,
            audit_fix_cpm_processor_query1,
            args
        ))

        return message


class AuditBidsProcessor(AuditProcessor):
    def process(self, args, issue):
        message = []

        yql_query = self._run_yql(
            query_body=audit_bids_processor_query1.format(**args),
            query_title=audit_bids_processor_title1
        )

        message.append(yql_query.share_url)

        if not yql_query.is_success:
            message.append('Failed to run yql')
        elif self._is_yql_result_empty(yql_query):
            issue.transitions['start_progress'].execute()
            issue.transitions['closed'].execute(comment='Новых проблем не обнаружено')

        return message


class AuditBidsClosedProcessor(AuditProcessor):

    def process(self, args, issue):
        messages = []

        issue_updated_at = datetime.strptime(issue.updatedAt[0:16], '%Y-%m-%dT%H:%M')
        issue_status_start_time = datetime.strptime(issue.statusStartTime[0:16], '%Y-%m-%dT%H:%M') + timedelta(minutes=1)
        if issue_updated_at > issue_status_start_time:
            self.__add_tag_to_issue(issue, args.get('startrek_api_url', ''), args.get('startrek_token', ''))

            issue.comments.create(text='Новые данные найдены в рамках атоматической проверки закрытых тикетов',
                                  summonees=[self.__get_on_duty_user(issue, args.get('abc_token', ''))])

            #  [{"id":"open1", "display":"Reopen","to":{"id":"1","key":"open","display":"Open"}}]
            issue.transitions['open1'].execute()

            audit_bids_processor = AuditBidsProcessor(self.yql_token)
            messages = audit_bids_processor.process(args, issue)

        return messages

    def __get_on_duty_user(self, issue, abc_token):
        from sandbox.projects.abc.client import AbcClient
        on_duty_user = None
        try:
            on_duty_user = AbcClient(abc_token).get_current_duty_login(3278, 'yabs_supbs_engine_duty_1')  # SUPBS Engine-1

        except Exception:
            logging.error('Cannot get On Duty Users', exc_info=True)

        if not on_duty_user:
            on_duty_user = issue.assignee

        return on_duty_user

    def __add_tag_to_issue(self, issue, startrek_api_url, startrek_token):
        from startrek_client import Startrek

        startrek = Startrek(
            useragent='autosupbs',
            base_url=startrek_api_url,
            token=startrek_token
        )

        startrek.issues.update(
            issue,
            tags={'add': ['audit_bids_reopened']}
        )


class AuditFixCPMProcessor(AuditProcessor):
    def process(self, args):
        message = []
        message.append(self._add_yql(
            audit_fix_cpm_processor_title1,
            audit_fix_cpm_processor_query1,
            args
        ))

        return message
