# -*- coding: utf-8 -*-

import logging
import re
from textwrap import dedent

from sandbox import sdk2
from sandbox.projects.yabs.auto_supbs_2.common import (
    BaseAutoSupBsIssueProcessor,
    TicketError
)
from sandbox.projects.yabs.auto_supbs_2.lib.distribution_processor_lib import DistributionProcessor


class AutoSupBsDistributionIssueProcessor(BaseAutoSupBsIssueProcessor):

    class Parameters(BaseAutoSupBsIssueProcessor.Parameters):
        yql_token = sdk2.parameters.String(
            'yql token sandbox vault name',
            default='AUTOSUPBS_YQL_TOKEN',
            required=True
        )
        debug_mode = sdk2.parameters.Bool(
            'enable debug mode',
            default=False,
            required=True
        )

    def process_issue(self, issue, comments):
        logging.info('Processing {}'.format(issue.key))
        description = issue.description
        if description is None:
            raise TicketError('Empty Description')

        start_date = re.findall(r'start:\s*(\d{4}-\d{2}-\d{2})', description, re.UNICODE)
        end_date = re.findall(r'end:\s*(\d{4}-\d{2}-\d{2})', description, re.UNICODE)
        yandexuid = re.findall(r'yandexuid:\s*(\d+)\s*', description, re.UNICODE)
        is_mobile = re.findall(r'mobile:\s*(\d+)\s*', description, re.UNICODE)

        pages = re.search(r'PageID:\s*(?P<pages>(\d+,?\s*)+)', description, re.UNICODE)
        pages = '' if not pages else pages.group('pages')

        orders = re.search(r'OrderID:\s*(?P<orders>(\d+,?\s*)+)', description, re.UNICODE)
        orders = '' if not orders else orders.group('orders')

        url = re.findall(r'URL:\s*(\S*)\s*', description, re.UNICODE)
        if not start_date:
            raise TicketError('Nothing matches "start: YYYY-MM-DD" in the Description')
        if not end_date:
            end_date = start_date
        if not yandexuid:
            raise TicketError('Nothing matches "yandexuid: XXXXXXXX" in the Description')
        if not is_mobile:
            raise TicketError('Nothing matches "mobile: X" where X = [0, 1] in the Description')

        if self.Parameters.debug_mode:
            debug_comment = '''
                **Following parameters were found in the Description:**\n
                 **start_date** = {start_date}
                 **end_date** = {end_date}
                 **yandexuid** = {yandexuid}
                 **is_mobile** = {is_mobile}
                 **pages** = {pages}
                 **orders** = {orders}
                 **url** = {url}
            '''.format(
                start_date=start_date[0],
                end_date=end_date[0],
                yandexuid=yandexuid[0],
                is_mobile=is_mobile[0],
                pages=str(pages),
                orders=str(orders),
                url=url[0] if url else 'None',
            )
            comments.append([dedent(debug_comment)])

        distribution_processor = DistributionProcessor(
            sdk2.Vault.data(self.owner, self.Parameters.yql_token),
        )
        comments.append(distribution_processor.process(
            start_date[0],
            end_date[0],
            yandexuid[0],
            is_mobile[0],
            pages if pages else None,
            orders if orders else None,
            url[0] if url else None,
        ))
