# -*- coding: utf-8 -*-

import logging
import re
import json
from collections import defaultdict
from datetime import datetime, timedelta
from sandbox import common
import datetime as dt
from sandbox.projects.yabs.auto_supbs_2.lib.matching_funnel_issue_processor.yql_query import get_chyt_analyzer_query, get_auction_analyzer_query
from sandbox.projects.yabs.auto_supbs_2.lib.matching_funnel_issue_processor.yql_query import get_auction_top_competitors_analyzer_query, check_yql_process
from sandbox.projects.yabs.auto_supbs_2.lib.matching_funnel_issue_processor.yql_query import get_matching_funnel_yql_query, make_yql_running_message, get_yql_title

from sandbox.projects.yabs.auto_supbs_2.lib.wiki_utils import make_spoiler
import requests

from sandbox import sdk2
from sandbox.projects.yabs.auto_supbs_2.common import (
    BaseAutoSupBsIssueProcessor,
    TicketError,
)
SLEEP_TIME = dt.timedelta(minutes=10)


class MatchingFunnelProcessYQLResult(BaseAutoSupBsIssueProcessor):
    NUM_COLS_BEFORE_COUNTS = 3
    NUM_COLS_AFTER_COUNTS = 1

    class Parameters(BaseAutoSupBsIssueProcessor.Parameters):
        kill_timeout = timedelta(minutes=20).total_seconds()

        yt_token = sdk2.parameters.String(
            'yt token sandbox vault name',
            default='ROBOT_AUTOSUPBS_YT_TOKEN',
            required=True
        )
        yql_token = sdk2.parameters.String(
            'yql token sandbox vault name',
            default='ROBOT_AUTOSUPBS_YQL_TOKEN',
            required=True
        )
        charts_token = sdk2.parameters.String(
            'yql token sandbox vault name',
            default='ROBOT_AUTOSUPBS_CHARTS_TOKEN',
            required=True
        )
        yql_funnel_root_path = sdk2.parameters.String(
            'Path to YT funnel root folder.\nYQL results will be placed at {{funnel_root}}/{{ticket_id}}/{{order_id}}',
            default='//home/bs-outdoor/lelby/matching_funnel_production',
            required=True
        )
        var_coef = sdk2.parameters.Float(
            'variance coef',
            default=1.0,
            required=True
        )
        top_competitors = sdk2.parameters.Integer(
            'the best N participants by the number of auctions where they were better than us',
            default=10,
        )
        graph_params_and_percentiles = sdk2.parameters.JSON(
            'Parameters which will be shown on graphs and their percentiles',
            default={
                'AuctionPosition': [10, 50, 90],
            },
            required=True
        )
        time_to_wait = sdk2.parameters.Integer('Time to wait between checks (seconds)',
                                               default=SLEEP_TIME.total_seconds())

    def yql_query_ch(self, q):
        from yql.api.v1.client import YqlClient
        from yql.client import operation
        yql_ch = YqlClient(token=sdk2.Vault.data(self.owner, 'ROBOT_AUTOSUPBS_YQL_TOKEN_TEMPORARY'))
        request = yql_ch.query(q, syntax_version=1, clickhouse_syntax=True)

        request.type = operation.YqlOperationType.CLICKHOUSE
        request.run()
        results = request.get_results()

        if not results.is_success:
            msg = '\n'.join([str(err) for err in results.errors])
            raise Exception('Error when executing query {}: {}'.format(q, msg))

        return results

    def get_chart_data(self, yql_query_id, percentiles=None):
        from library.python import resource
        param_name = "AuctionPosition"
        logging.info("param_name: {}".format(param_name))
        names = [param_name + str(percent) for percent in percentiles]
        logging.info(names)
        series = '\n'.join(
            'series.{name}.data.push([row.EventDay, row.{name}])'
                .format(name=name) for name in names
        )
        data = ',\n'.join(
            """
            {name} : {{
                name: '{param_name} at {percent} percentile',
                data: []
            }}
            """.format(name=name, param_name=param_name, percent=percent)
            for name, percent in zip(names, percentiles)
        )

        js = resource.find('/datalens_js_template').format(series=series, data=data)
        url = resource.find('/datalens_url_template').format(yql_query_id=yql_query_id)
        graph = resource.find('/datalens_graph_template').format(param_name=param_name)
        return {
            'type': 'graph_node',
            'key': 'Users/robot-yabs-autosupbs/AutoSUPBSCharts/{name}'.format(
                name=datetime.now().strftime("%H:%M:%S %d.%m.%Y {}".format(param_name))
            ),
            "data": {
                "js": js,
                "ui": "",
                "url": url,
                "graph": graph,
                "params": "",
                "shared": "",
                "statface_graph": "",
            }
        }

    def build_chart(self, yql_query_id):
        chart_data = self.get_chart_data(yql_query_id, self.Parameters.graph_params_and_percentiles["AuctionPosition"])
        headers = {
            'authorization': 'OAuth {token}'.format(token=sdk2.Vault.data(self.owner, self.Parameters.charts_token)),
            'content-type': 'application/json;charset=UTF-8',
            'user-agent': '{login} ({login}@yandex-team.ru)'.format(login='robot-yabs-autosupbs')
        }
        logging.info(chart_data)
        response = requests.post(
            url='https://api.charts.yandex.net/v1/charts',
            headers=headers,
            data=json.dumps(chart_data)
        )
        if response.status_code != 200:
            logging.error('{}'.format(chart_data))
            raise Exception(
                "Error making a chart, code {} {} {}".format(response.status_code, response.reason, response.content))
        return response.json()

    def build_graph(self, ticket_id, order_id, yt_client):
        select_percentiles = ',\n '.join(
            'quantile({})(cnt) as AuctionPosition{}'.format((int(item) / 100.0), int(item)) for item in
            self.Parameters.graph_params_and_percentiles["AuctionPosition"])
        logging.info("Select percentiles part: {}".format(select_percentiles))
        query = get_auction_analyzer_query(ticket_id, order_id, self.Parameters.yql_funnel_root_path
                                           , select_percentiles)
        res = self.yql_query_ch(query)
        logging.info("clickhouse url: {}".format(res.operation_id))
        return '{{{{iframe frameborder="0" width="100%" height="400px" src="https://charts.yandex-team.ru/preview/editor/{entryId}?_embedded=1"}}}}'.format(
            entryId=self.build_chart(res.operation_id)['entryId'])

    def process_yql_result_matching_funnel(self, comments, ticket_id, order_id, yt_client, issue):
        import yt.clickhouse as chyt

        logging.info('Processing {}, order_id={}'.format(ticket_id, order_id))
        query_text = get_chyt_analyzer_query(ticket_id, order_id, self.Parameters.yql_funnel_root_path)
        logging.info('Chyt query:\n {}'.format(query_text))
        q = chyt.execute(query_text, alias='*lelby_ckique', client=yt_client)
        res, side_effect_filtrations, dates, side_dates = self._premake_wiki_data(q)
        main_wiki_data = self._make_wiki_data(res, dates)
        side_effects_wiki_data = self._make_wiki_data(side_effect_filtrations, side_dates)

        comments.append([make_spoiler('Matching Funnel for OrderID {} (click to open):'.format(order_id),
                        [main_wiki_data, make_spoiler('MatchingSideEffect-filtration:', side_effects_wiki_data)])
        ])

    def process_yql_result_competitor_analysis(self, comments, ticket_id, order_id, yt_client, issue):
        import yt.clickhouse as chyt

        logging.info('Processing {}, order_id={}'.format(ticket_id, order_id))
        auction_top_text = get_auction_top_competitors_analyzer_query(ticket_id, order_id,
                                                                      self.Parameters.yql_funnel_root_path
                                                                      , self.Parameters.top_competitors)
        graphic = self.build_graph(ticket_id, order_id, yt_client)
        auction_top = chyt.execute(auction_top_text, alias='*lelby_ckique', client=yt_client)
        logging.info('auction_top chyt query:\n {}'.format(auction_top))

        top_orderid_wiki = self._make_count_list_data(self.fill_empty_cell(auction_top, issue))
        comments.append([make_spoiler('Competitor Analysis for OrderID {} (click to open):'.format(order_id),
                        [make_spoiler('Chart :', graphic),
                        make_spoiler('Top {} Competitors :'.format(self.Parameters.top_competitors), top_orderid_wiki)])
        ])

    def round_up_middle_pos_val(self, stat):
        if stat == 'Null':
            return 'Null'
        sp_stat = stat.split(',')
        sp_stat[2] = str(int(float(sp_stat[2]) * 1e6))
        return '({})'.format(';'.join(sp_stat))

    def fill_empty_cell(self, data, issue):
        import datetime
        start_date = datetime.datetime.strptime(issue.start, "%Y-%m-%d")
        end_date = datetime.datetime.strptime(issue.end, "%Y-%m-%d")
        logging.info("start day:{}".format(str(start_date.date())))
        logging.info("end day:{}".format(str(end_date.date())))
        top_competitors = int(self.Parameters.top_competitors)
        dic = defaultdict(list)
        for item in data:
            logging.info(item)
            dic[item['_date']] = item['order_pval']
        res = []
        logging.info('start_date: {}'.format(start_date.date()))
        logging.info('end_date: {}'.format(end_date.date()))
        while start_date <= end_date:
            order_val = []
            if str(start_date.date()) in dic:
                order_val = dic[str(start_date.date())]
            while len(order_val) != top_competitors:
                order_val.append('Null')
            start_date = start_date + datetime.timedelta(days=1)
            res.append((str(start_date.date()), order_val))
        logging.info("filled table of participants:{}".format(res))
        return res

    def _make_count_list_data(self, data):
        top_competitors = int(self.Parameters.top_competitors)
        title_row = '|| {} ||'.format(' | '.join('**{}**'.format(item[0]) for item in data))
        wiki_rows = [
            '|| {} ||'.format(' | '.join('{}'.format("(order_id; count; avg_pval*1e-6)") for item in data))]
        for i in range(top_competitors):
            wiki_rows.append(
                '|| {} ||'.format(' | '.join(self.round_up_middle_pos_val(item[1][i]) for item in data)))
        logging.info("wiki rows for top_competitors: {}".format(wiki_rows))
        return '#|\n' + title_row + '\n'.join(wiki_rows) + '\n|#'

    def _premake_wiki_data(self, q):
        from yabs.server.proto.filter_reason import filter_reason_pb2
        dates = set()
        reasons = set()
        cells = []
        side_dates = set()
        side_reasons = set()
        side_cells = []
        for item in q:
            if not item['reasons']:
                dates.add(item['_date'])
                reasons.add(-1)
                cells.append({
                    'reason_id': -1,
                    'name': None,
                    'reason_info': None,
                    'count': item['summ'],
                    'date': item['_date']
                })
            else:
                for reason_id in map(int, item['reasons'].split(',')):
                    reason_proto = filter_reason_pb2._FILTERREASONENUM.values_by_number[reason_id]
                    reason_info = reason_proto.GetOptions().Extensions[filter_reason_pb2.filter_reason_info]
                    cell = {
                        'reason_id': reason_id,
                        'name': reason_proto.name,
                        'reason_info': reason_info,
                        'count': item['summ'],
                        'date': item['_date']
                    }
                    if filter_reason_pb2.FilterReasonTagEnum.MatchingSideEffect not in reason_info.tags:
                        dates.add(item['_date'])
                        reasons.add(reason_id)
                        cells.append(cell)
                    else:
                        side_dates.add(item['_date'])
                        side_reasons.add(reason_id)
                        side_cells.append(cell)

        dates = sorted(list(dates))
        side_dates = sorted(list(side_dates))

        res = [[0] * (len(dates) + self.NUM_COLS_BEFORE_COUNTS) for i in range(len(reasons))]
        side_effect_filtrations = [[0] * (len(side_dates) + self.NUM_COLS_BEFORE_COUNTS) for i in
                                   range(len(side_reasons))]

        def gen_to_idx(some):
            some_to_idx = dict()
            for i, x in enumerate(some):
                some_to_idx[x] = i
            return some_to_idx

        date_to_idx = gen_to_idx(dates)
        reason_to_idx = gen_to_idx(reasons)
        side_date_to_idx = gen_to_idx(side_dates)
        side_reason_to_idx = gen_to_idx(side_reasons)

        for cell in cells:
            i = reason_to_idx[cell['reason_id']]
            res[i][0] = cell['reason_id']
            res[i][1] = cell['name']
            res[i][2] = cell['reason_info']
            res[i][date_to_idx[cell['date']] + self.NUM_COLS_BEFORE_COUNTS] = cell['count']

        for cell in side_cells:
            i = side_reason_to_idx[cell['reason_id']]
            side_effect_filtrations[i][0] = cell['reason_id']
            side_effect_filtrations[i][1] = cell['name']
            side_effect_filtrations[i][2] = cell['reason_info']
            side_effect_filtrations[i][side_date_to_idx[cell['date']] + self.NUM_COLS_BEFORE_COUNTS] = cell['count']
        return res, side_effect_filtrations, dates, side_dates

    def _make_wiki_data(self, data, dates):
        def make_reason_additional_info(reason_info):
            def make_startrek_ticket_link(ticket_id):
                return 'https://st.yandex-team.ru/' + ticket_id

            res = []
            if reason_info.comment:
                res.append(reason_info.comment)
            if reason_info.tickets:
                res.append('\n'.join(map(make_startrek_ticket_link, reason_info.tickets)))
            if reason_info.responsible:
                res.append('Responsible: ' + reason_info.responsible)
            return '\n'.join(res)

        titles = ['ReasonID', 'Name', 'Description'] + dates + ['Additional Info']
        title_row = '|| {} ||'.format(' | '.join('**{}**'.format(x) for x in titles))

        def make_wiki_row(data_row):
            logging.info('Current row: {}'.format(str(data_row)))
            counts = data_row[self.NUM_COLS_BEFORE_COUNTS:]
            logging.info("Counts: {}".format(str(counts)))
            mean = sum(counts) / len(counts)
            var = (sum(map(lambda x: (x - mean) ** 2, counts)) / len(counts)) ** 0.5
            if data_row[1] is None:
                data_row = ['' if el is None else el for el in data_row]
                data_row.append('')
            else:
                data_row.append(make_reason_additional_info(data_row[2]))

            before_counts = data_row[:self.NUM_COLS_BEFORE_COUNTS]
            after_counts = data_row[-self.NUM_COLS_AFTER_COUNTS:]
            ans_before_counts = ' | '.join(map(str, before_counts))

            def gen_count_cell(el):
                col = 'blue'
                var_coef = self.Parameters.var_coef
                if el < mean - var_coef * var:
                    col = 'red'
                elif el > mean + var_coef * var:
                    col = 'green'
                return '!!({}){}!!'.format(col, el)

            ans_counts = ' | '.join(map(gen_count_cell, counts))
            ans_after_counts = ' | '.join(map(str, after_counts))
            return '|| {} ||'.format(ans_before_counts + ' | ' + ans_counts + ' | ' + ans_after_counts)

        data.sort(key=lambda x: x[self.NUM_COLS_BEFORE_COUNTS], reverse=True)
        rows = map(make_wiki_row, data)
        wiki_rows = '\n'.join(rows)
        return '#|\n' + title_row + wiki_rows + '\n|#'

    def run_yql_process(self, issue, comments, orders, yql_token):
        from ads.libs.yql import run_yql_query
        page_id = re.search(
            u'PageID:\**\s*\**(\d+)\**\s*',  # noqa
            issue.description,
            re.UNICODE
        )
        imp_id = re.search(
            u'ImpID:\**\s*\**(\d+)\**\s*',  # noqa
            issue.description,
            re.UNICODE
        )

        if page_id:
            page_id = page_id.group(1)
        if imp_id:
            imp_id = imp_id.group(1)

        date_from = issue.start
        date_to = issue.end
        if not orders:
            raise TicketError('OrderID was not found in the Description')

        logging.info('The following orders were found {}'.format(orders))
        self.Context.auction_yql_query = {}
        self.Context.yql_query = {}

        for order_id in orders:
            title = get_yql_title(order_id, page_id, imp_id, date_from, date_to)

            query_text, auction_query_text = get_matching_funnel_yql_query(issue.key, order_id, page_id, imp_id,
                                                                            date_from, date_to,
                                                                            self.Parameters.yql_funnel_root_path)
            logging.info("query_text : {}".format(query_text))
            logging.info("auction_query_text : {}".format(auction_query_text))

            auction_yql_query = run_yql_query(query=auction_query_text, db='hahn',
                                                token=yql_token, is_async=True, title=title)

            yql_query = run_yql_query(query=query_text, db='hahn',
                                        token=yql_token, is_async=True, title=title)

            self.Context.yql_query[order_id] = {'operation_id': yql_query.operation_id, 'status': 'processing'}
            self.Context.auction_yql_query[order_id] = {'operation_id': auction_yql_query.operation_id, 'status': 'processing'}

            logging.info('auction_yql_query: {}'.format(self.Context.auction_yql_query))
            logging.info('yql_query: {}'.format(self.Context.yql_query))

            comments.append(make_yql_running_message(order_id, page_id, imp_id, date_from, date_to, auction_yql_query.share_url, yql_query.share_url))

        self.Context.task_type = 'YQL_RUNNING'

    def process_issue(self, issue, comments):
        from yabs.yt_wrapper import YtClient
        from startrek_client import Startrek

        yql_token = sdk2.Vault.data(self.owner, self.Parameters.yql_token)
        self.Context.ticket = issue.key
        startrek = Startrek(
            useragent='autosupbs',
            base_url=self.Parameters.startrek_api_url,
            token=sdk2.Vault.data(self.owner, self.Parameters.startrek_token),
        )
        if not self.Context.launched:
            startrek.issues.update(
                issue,
                tags={'remove': ['supbs_need_matching_funnel']}
            )
            logging.info('supbs_need_matching_funnel tag from ticket:{} was removed'.format(issue.key))

        self.Context.launched = True

        logging.info('Processing {}'.format(issue.key))
        if issue.description is None:
            raise TicketError('Empty Description')
        orders = set(re.findall(
            u'БК:\**\s*\**(\d+)\**\s*',  # noqa
            issue.description,
            re.UNICODE
        ))

        if issue.start is None or issue.end is None or not orders:
            logging.info("start_date:{}, end_date: {} orders:{} they shouldn`t be None ".format(issue.start, issue.end, orders))
            comments.append(["If you want to run Matching Funnel, please specify Start, End dates and Order Number in ticket"])
            return

        if not orders:
            raise TicketError('OrderID was not found in the Description')

        if self.Context.task_type != 'YQL_RUNNING':
            self.run_yql_process(issue, comments, orders, yql_token)
            for comment in comments:
                issue.comments.create(text='\n'.join(comment))
            raise sdk2.WaitTime(self.Parameters.time_to_wait)
        else:
            logging.info('auction_yql_query status: {}'.format(self.Context.auction_yql_query))
            logging.info('yql_query status: {}'.format(self.Context.yql_query))
            logging.info('The following orders were found {}'.format(orders))
            yt_client = YtClient(proxy='hahn', token=sdk2.Vault.data(self.owner, self.Parameters.yt_token))
            finished_all_process = True
            try:
                for order_id in orders:
                    if self.Context.yql_query[order_id]['status'] == 'processing':
                        if check_yql_process(self.Context.yql_query[order_id]['operation_id'], yql_token):
                            self.Context.yql_query[order_id]['status'] = 'done'
                            self.process_yql_result_matching_funnel(comments, issue.key, order_id, yt_client, issue)
                        else:
                            logging.info('YQL for {} {} is not finished yet'.format(issue.key, order_id))
                            finished_all_process = False

                    if self.Context.auction_yql_query[order_id]['status'] == 'processing':
                        if check_yql_process(self.Context.auction_yql_query[order_id]['operation_id'], yql_token):
                            self.Context.auction_yql_query[order_id]['status'] = 'done'
                            self.process_yql_result_competitor_analysis(comments, issue.key, order_id, yt_client, issue)
                        else:
                            logging.info('YQL auction for {} {} is not finished yet'.format(issue.key, order_id))
                            finished_all_process = False
            except Exception as e:
                raise common.errors.TaskFailure(e)

            if finished_all_process:
                logging.info("all the process is over")
            else:
                for comment in comments:
                    issue.comments.create(text='\n'.join(comment))
                raise sdk2.WaitTime(self.Parameters.time_to_wait)
