# -*- coding: utf-8 -*-

from textwrap import dedent
from datetime import datetime, timedelta

import json
import requests
import logging

from sandbox.projects.yabs.auto_supbs_2.lib.wiki_utils import make_spoiler


class OrderProcessor():

    def __init__(self, yt_token, caesar_adgroups_path, yql_token, charts_token, start_date, end_date, params):
        self.yt_token = yt_token
        self.yql_token = yql_token
        self.caesar_adgroups_path = caesar_adgroups_path
        self.charts_token = charts_token
        self.start_date = start_date
        self.end_date = end_date
        self.params = params

    def yql_query_ch(self, q):
        from yql.api.v1.client import YqlClient
        from yql.client import operation

        yql_ch = YqlClient(token=self.yql_token)
        request = yql_ch.query(q, syntax_version=1, clickhouse_syntax=True)
        request.type = operation.YqlOperationType.CLICKHOUSE
        request.run()
        results = request.get_results()
        if not results.is_success:
            msg = '\n'.join([str(err) for err in results.errors])
            raise Exception('Error when executing query {}: {}'.format(q, msg))
        return results

    def get_chart_data(self, yql_query_id, param_names, percentiles=None):
        all_data = []
        all_series = []
        for param_name in param_names:
            if percentiles:
                names = [param_name + str(percent) for percent in percentiles]
            else:
                names = [param_name]
            series = '\n'.join(
                'series.{name}.data.push([row.EventHour, row.{name}])'
                .format(name=name) for name in names
            )
            if percentiles:
                data = ',\n'.join(
                    """
                    {name} : {{
                        name: '{param_name} at {percent} percentile',
                        data: []
                    }}
                    """.format(name=name, param_name=param_name, percent=percent)
                    for name, percent in zip(names, percentiles)
                )
            else:
                data = """
                    {name} : {{
                        name: '{name}',
                        data: []
                    }}
                    """.format(name=param_name)
            all_data.append(data)
            all_series.append(series)

        return {
            'type': 'graph_node',
            'key': 'Users/robot-yabs-autosupbs/AutoSUPBSCharts/{name}'.format(
                name=datetime.now().strftime("%H:%M:%S %d.%m.%Y {}".format(param_name))
            ),
            "data": {
                "js": """
                    // подключаем модуль для работы с датами
                    const moment = require('vendor/moment/v2.21');

                    // подключаем модуль источника данных
                    const YQL= require('libs/yql/v1');

                    // запрашиваем загруженные данные
                    const loadedData = ChartEditor.getLoadedData();

                    // преобразовываем данные
                    const preparedData = YQL.processOperationData(
                        loadedData.yqlOperationSource
                    );

                    const series = preparedData.reduce((series, row) => {
                    """ +
                    ',\n'.join(all_series) +
                    """
                        return series;
                    }, {
                    """ +
                    ',\n'.join(all_data) +
                    """
                    })

                    // экспортируем данные для отрисовки
                    module.exports = Object.values(series);
                """,
                "ui": "",
                "url": """
                    // подключаем модуль источника данных
                    const YQL = require('libs/yql/v1');

                    // формируем источник данных
                    const yqlOperationSource = YQL.buildOperationSource({
                        operationId: '""" + str(yql_query_id) + """',
                    });

                    // экспортируем источник данных
                    module.exports = {
                        yqlOperationSource: yqlOperationSource
                    };
                """,
                "graph": """
                    module.exports = {
                        title: {
                            text: \" """ + '/'.join(param_names) + """\"
                        },
                        xAxis: {
                            type: 'datetime'
                        }
                    };
                """,
                "params": "",
                "shared": "",
                "statface_graph": "",
            }
        }

    def build_chart(self, yql_query_id, param_names, percentiles=None):
        chart_data = self.get_chart_data(yql_query_id, param_names, percentiles)
        headers = {
            'authorization': 'OAuth {token}'.format(token=self.charts_token),
            'content-type': 'application/json;charset=UTF-8',
            'user-agent': '{login} ({login}@yandex-team.ru)'.format(login='robot-yabs-autosupbs')
        }
        response = requests.post(
            url='https://api.charts.yandex.net/v1/charts',
            headers=headers,
            data=json.dumps(chart_data)
        )
        if response.status_code != 200:
            logging.error('{}'.format(chart_data))
            raise Exception("Error making a chart, code {} {} {}".format(response.status_code, response.reason, response.content))
        return response.json()

    def get_yql_query_text(self, params, order_id):
        quantiles_query = ',\n'.join(
            ',\n'.join(
                'quantile({part})({param}) as {param}{percent}'
                .format(part=percent / 100.0, param=param, percent=percent)
                for percent in percentiles
            )
            for param, percentiles in params.iteritems()
        )

        return """
            use yabsclickhouse;

            SELECT
                intDiv(Cast(EventTime, 'UInt64'), 60 * 60) * (60 * 60) * 1000 As EventHour,
                {quantiles_query},
                countIf(CounterType == 1) as Shows,
                countIf(CounterType == 2) as Clicks
            FROM
                ChEventArchive
            WHERE
                OrderID = {order_id} and
                EventDate >= '{start_date}' and
                EventDate <= '{end_date}'
            GROUP BY
                EventHour
            ORDER BY
                EventHour asc;
        """.format(
            quantiles_query=quantiles_query,
            order_id=order_id,
            start_date=self.start_date,
            end_date=self.end_date
        )

    def build_graphs(self, order_id):
        if self.start_date is None or self.end_date is None:
            return ["**No dates set to build graphs from logs!!!**"]
        message = []

        query_text = self.get_yql_query_text(self.params, order_id)

        logging.info("Getting clickhouse query from YQL")
        logging.info("Clickhouse query: {}".format(query_text))

        res = self.yql_query_ch(query_text)

        logging.info("YQL operation id: {}".format(res.operation_id))

        add_chart = lambda chart_info: message.append(
            '{{{{iframe frameborder="0" width="100%" height="400px" src="https://charts.yandex-team.ru/preview/editor/{entryId}?_embedded=1"}}}}'
            .format(entryId=chart_info['entryId'])
        )

        for param, percentiles in self.params.iteritems():
            add_chart(self.build_chart(res.operation_id, [param], percentiles))
        add_chart(self.build_chart(res.operation_id, ['Shows', 'Clicks']))
        message = [make_spoiler("Graphs from ChEventArchive", message)]
        message.append("https://yql.yandex-team.ru/Operations/{}".format(res.operation_id))

        return message

    def get_entity_from_caesar(self, entity, entity_id, lookup_profile_dump_text):
        import yt.yson as yson

        caesar_url = 'http://lookup-profile-caesar1.n.yandex-team.ru/lookup_profile/{}s/?{}ID={}'.format(entity, entity, entity_id)
        logging.info('Loading info from Caesar for {} with ID {} GET {}'.format(entity, entity_id, caesar_url))

        response = yson.yson_to_json(yson.loads(requests.get(caesar_url).content))
        lookup_profile_dump_text.append('Full Dump for {} {} {}'.format(entity, entity_id, caesar_url))
        lookup_profile_dump_text.append(json.dumps(response, indent=4, sort_keys=True, ensure_ascii=False))

        return response[0]

    def get_banner_ids_by_order_id_at_link(self, link, order_id):
        try:
            params = {'order-id': order_id}
            response = requests.get(link, params=params)
            if response.status_code != 200:
                logging.info(
                    "Error while getting response from: {} with params: {}, and status_code {}".format(link, params,
                                                                                                       response.status_code))
                return set(), set()

            response = response.json()
            if len(response) == 0:
                logging.info("Empty response")
                return set(), set()

            response = response[0]
            if "Banners" not in response:
                logging.info("No banners in response")
                return set(), set()

            banner_ids = set()
            banner_group_export_id = set()
            banners = response["Banners"]
            for banner in banners:
                banner_ids.add(banner["BannerID"])
                banner_group_export_id.add(banner["GroupExportID"])

            return banner_ids, banner_group_export_id

        except Exception as e:
            logging.warning("Error occurs while processing response: {}".format(e))
            return set(), set()

    def get_banner_ids_by_order_id(self, order_id):
        an_banner_ids, an_banner_group_export_ids = self.get_banner_ids_by_order_id_at_link('https://an.yandex.ru/audit',
                                                                                            order_id)
        logging.info("Banners from an.yandex.ru: {}".format(an_banner_ids))
        logging.info("GroupExportID from an.yandex.ru: {}".format(an_banner_group_export_ids))

        yabs_banner_ids, yabs_banner_group_export_ids = self.get_banner_ids_by_order_id_at_link('https://yabs.yandex.ru/audit',
                                                                                                order_id)
        logging.info("Banners from yabs.yandex.ru: {}".format(yabs_banner_ids))
        logging.info("GroupExportID from yabs.yandex.ru: {}".format(yabs_banner_group_export_ids))
        return sorted(list(an_banner_ids | yabs_banner_ids)), list(an_banner_group_export_ids | yabs_banner_group_export_ids)

    def clean_info_from_dict(self, info, leave):
        clean_info = {}
        for path in leave:
            data = info
            for key in path:
                if not isinstance(data, dict) or key not in data:
                    data = None
                    break
                data = data[key]

            if data is not None:
                curr_info = clean_info
                last_key = path[-1]
                for key in path[:-1]:
                    if key not in curr_info:
                        curr_info[key] = dict()
                    curr_info = curr_info[key]
                curr_info[last_key] = data

        return clean_info

    def __add_info(self, message, text, info, lookup_link):
        message.append(make_spoiler(text, "%%{}%%\n{}".format(json.dumps(info, indent=4), lookup_link)))

    def __process_entities(self, entity_name, entity_name_capital, entity_key_field, entity_ids, trim_info, message, lookup_profile_dump_text, max_entity_count):
        logging.info("Processing {}s".format(entity_name_capital))
        logging.info("All {} ids for order: {}".format(entity_name, entity_ids))

        entity_message = []

        for entity_id in entity_ids:
            entity_info = self.get_entity_from_caesar(entity_name_capital, entity_id, lookup_profile_dump_text)

            entity_link = "https://lookup-profile-caesar1.n.yandex-team.ru/lookup_profile/{capital}s/?{capital}ID={id}".format(
                capital=entity_name_capital,
                id=entity_id
            )

            self.__add_info(
                entity_message,
                "Info for {} {}".format(entity_name_capital, entity_id),
                self.clean_info_from_dict(
                    entity_info,
                    trim_info
                ),
                entity_link
            )

        if len(entity_message) == 0:
            message.append("**No {}s found for order!**".format(entity_name))
        else:
            if len(entity_message) == 1:
                message.append(entity_message[0])
            else:
                if len(entity_message) > max_entity_count:
                    entity_message = [
                        "**Showing only the first {count} {name}s, you can find other {name}s in resource dump**".format(
                            count=max_entity_count,
                            name=entity_name
                        )
                    ] + entity_message[:max_entity_count]
                message.append(
                    make_spoiler(
                        title="Trimmed info for all {} {}s".format(len(entity_ids), entity_name),
                        content="\n\n".join(entity_message)
                    )
                )

    def process_order_from_caesar(self, order_id, task_id, max_entity_count):
        message = []
        message.append("Selects from CaeSaR")

        lookup_profile_dump_text = []

        banner_ids, banner_group_export_ids = self.get_banner_ids_by_order_id(order_id)

        self.__process_entities(
            entity_name="order",
            entity_name_capital="Order",
            entity_key_field="order",
            entity_ids=[order_id],
            trim_info=[
                ['OrderID'],
                ['Resources', 'DirectBannersLogFields', 'DirectOrderID'],
                ['Resources', 'DirectBannersLogFields', 'OrderType'],
                ['Resources', 'DirectBannersLogFields', 'GroupOrderID'],
                ['Resources', 'DirectBannersLogFields', 'ContentType'],
                ['Flags'],
                ['Resources', 'ESSMultipliers'],
                ['Resources', 'ShowConditions'],
                ['ShowConditions', 'ContextID']
            ],
            message=message,
            lookup_profile_dump_text=lookup_profile_dump_text,
            max_entity_count=max_entity_count
        )

        self.__process_entities(
            entity_name="banner",
            entity_name_capital="Banner",
            entity_key_field="banner",
            entity_ids=banner_ids,
            trim_info=[
                ['BannerID'],
                ['OrderID'],
                ['Resources', 'Href'],
                ['Resources', 'HrefText'],
                ['Resources', 'Title'],
                ['Resources', 'Body'],
                ['Resources', 'TargetDomain'],
                ['Resources', 'Site'],
                ['Resources', 'GroupBannerID'],  # old name GroupExportID
                ['Flags'],
                ['ModerationData'],
                ['Options'],
                ['PriorityID']
            ],
            message=message,
            lookup_profile_dump_text=lookup_profile_dump_text,
            max_entity_count=max_entity_count
        )

        self.__process_entities(
            entity_name="ad_group",
            entity_name_capital="AdGroup",
            entity_key_field="ad-group",
            entity_ids=banner_group_export_ids,
            trim_info=[
                ['AdGroupID'],
                ['OrderID'],
                ['ShowConditions', 'ContextID']
            ],
            message=message,
            lookup_profile_dump_text=lookup_profile_dump_text,
            max_entity_count=max_entity_count
        )

        show_conditions = []
        order_info = self.get_entity_from_caesar('Order', order_id, [])
        if 'ShowConditions' in order_info:
            show_conditions.append(order_info['ShowConditions'])

        message.append(self.__get_phrase_info_from_caesar(banner_group_export_ids, show_conditions))
        message.append(self.__get_patterns_from_caesar(show_conditions))
        message.append(self.__get_additional_queries())

        return message, lookup_profile_dump_text

    def process_order_genocide(self, order_id, task_id, max_entity_count):
        message = []
        message.append("Genocide")

        message.append(self.__get_banners_genocide_info(order_id))
        message.append(self.__get_banners_genocide_history(order_id))

        return message

    def __get_prettytable(self, columns):
        from prettytable import PrettyTable
        tb = PrettyTable()
        tb.field_names = columns
        tb.align = "l"  # The allowed strings are "l", "r" and "c" for left, right and centre alignment, respectively
        tb.border = True

        return tb

    def __get_banners_genocide_info(self, order_id):
        logging.info("Get Genocide Info for Order {}".format(order_id))

        genocide_query = """
        use hahn;
        $date_format = DateTime::Format("%Y-%m-%d");
        SELECT
            banner.BannerID,
            banner.GroupExportID,
            $date_format(AddTimezone(DateTime::FromSeconds(CAST(genocide.GenerationTime as Uint32)), 'Europe/Moscow')) as GenerationTime,
            genocide.LoadProbability as LoadProbability,
        FROM
            `home/yabs-cs/key_1/v2/bs_export/genocide/genocide_results` as genocide
        JOIN `home/yabs-cs/export/YTBanner` as banner on (banner.GroupExportID == Yson::ConvertToInt64(genocide.ObjectInfo['GroupExportID']))
        WHERE
            Yson::ConvertToInt64(ObjectInfo['OrderID']) == {order_id} and genocide.WasLoaded = false
            and banner.OrderID == {order_id}
        ORDER BY
            banner.GroupExportID,
            banner.BannerID
        """.format(order_id=order_id)

        query_result, query_url = self.__execute_yql_query(genocide_query)
        if not query_result:
            return '**No Genocide Banners found for the Order!**'

        tb = self.__get_prettytable(['BannerID', 'GroupExportID', 'GenerationTime', 'LoadProbability'])
        for item in query_result:
            tb.add_row(item)

        return '<{{{name}\n%%{res}%%}}>'.format(name='Unloaded Banners (genocide)', res="YQL Query (({})) \n {}".format(query_url, tb))

    def __get_banners_genocide_history(self, order_id):
        start_date = (datetime.now() + timedelta(days=-60)).strftime("%Y-%m-%d")
        end_date = (datetime.now()).strftime("%Y-%m-%d")

        logging.info("Get Genocide History for Order {} for last 2 months".format(order_id))

        genocide_query = """
        use hahn;
        PRAGMA yt.InferSchema = '1';

        SELECT
            banner.BannerID,
            banner.GroupExportID,
            ListSort(AGGREGATE_LIST(AsTuple(GenerationTime, NeedLoad, LoadProbability)))
        FROM
            (
                SELECT
                    genocide.GroupExportID as GroupExportID,
                    genocide.NeedLoad as NeedLoad,
                    genocide.LoadProbability as LoadProbability,
                    TableName() as GenerationTime
                FROM
                    range(`//home/yabs-cs/key_1/v2/static_rank/genocide/results_archive`, '{start_date}', '{end_date}') as genocide
                WHERE
                    OrderID = {order_id} and GroupVerdict != 0
            )
            as genocide
        JOIN `home/yabs-cs/export/YTBanner` as banner on (banner.GroupExportID == genocide.GroupExportID)
        WHERE
            banner.OrderID == {order_id}
        GROUP BY
            banner.BannerID,
            banner.GroupExportID
        ORDER BY
            banner.GroupExportID,
            banner.BannerID
        """.format(order_id=order_id, start_date=start_date, end_date=end_date)

        query_result, query_url = self.__execute_yql_query(genocide_query)
        if not query_result:
            return '**No Genocide Banners History found for the Order!**'

        tb = self.__get_prettytable(['BannerID', 'GroupExportID', 'Status History for last 2 months'])

        for item in query_result:
            intervals = []

            for generation_time, need_load, load_probability in item[2]:
                status = 'Loaded' if need_load else 'Not Loaded'
                if not intervals or intervals[-1][1] != status:
                    intervals.append((generation_time[0:16], status, "{:.3f}".format(load_probability)))

            if len(intervals) == 1 and intervals[-1][1] == 'Loaded':
                continue

            result = []
            for interval in intervals:
                result.append(" ".join(interval))

            tb.add_row([item[0], item[1], '\n'.join(result)])

        return '<{{{name}\n%%{res}%%}}>'.format(name='Genocide Banners History', res="YQL Query (({})) \n {}".format(query_url, tb))

    def __get_patterns_from_caesar(self, show_conditions):
        logging.info("Process ShowConditions (Patterns) from Order and AdGroups")

        operations = self.__get_descriptions('use hahn; select OperationID, Description from `//home/yabs/dict/Operation`')
        keywords = self.__get_descriptions('use hahn; select KeywordID, Description from `//home/yabs/dict/Keyword`')

        tb = self.__get_prettytable(['ContextID', 'Expression'])
        for show_condition in show_conditions:
            current_row = []
            current_row.append(show_condition.get('ContextID', ''))

            if 'Expression' in show_condition:
                expressions = self.__process_expressions(show_condition['Expression'], operations, keywords)
                current_row.append(json.dumps(expressions, indent=4, sort_keys=True))

            tb.add_row(current_row)

        return '<{{{name}\n%%{res}%%}}>'.format(name='Patterns from ShowConditions in Order and  AdGroups', res=tb)

    def __process_expressions(self, expressions, operations, keywords):
        if type(expressions) == list:
            for item in expressions:
                item = self.__process_expressions(item, operations, keywords)

        elif type(expressions) == dict:
            for key in expressions:
                if key == 'keyword':
                    keyword_key = str(expressions[key]).decode('utf-8')
                    expressions[key] = keywords.get(int(expressions[key]), '') + u' (' + keyword_key + u')'
                elif key == 'operation':
                    expressions[key] = "{} ({})".format(operations.get(int(expressions[key]), ''), str(expressions[key]))
                    expressions[key] = expressions[key].decode('utf-8')
                elif key == 'value':
                    expressions[key] = expressions[key].decode('utf-8')
                else:
                    expressions[key] = self.__process_expressions(expressions[key], operations, keywords)

        return expressions

    def __get_descriptions(self, query):
        descriptions = dict()

        query_result, query_url = self.__execute_yql_query(query)
        for row in query_result:
            descriptions[int(row[0])] = row[1]

        return descriptions

    def __execute_yql_query(self, query_text):
        from yql.api.v1.client import YqlClient
        client = YqlClient(token=self.yql_token)

        request = client.query(query_text)
        request.run()

        query_result = request.get_results()
        if not query_result.is_success:
            msg = '\n'.join([str(err) for err in query_result.errors])
            logging.error('Error when executing query %s: %s' % (query_text, msg))
            return list(), request.web_url

        result_rows = []
        for result in query_result:
            if result.fetch_full_data():
                result_rows.extend(result.rows)

        return result_rows, request.web_url

    @staticmethod
    def ret_given(val):
        return val

    def __get_phrase_info_from_caesar(self, ad_group_ids, show_conditions):
        logging.info("All banner group export ids for order: {}".format(ad_group_ids))

        from prettytable import PrettyTable
        tb = PrettyTable()
        tb.field_names = ['AdGroupID', 'PhraseID', 'ContextType', 'Cost', 'FlatCost', 'UpdateTime', 'DeleteTime', 'Suspended']
        tb.align = "l"  # The allowed strings are "l", "r" and "c" for left, right and centre alignment, respectively
        tb.border = True

        for ad_group_id in ad_group_ids:
            ad_group_info = self.get_entity_from_caesar('AdGroup', ad_group_id, [])

            if 'ShowConditions' in ad_group_info:
                show_conditions.append(ad_group_info['ShowConditions'])

            phrases = ad_group_info.get('Phrases', {})
            for phase in phrases.get('PhrasesList', []):
                current_row = [ad_group_id]

                for column in tb.field_names[1:]:  # except AdGroupID
                    current_row.append(phase[column])

                tb.add_row(current_row)

        return '<{{{name}\n%%{res}%%}}>'.format(name='Phrases from AdGroups', res=tb)

    def __get_additional_queries(self):
        queries = []
        query_name = 'bs-filter-log [legacy]'
        new_query = '''
            pragma yt.DataSizePerJob='500000000';
            PRAGMA yt.PoolTrees = "physical";
            PRAGMA yt.TentativePoolTrees = "cloud";

            SELECT
                reason,
                description,
                COUNT(*)
            FROM
                hahn.[home/logfeller/logs/bs-filter-log/1d/<DATE>]
            where orderid = <OREDERID>
            group by
                reason,
                description
        '''
        queries.append((query_name, dedent(new_query)))

        query_name = 'bs-match-log'
        new_query = '''
            SELECT
                stage,
                COUNT(*)
            FROM hahn.[home/logfeller/logs/bs-match-log/1d/<DATE>]
            where orderid = "ORDERID" and entrytype = "MATCH"
            group by stage;
        '''
        queries.append((query_name, dedent(new_query)))

        query_name = 'bs-match-log (with Filter Reasons)'
        new_query = '''
            PRAGMA File('yabs.so', 'https://proxy.sandbox.yandex-team.ru/last/YABS_UDF/libyabs_udf.so');
            PRAGMA udf('yabs.so');

            SELECT
                reasonid,
                stage,
                object,
                some(reason) as reason,
                some(Yabs::StrFilterReasonDescription(reasonid)) as description,
                COUNT(distinct hitlogid) as affected_hits,
                COUNT(*) as cnt
            FROM
                hahn.`home/logfeller/logs/bs-match-log/1d/<DATE>`
            where orderid = '<ORDERID>'
            group by
                reasonid, stage, object
            order by affected_hits desc;
        '''
        queries.append((query_name, dedent(new_query)))

        return '<{{The following queries may also be useful\n{queries}}}>'.format(
            queries=''.join(['<{{{n}\n%%{q}%%\n}}>\n'.format(
                n=query[0],
                q=query[1]
            ) for query in queries])
        )
